# Marketplace Feature - Implementation Plan

## Overview
Create a Marketplace where providers/admins can purchase and download completed self-purchased Paradigm Profiles. Pricing is based on assessment results/formula.

---

## 1. Database Changes

### 1.1 Migration: Add marketplace fields to `paradigm_assessment_results`
```php
// database/migrations/YYYY_MM_DD_HHMMSS_add_marketplace_fields_to_paradigm_assessment_results.php
- Add: `is_available_in_marketplace` (boolean, default: false)
- Add: `marketplace_listed_at` (timestamp, nullable)
- Add: `marketplace_purchase_count` (integer, default: 0)
```

### 1.2 Migration: Create `paradigm_marketplace_purchases` table
```php
// database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_marketplace_purchases_table.php
Fields:
- id
- result_id (foreign key to paradigm_assessment_results)
- purchaser_id (foreign key to users - provider/admin who bought it)
- purchase_price (decimal) - credits paid
- purchased_at (timestamp)
- wallet_transaction_id (foreign key to wallet_history)
- is_sold (boolean, default: true) - Marks profile as sold when purchased
- linked_practitioner_id (foreign key to users, nullable) - Practitioner who owns this profile
- created_at, updated_at
```

### 1.3 Migration: Create `paradigm_link_requests` table
```php
// database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_link_requests_table.php
Fields:
- id
- result_id (foreign key to paradigm_assessment_results)
- marketplace_purchase_id (foreign key to paradigm_marketplace_purchases)
- practitioner_id (foreign key to users - practitioner who bought the profile)
- client_id (foreign key to users - client who owns the profile)
- status (enum: 'pending', 'accepted', 'declined') - default: 'pending'
- requested_at (timestamp)
- responded_at (timestamp, nullable)
- created_at, updated_at
```

### 1.4 Migration: Create `paradigm_practitioner_transfers` table
```php
// database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_practitioner_transfers_table.php
Fields:
- id
- marketplace_purchase_id (foreign key to paradigm_marketplace_purchases)
- from_practitioner_id (foreign key to users - current practitioner)
- to_practitioner_id (foreign key to users - new practitioner)
- transfer_price (decimal) - credits paid by new practitioner
- transfer_status (enum: 'pending', 'completed', 'cancelled') - default: 'pending'
- wallet_transaction_id_from (foreign key to wallet_history - credit to old practitioner)
- wallet_transaction_id_to (foreign key to wallet_history - debit from new practitioner)
- transferred_at (timestamp, nullable)
- created_at, updated_at
```

### 1.5 Migration: Create `paradigm_marketplace_pricing_rules` table
```php
// database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_marketplace_pricing_rules_table.php
Fields:
- id
- rule_name (string) - e.g., "Type A Profile", "High Score Profile"
- paradigm_type (string, nullable) - specific type (e.g., "A", "B", "C")
- min_score (decimal, nullable) - minimum score threshold
- max_score (decimal, nullable) - maximum score threshold
- price_credits (decimal) - price in credits
- is_active (boolean, default: true)
- sort_order (integer, default: 0)
- created_at, updated_at
```

---

## 2. Models

### 2.1 Update `ParadigmAssessmentResult.php`
- Add `is_available_in_marketplace` to fillable
- Add `marketplace_listed_at` to fillable and casts
- Add relationship: `marketplacePurchases()` - HasMany to ParadigmMarketplacePurchase
- Add method: `isAvailableInMarketplace()` - check if listed and completed
- Add method: `getMarketplacePrice()` - calculate price based on pricing rules

### 2.2 Create `ParadigmMarketplacePurchase.php`
- Relationships: `result()`, `purchaser()`, `walletTransaction()`, `linkedPractitioner()`, `linkRequest()`, `transfers()`
- Add method: `isSold()` - check if profile is sold
- Add method: `markAsSold()` - mark profile as sold and link to practitioner

### 2.3 Create `ParadigmLinkRequest.php`
- Relationships: `result()`, `marketplacePurchase()`, `practitioner()`, `client()`
- Add method: `accept()` - accept link request
- Add method: `decline()` - decline link request

### 2.4 Create `ParadigmPractitionerTransfer.php`
- Relationships: `marketplacePurchase()`, `fromPractitioner()`, `toPractitioner()`, `walletTransactionFrom()`, `walletTransactionTo()`
- Add method: `complete()` - complete transfer and update ownership

### 2.5 Create `ParadigmMarketplacePricingRule.php`
- Standard model with fillable fields

---

## 3. Controllers

### 3.1 Create `app/Http/Controllers/Provider/ParadigmMarketplaceController.php`
**Methods:**
- `index()` - List available marketplace profiles (only completed self-purchased, not sold)
- `show($resultId)` - View profile details (name, surname only) before purchase
- `purchase($resultId)` - Purchase profile with credits, mark as sold, create link request
- `download($resultId)` - Download purchased profile PDF
- `myPurchases()` - List profiles purchased by current provider/admin
- `transferProfile($purchaseId)` - Show transfer form (enter new practitioner email, set price)
- `processTransfer($purchaseId)` - Process transfer: deduct credits from new practitioner, credit old practitioner
- `linkRequests()` - Show pending link requests from clients

### 3.2 Update `app/Http/Controllers/Admin/ParadigmPricingController.php`
**New Tab: "Marketplace Pricing"**
- `marketplacePricing()` - Show pricing rules management
- `storeMarketplacePricing()` - Create/update pricing rule
- `deleteMarketplacePricing()` - Delete pricing rule

### 3.3 Update `app/Http/Controllers/Public/ParadigmAssessmentController.php`
**Auto-list in marketplace:**
- In `submit()` method, after assessment completion:
  - Find related payment by matching: email, template_id, provider_id, created_at
  - If payment exists AND `payment->self_purchased == 1`:
    - Set `result->is_available_in_marketplace = true`
    - Set `result->marketplace_listed_at = now()`
    - Save result

### 3.4 Create `app/Http/Controllers/Provider/ParadigmLinkRequestController.php`
**Methods:**
- `index()` - List link requests for practitioner (pending/accepted/declined)
- `accept($requestId)` - Accept link request from client
- `decline($requestId)` - Decline link request from client

### 3.5 Create `app/Http/Controllers/User/ParadigmLinkRequestController.php`
**Methods:**
- `index()` - List link requests for user (pending/accepted/declined)
- `accept($requestId)` - Accept link request from practitioner
- `decline($requestId)` - Decline link request from practitioner

---

## 4. Views

### 4.1 Create `resources/views/provider/paradigm/marketplace/index.blade.php`
- List of available profiles
- Show: Client Name, Client Surname, Paradigm Type, Price, Purchase button
- Filter by Paradigm Type
- Search by name

### 4.2 Create `resources/views/provider/paradigm/marketplace/show.blade.php`
- Profile preview (name, surname only)
- Price display
- Purchase button (checks wallet balance)

### 4.3 Create `resources/views/provider/paradigm/marketplace/my-purchases.blade.php`
- List of purchased profiles
- Download buttons
- Transfer button (enter new practitioner email, set transfer price)
- Show current linked client status

### 4.4 Create `resources/views/provider/paradigm/link-requests/index.blade.php`
- List of link requests from clients
- Show: Client name, Profile type, Request date, Accept/Decline buttons
- Filter by status (pending/accepted/declined)

### 4.5 Create `resources/views/user/paradigm/link-requests/index.blade.php`
- List of link requests from practitioners
- Show: Practitioner name, Profile type, Request date, Accept/Decline buttons
- Filter by status (pending/accepted/declined)

### 4.6 Create `resources/views/provider/paradigm/marketplace/transfer.blade.php`
- Form to transfer profile to another practitioner
- Fields: New Practitioner Email, Transfer Price (credits)
- Show current profile details
- Show current linked client

### 4.7 Update `resources/views/admin/paradigm/pricing/index.blade.php`
**Add new tab: "Marketplace Pricing"**
- Table of pricing rules
- Form to add/edit pricing rule
- Fields: Rule Name, Paradigm Type (dropdown), Min Score, Max Score, Price (Credits), Active

---

## 5. Routes

### 5.1 Provider Routes
```php
Route::middleware(['auth', 'provider'])->group(function() {
    // Marketplace routes
    Route::get('/provider/paradigm/marketplace', [ParadigmMarketplaceController::class, 'index'])->name('provider.paradigm.marketplace');
    Route::get('/provider/paradigm/marketplace/{resultId}', [ParadigmMarketplaceController::class, 'show'])->name('provider.paradigm.marketplace.show');
    Route::post('/provider/paradigm/marketplace/{resultId}/purchase', [ParadigmMarketplaceController::class, 'purchase'])->name('provider.paradigm.marketplace.purchase');
    Route::get('/provider/paradigm/marketplace/{resultId}/download', [ParadigmMarketplaceController::class, 'download'])->name('provider.paradigm.marketplace.download');
    Route::get('/provider/paradigm/marketplace/my-purchases', [ParadigmMarketplaceController::class, 'myPurchases'])->name('provider.paradigm.marketplace.my-purchases');
    
    // Transfer routes
    Route::get('/provider/paradigm/marketplace/{purchaseId}/transfer', [ParadigmMarketplaceController::class, 'transferProfile'])->name('provider.paradigm.marketplace.transfer');
    Route::post('/provider/paradigm/marketplace/{purchaseId}/transfer', [ParadigmMarketplaceController::class, 'processTransfer'])->name('provider.paradigm.marketplace.transfer.process');
    
    // Link request routes
    Route::get('/provider/paradigm/link-requests', [ParadigmLinkRequestController::class, 'index'])->name('provider.paradigm.link-requests');
    Route::post('/provider/paradigm/link-requests/{requestId}/accept', [ParadigmLinkRequestController::class, 'accept'])->name('provider.paradigm.link-requests.accept');
    Route::post('/provider/paradigm/link-requests/{requestId}/decline', [ParadigmLinkRequestController::class, 'decline'])->name('provider.paradigm.link-requests.decline');
});
```

### 5.3 User Routes
```php
Route::middleware(['auth', 'auc'])->group(function() {
    // Link request routes for users
    Route::get('/user/paradigm/link-requests', [User\ParadigmLinkRequestController::class, 'index'])->name('user.paradigm.link-requests');
    Route::post('/user/paradigm/link-requests/{requestId}/accept', [User\ParadigmLinkRequestController::class, 'accept'])->name('user.paradigm.link-requests.accept');
    Route::post('/user/paradigm/link-requests/{requestId}/decline', [User\ParadigmLinkRequestController::class, 'decline'])->name('user.paradigm.link-requests.decline');
});
```

### 5.2 Admin Routes (in existing pricing group)
```php
Route::prefix('pricing')->name('pricing.')->group(function () {
    // ... existing routes ...
    Route::get('/marketplace', [ParadigmPricingController::class, 'marketplacePricing'])->name('marketplace.index');
    Route::post('/marketplace', [ParadigmPricingController::class, 'storeMarketplacePricing'])->name('marketplace.store');
    Route::delete('/marketplace/{ruleId}', [ParadigmPricingController::class, 'deleteMarketplacePricing'])->name('marketplace.delete');
});
```

---

## 6. Menu Updates

### 6.1 Update `resources/views/provider/partials/header.blade.php`
Add Marketplace and Link Requests menu items:
```blade
@if(Auth::user()->user_type == 1 || (Auth::user()->category && Auth::user()->category->slug === 'practitioner'))
<li class="{{ request()->is('provider/paradigm/marketplace*') ? 'active' : '' }}">
    <a href="{{ route('provider.paradigm.marketplace') }}" class="{{ request()->is('provider/paradigm/marketplace*') ? 'active' : '' }}">
        <i class="ti ti-shopping-cart"></i><span>{{__('Marketplace')}}</span>
    </a>
</li>
<li class="{{ request()->is('provider/paradigm/link-requests*') ? 'active' : '' }}">
    <a href="{{ route('provider.paradigm.link-requests') }}" class="{{ request()->is('provider/paradigm/link-requests*') ? 'active' : '' }}">
        <i class="ti ti-link"></i><span>{{__('Link Requests')}}</span>
    </a>
</li>
@endif
```

### 6.2 Update `resources/views/user/partials/sidebar.blade.php`
Add Link Requests menu item:
```blade
<li class="{{ request()->is('user/paradigm/link-requests*') ? 'active' : '' }}">
    <a href="{{ route('user.paradigm.link-requests') }}" class="{{ request()->is('user/paradigm/link-requests*') ? 'active' : '' }}">
        <i class="ti ti-link"></i><span>{{__('Link Requests')}}</span>
    </a>
</li>
```

---

## 7. Pricing Logic

### 7.1 Price Calculation Method
```php
// In ParadigmAssessmentResult model
public function getMarketplacePrice()
{
    $result = $this;
    $paradigmType = $result->paradigm_type; // e.g., "A", "B", "C"
    $scores = $result->scores_json; // Get scores
    
    // Find matching pricing rule
    $rule = ParadigmMarketplacePricingRule::where('is_active', true)
        ->where(function($query) use ($paradigmType, $scores) {
            // Match by type
            $query->where(function($q) use ($paradigmType) {
                $q->whereNull('paradigm_type')
                  ->orWhere('paradigm_type', $paradigmType);
            });
            
            // Match by score range (if scores available)
            if ($scores && isset($scores['total_score'])) {
                $totalScore = $scores['total_score'];
                $query->where(function($q) use ($totalScore) {
                    $q->whereNull('min_score')
                      ->orWhere('min_score', '<=', $totalScore);
                })
                ->where(function($q) use ($totalScore) {
                    $q->whereNull('max_score')
                      ->orWhere('max_score', '>=', $totalScore);
                });
            }
        })
        ->orderBy('sort_order')
        ->first();
    
    return $rule ? $rule->price_credits : 0; // Default to 0 if no rule matches
}
```

---

## 8. Purchase Flow

### 8.1 Purchase Process
1. Provider clicks "Purchase" on marketplace profile
2. Check wallet balance (sufficient credits?)
3. If yes:
   - Deduct credits from provider wallet
   - Create `ParadigmMarketplacePurchase` record
   - Mark `is_sold = true` and set `linked_practitioner_id = purchaser_id`
   - Create `WalletHistory` record (debit)
   - **Create `ParadigmLinkRequest` record** (status: 'pending')
     - Link to marketplace purchase
     - Link to practitioner (purchaser)
     - Link to client (from result)
   - Remove profile from marketplace (is_sold = true)
   - Redirect to download page
4. If no:
   - Redirect to wallet top-up page

### 8.2 Download Authorization
- Only allow download if `ParadigmMarketplacePurchase` exists for this provider + result
- Check `purchaser_id == Auth::id()` OR `linked_practitioner_id == Auth::id()`

### 8.3 Link Request Process
1. **When Practitioner Purchases:**
   - Create `ParadigmLinkRequest` with status 'pending'
   - Client receives notification (email/WhatsApp)
   - Client sees request in "Link Requests" menu

2. **Client Accepts:**
   - Update `ParadigmLinkRequest.status = 'accepted'`
   - Update `ParadigmMarketplacePurchase.linked_practitioner_id` (if not already set)
   - Practitioner and client are now linked
   - Both parties receive confirmation

3. **Client Declines:**
   - Update `ParadigmLinkRequest.status = 'declined'`
   - Practitioner receives notification
   - Profile remains with practitioner but client is not linked

### 8.4 Transfer Process
1. **Practitioner Initiates Transfer:**
   - Practitioner enters new practitioner email
   - Practitioner sets transfer price (credits)
   - Create `ParadigmPractitionerTransfer` record (status: 'pending')

2. **New Practitioner Accepts:**
   - Check new practitioner wallet balance
   - If sufficient:
     - Deduct credits from new practitioner wallet
     - Credit credits to old practitioner wallet
     - Update `ParadigmMarketplacePurchase.linked_practitioner_id = new_practitioner_id`
     - Update `ParadigmLinkRequest.practitioner_id = new_practitioner_id` (if accepted)
     - Update transfer status to 'completed'
   - If insufficient:
     - Show error, redirect to wallet top-up

3. **Transfer Rules:**
   - Only self-purchased profiles can be transferred
   - Old practitioner sets the price
   - New practitioner must have sufficient credits
   - Link request status transfers with the profile

---

## 9. Auto-Listing Logic

### 9.1 When to List in Marketplace
- Assessment is completed (`completed_at` is set)
- Payment has `self_purchased = 1`
- Assessment result exists and is linked to payment
- Auto-set `is_available_in_marketplace = true` when assessment completes
- **Only show profiles where `is_sold = false` in marketplace**

---

## 10. Security & Permissions

### 10.1 Middleware
- Add middleware check in `CheckUserPermission.php`:
  - Route: `provider.paradigm.marketplace*`
  - Allow: Admin (user_type == 1) OR Provider with category "practitioner"

### 10.2 Data Privacy
- Marketplace only shows: Name, Surname, Paradigm Type, Price
- Full profile details only visible after purchase
- PDF download only available to purchasers
- Link requests only visible to involved parties (practitioner and client)

### 10.3 Transfer Security
- Only the current linked practitioner can initiate transfer
- New practitioner must exist and be a practitioner (user_type = 2, category = 'practitioner')
- Transfer price must be positive
- Both wallet transactions must succeed or both fail (transaction rollback)

---

## 11. Implementation Order

1. **Phase 1: Database & Models**
   - Create migrations
   - Create/update models
   - Add relationships

2. **Phase 2: Pricing System**
   - Create pricing rules table
   - Add pricing tab in admin
   - Implement price calculation logic

3. **Phase 3: Marketplace Listing**
   - Auto-list completed self-purchased profiles
   - Create marketplace index view

4. **Phase 4: Purchase Flow**
   - Implement purchase controller
   - Wallet deduction
   - Purchase tracking

5. **Phase 5: Download & Access**
   - Download authorization
   - My Purchases page
   - PDF download for purchased profiles

6. **Phase 6: Menu & Routes**
   - Add menu item
   - Add routes
   - Add middleware checks

---

## 12. Files to Create/Modify

### New Files (15):
1. `database/migrations/YYYY_MM_DD_HHMMSS_add_marketplace_fields_to_paradigm_assessment_results.php`
2. `database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_marketplace_purchases_table.php`
3. `database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_link_requests_table.php`
4. `database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_practitioner_transfers_table.php`
5. `database/migrations/YYYY_MM_DD_HHMMSS_create_paradigm_marketplace_pricing_rules_table.php`
6. `app/Models/ParadigmMarketplacePurchase.php`
7. `app/Models/ParadigmLinkRequest.php`
8. `app/Models/ParadigmPractitionerTransfer.php`
9. `app/Models/ParadigmMarketplacePricingRule.php`
10. `app/Http/Controllers/Provider/ParadigmMarketplaceController.php`
11. `app/Http/Controllers/Provider/ParadigmLinkRequestController.php`
12. `app/Http/Controllers/User/ParadigmLinkRequestController.php`
13. `resources/views/provider/paradigm/marketplace/index.blade.php`
14. `resources/views/provider/paradigm/marketplace/my-purchases.blade.php`
15. `resources/views/provider/paradigm/marketplace/transfer.blade.php`
16. `resources/views/provider/paradigm/link-requests/index.blade.php`
17. `resources/views/user/paradigm/link-requests/index.blade.php`

### Modified Files (6):
1. `app/Models/ParadigmAssessmentResult.php` - Add marketplace fields and methods
2. `app/Http/Controllers/Admin/ParadigmPricingController.php` - Add marketplace pricing tab
3. `app/Http/Controllers/Public/ParadigmAssessmentController.php` - Auto-list in marketplace
4. `resources/views/admin/paradigm/pricing/index.blade.php` - Add marketplace pricing tab
5. `resources/views/provider/partials/header.blade.php` - Add Marketplace and Link Requests menu items
6. `resources/views/user/partials/sidebar.blade.php` - Add Link Requests menu item
7. `routes/web.php` - Add marketplace, link request, and transfer routes

---

## 13. Pricing Formula Examples

### Example Rules:
1. **Default Rule**: All profiles = 5 credits
2. **Type A Profiles**: Paradigm Type = "A" → 10 credits
3. **High Score Profiles**: Total Score >= 80 → 15 credits
4. **Type B with High Score**: Type = "B" AND Score >= 75 → 12 credits

### Admin Interface:
- Rule Name: "Type A Profiles"
- Paradigm Type: [Dropdown: A, B, C, D, etc.]
- Min Score: [Optional]
- Max Score: [Optional]
- Price (Credits): 10
- Active: [Toggle]

---

## 14. Testing Checklist

- [ ] Self-purchased profiles auto-list in marketplace after completion
- [ ] Only completed self-purchased profiles show in marketplace (not sold)
- [ ] Pricing rules calculate correctly based on type/score
- [ ] Provider can purchase profile (credits deducted)
- [ ] Profile marked as sold when purchased
- [ ] Link request created when practitioner purchases profile
- [ ] Client receives link request notification
- [ ] Client can accept link request
- [ ] Client can decline link request
- [ ] Practitioner can see link requests in menu
- [ ] User can see link requests in menu
- [ ] Provider can download purchased profile
- [ ] Provider cannot download unpurchased profile
- [ ] My Purchases page shows all purchased profiles
- [ ] Practitioner can transfer profile to another practitioner
- [ ] Transfer deducts credits from new practitioner
- [ ] Transfer credits old practitioner
- [ ] Transfer updates linked practitioner ID
- [ ] Link request transfers with profile
- [ ] Only self-purchased profiles can be transferred
- [ ] Admin can create/edit/delete pricing rules
- [ ] Menu items only visible to appropriate users
- [ ] Wallet balance checked before purchase and transfer

