<?php

namespace Modules\Communication\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Modules\Communication\app\Http\Requests\SendSmsRequest;
use Modules\Communication\app\Repositories\Contracts\SmsInterface;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SmsController extends Controller
{
    protected $smsRepository;

    public function __construct(SmsInterface $smsRepository)
    {
        $this->smsRepository = $smsRepository;
    }

    /**
     * Send an SMS message (API endpoint).
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function sendsms(Request $request): JsonResponse
    {
        try {
            $otpType = $request->input('otp_type');
            $toNumber = $request->input('to_number');
            $notificationType = $request->input('notification_type');
            $type = $request->input('type');
            $userName = $request->input('user_name', 'User');
            $otp = $request->input('otp', '123456');
            $content = $request->input('content', '');
            
            if (!$toNumber) {
                return response()->json(['error' => 'Missing phone number'], 400);
            }
            
            // Replace placeholders in the content
            if (!empty($content)) {
                $content = str_replace(
                    ['{{user_name}}', '{{otp}}'],
                    [$userName, $otp],
                    $content
                );
            } else {
                // Fallback message if no template content
                $content = "Hello {$userName}, Your OTP code is: {$otp}. This code will expire in few minutes. Do not share this code with anyone.";
            }
            
            $data = [
                'otp_type' => $otpType,
                'to_number' => $toNumber,
                'notification_type' => $notificationType,
                'type' => $type,
                'content' => $content,
                'otp' => $otp,
                'user_name' => $userName
            ];
            
            $response = $this->smsRepository->sendSms($data);
            return response()->json($response);
        } catch (\Exception $e) {
            Log::error('SMS sending error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to send SMS'], 500);
        }
    }


    /**
     * Send SMS for OTP verification (registration/login).
     * This method handles template variable replacement for OTP messages.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function sendOtpSms(Request $request): JsonResponse
    {
        try {
            Log::info('OTP SMS Request received', [
                'to_number' => $request->input('to_number'),
                'has_user_name' => $request->has('user_name'),
                'has_otp' => $request->has('otp'),
                'has_content' => $request->has('content')
            ]);

            // Validate required fields
            $request->validate([
                'to_number' => 'required|string|max:20',
                'content' => 'required|string|max:1600',
            ]);

            // Get the content which may contain template variables
            $content = $request->input('content');
            
            // Check if template variables exist and replace them if user_name and otp are provided
            if ($request->has('user_name') && $request->has('otp')) {
                $content = str_replace(
                    ['{{user_name}}', '{{otp}}'],
                    [$request->input('user_name'), $request->input('otp')],
                    $content
                );
                Log::info('OTP placeholders replaced in content');
            } else {
                Log::warning('OTP SMS missing user_name or otp parameters');
            }

            // Prepare data for SMS/WhatsApp sending
            $smsData = [
                'to_number' => $request->input('to_number'),
                'content' => $content,
                'otp' => $request->input('otp'), // Pass OTP for WhatsApp template
                'user_name' => $request->input('user_name'), // Pass username for potential use
            ];

            Log::info('Calling smsRepository->sendSms', ['has_otp' => !empty($smsData['otp'])]);

            // Send the SMS/WhatsApp
            $response = $this->smsRepository->sendSms($smsData);

            Log::info('SMS Repository Response', ['response' => $response]);

            return response()->json($response, $response['code'] ?? 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('SMS Validation failed', [
                'errors' => $e->errors()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors(),
                'code' => 422
            ], 422);
        } catch (\Exception $e) {
            Log::error('SMS sending failed: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to send SMS: ' . $e->getMessage(),
                'code' => 500
            ], 500);
        }
    }
}