<?php

namespace Modules\Leads\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Modules\Leads\app\Models\UserFormInput;
use Modules\Leads\app\Models\LeadUnlock;
use Modules\Leads\app\Models\LeadPricingRule;
use Modules\Categories\app\Models\Categories;
use App\Models\WalletHistory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

/**
 * AdminMarketplaceAnalyticsController
 * 
 * Controller for marketplace analytics and performance metrics.
 * Does NOT affect existing admin analytics functionality.
 */
class AdminMarketplaceAnalyticsController extends Controller
{
    /**
     * Display marketplace analytics dashboard
     */
    public function index()
    {
        $data['page'] = 'admin.marketplace.analytics';
        $data['title'] = __('Marketplace Analytics');
        $data['asset'] = 'marketplace-analytics';
        
        return view('leads::admin.analytics.index', $data);
    }

    /**
     * Get marketplace overview statistics
     */
    public function getOverviewStats(Request $request): JsonResponse
    {
        try {
            $dateRange = $request->get('date_range', '30'); // days
            $startDate = now()->subDays($dateRange)->startOfDay();
            $endDate = now()->endOfDay();

            // Total marketplace leads
            $totalLeads = UserFormInput::where('lead_source', 'marketplace')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

            // Total revenue from marketplace
            $totalRevenue = WalletHistory::where('transaction_id', 'LIKE', 'MARKETPLACE-LEAD-%')
                ->where('type', 'debit')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            // Total unlocks
            $totalUnlocks = LeadUnlock::whereHas('lead', function($query) use ($startDate, $endDate) {
                $query->where('lead_source', 'marketplace')
                      ->whereBetween('created_at', [$startDate, $endDate]);
            })->count();

            // Active providers
            $activeProviders = LeadUnlock::whereHas('lead', function($query) use ($startDate, $endDate) {
                $query->where('lead_source', 'marketplace')
                      ->whereBetween('created_at', [$startDate, $endDate]);
            })->distinct('provider_id')->count('provider_id');

            // Conversion rate (leads with at least one unlock)
            $convertedLeads = UserFormInput::where('lead_source', 'marketplace')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->whereHas('unlocks')
                ->count();

            $conversionRate = $totalLeads > 0 ? ($convertedLeads / $totalLeads) * 100 : 0;

            // Average unlock price
            $avgUnlockPrice = UserFormInput::where('lead_source', 'marketplace')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->where('calculated_unlock_price', '>', 0)
                ->avg('calculated_unlock_price');

            return response()->json([
                'success' => true,
                'data' => [
                    'total_leads' => $totalLeads,
                    'total_revenue' => round($totalRevenue, 2),
                    'total_unlocks' => $totalUnlocks,
                    'active_providers' => $activeProviders,
                    'conversion_rate' => round($conversionRate, 2),
                    'avg_unlock_price' => round($avgUnlockPrice, 2),
                    'date_range' => $dateRange
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching marketplace overview stats: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch analytics data'
            ], 500);
        }
    }

    /**
     * Get marketplace leads by category
     */
    public function getLeadsByCategory(Request $request): JsonResponse
    {
        try {
            $dateRange = $request->get('date_range', '30');
            $startDate = now()->subDays($dateRange)->startOfDay();
            $endDate = now()->endOfDay();

            $categoryStats = UserFormInput::where('lead_source', 'marketplace')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->with('category:id,name')
                ->selectRaw('category_id, COUNT(*) as lead_count, AVG(calculated_unlock_price) as avg_price, SUM(CASE WHEN is_unlocked = 1 THEN 1 ELSE 0 END) as unlocked_count')
                ->groupBy('category_id')
                ->get()
                ->map(function($item) {
                    return [
                        'category_name' => $item->category->name ?? 'Unknown',
                        'lead_count' => $item->lead_count,
                        'unlocked_count' => $item->unlocked_count,
                        'unlock_rate' => $item->lead_count > 0 ? round(($item->unlocked_count / $item->lead_count) * 100, 2) : 0,
                        'avg_price' => round($item->avg_price, 2),
                        'total_revenue' => round($item->unlocked_count * $item->avg_price, 2)
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $categoryStats
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching leads by category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch category analytics'
            ], 500);
        }
    }

    /**
     * Get marketplace performance over time
     */
    public function getPerformanceOverTime(Request $request): JsonResponse
    {
        try {
            $dateRange = $request->get('date_range', '30');
            $startDate = now()->subDays($dateRange)->startOfDay();
            $endDate = now()->endOfDay();

            // Daily performance data
            $dailyStats = UserFormInput::where('lead_source', 'marketplace')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->selectRaw('DATE(created_at) as date, COUNT(*) as leads, AVG(calculated_unlock_price) as avg_price')
                ->groupBy('date')
                ->orderBy('date')
                ->get()
                ->map(function($item) {
                    // Get unlock count for this date
                    $unlocks = LeadUnlock::whereHas('lead', function($query) use ($item) {
                        $query->where('lead_source', 'marketplace')
                              ->whereDate('created_at', $item->date);
                    })->count();

                    return [
                        'date' => $item->date,
                        'leads' => $item->leads,
                        'unlocks' => $unlocks,
                        'avg_price' => round($item->avg_price, 2),
                        'revenue' => round($unlocks * $item->avg_price, 2)
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $dailyStats
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching performance over time: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch performance data'
            ], 500);
        }
    }

    /**
     * Get top performing providers
     */
    public function getTopProviders(Request $request): JsonResponse
    {
        try {
            $dateRange = $request->get('date_range', '30');
            $startDate = now()->subDays($dateRange)->startOfDay();
            $endDate = now()->endOfDay();

            $topProviders = LeadUnlock::whereBetween('created_at', [$startDate, $endDate])
                ->whereHas('lead', function($query) {
                    $query->where('lead_source', 'marketplace');
                })
                ->with('provider:id,name,email')
                ->selectRaw('provider_id, COUNT(*) as unlocks, SUM(credits_spent) as total_spent, AVG(credits_spent) as avg_spent')
                ->groupBy('provider_id')
                ->orderBy('unlocks', 'desc')
                ->limit(10)
                ->get()
                ->map(function($item) {
                    return [
                        'provider_name' => $item->provider->name ?? 'Unknown',
                        'provider_email' => $item->provider->email ?? '',
                        'unlocks' => $item->unlocks,
                        'total_spent' => round($item->total_spent, 2),
                        'avg_spent' => round($item->avg_spent, 2)
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $topProviders
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching top providers: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch provider analytics'
            ], 500);
        }
    }

    /**
     * Get marketplace pricing analytics
     */
    public function getPricingAnalytics(Request $request): JsonResponse
    {
        try {
            $dateRange = $request->get('date_range', '30');
            $startDate = now()->subDays($dateRange)->startOfDay();
            $endDate = now()->endOfDay();

            // Pricing distribution
            $pricingDistribution = UserFormInput::where('lead_source', 'marketplace')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->where('calculated_unlock_price', '>', 0)
                ->selectRaw('
                    CASE 
                        WHEN calculated_unlock_price < 25 THEN "0-25"
                        WHEN calculated_unlock_price < 50 THEN "25-50"
                        WHEN calculated_unlock_price < 100 THEN "50-100"
                        WHEN calculated_unlock_price < 200 THEN "100-200"
                        ELSE "200+"
                    END as price_range,
                    COUNT(*) as count,
                    AVG(calculated_unlock_price) as avg_price
                ')
                ->groupBy('price_range')
                ->orderBy('price_range')
                ->get();

            // Active pricing rules
            $activePricingRules = LeadPricingRule::with('category:id,name')
                ->where('active', true)
                ->get()
                ->map(function($rule) {
                    return [
                        'category_name' => $rule->category->name ?? 'Unknown',
                        'base_price' => $rule->base_price,
                        'price_per_field' => $rule->price_per_field,
                        'price_per_100_chars' => $rule->price_per_100_chars,
                        'min_price' => $rule->min_price,
                        'max_price' => $rule->max_price
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'pricing_distribution' => $pricingDistribution,
                    'active_pricing_rules' => $activePricingRules
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching pricing analytics: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch pricing analytics'
            ], 500);
        }
    }

    /**
     * Export marketplace analytics data
     */
    public function exportAnalytics(Request $request): JsonResponse
    {
        try {
            $exportType = $request->get('type', 'overview'); // overview, leads, providers, pricing
            $dateRange = $request->get('date_range', '30');
            $startDate = now()->subDays($dateRange)->startOfDay();
            $endDate = now()->endOfDay();

            $data = [];

            switch ($exportType) {
                case 'leads':
                    $data = UserFormInput::where('lead_source', 'marketplace')
                        ->whereBetween('created_at', [$startDate, $endDate])
                        ->with(['category:id,name', 'service:id,source_name'])
                        ->get()
                        ->map(function($lead) {
                            return [
                                'ID' => $lead->id,
                                'Category' => $lead->category->name ?? 'Unknown',
                                'Service' => $lead->service->source_name ?? 'Unknown',
                                'Unlock Price' => $lead->calculated_unlock_price,
                                'Quality Score' => $lead->quality_score,
                                'Status' => $lead->marketplace_status,
                                'Unlocks' => $lead->unlock_count,
                                'Created At' => $lead->created_at->format('Y-m-d H:i:s'),
                                'Expires At' => $lead->expires_at ? $lead->expires_at->format('Y-m-d H:i:s') : 'Never'
                            ];
                        });
                    break;

                case 'providers':
                    $data = LeadUnlock::whereBetween('created_at', [$startDate, $endDate])
                        ->whereHas('lead', function($query) {
                            $query->where('lead_source', 'marketplace');
                        })
                        ->with(['provider:id,name,email', 'lead:id,category_id'])
                        ->get()
                        ->map(function($unlock) {
                            return [
                                'Provider Name' => $unlock->provider->name ?? 'Unknown',
                                'Provider Email' => $unlock->provider->email ?? 'Unknown',
                                'Lead ID' => $unlock->lead_id,
                                'Credits Spent' => $unlock->credits_spent,
                                'Connection Status' => $unlock->connection_status,
                                'Unlocked At' => $unlock->unlocked_at->format('Y-m-d H:i:s')
                            ];
                        });
                    break;

                default: // overview
                    $overviewStats = $this->getOverviewStats($request);
                    $data = $overviewStats->getData()->data;
                    break;
            }

            return response()->json([
                'success' => true,
                'data' => $data,
                'export_type' => $exportType,
                'date_range' => $dateRange
            ]);

        } catch (\Exception $e) {
            Log::error('Error exporting analytics: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to export analytics data'
            ], 500);
        }
    }
}
