<?php

namespace Modules\Leads\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Modules\Categories\app\Models\Categories;
use Modules\Leads\app\Models\LeadPricingRule;

class AdminPricingController extends Controller
{
    /**
     * Display the pricing management page
     */
    public function index()
    {
        try {
            $data = [
                'title' => __('Lead Pricing'),
                'heading' => __('Marketplace Lead Pricing'),
                'breadcrumb' => [
                    ['title' => __('Dashboard'), 'url' => route('admin.dashboard')],
                    ['title' => __('Leads'), 'url' => route('admin.leads')],
                    ['title' => __('Pricing Management'), 'url' => ''],
                ]
            ];
            
            return view('leads::pricing.index', $data);
        } catch (\Exception $e) {
            Log::error('Error loading pricing page: ' . $e->getMessage());
            return redirect()->route('admin.leads')->with('error', 'Unable to load pricing page');
        }
    }

    /**
     * Get all pricing rules (DataTable)
     */
    public function list(Request $request): JsonResponse
    {
        try {
            $query = LeadPricingRule::with('category')
                ->orderBy('created_at', 'desc');

            // Search filter
            if ($request->has('search') && !empty($request->search['value'])) {
                $search = $request->search['value'];
                $query->whereHas('category', function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            }

            $totalRecords = LeadPricingRule::count();
            $filteredRecords = $query->count();

            // Pagination
            if ($request->has('start') && $request->has('length')) {
                $query->skip($request->start)->take($request->length);
            }

            $rules = $query->get();

            $data = $rules->map(function ($rule) {
                return [
                    'id' => $rule->id,
                    'category' => $rule->category->name ?? 'N/A',
                    'base_price' => '$' . number_format($rule->base_price, 2),
                    'price_per_field' => '$' . number_format($rule->price_per_field, 2),
                    'price_per_100_chars' => '$' . number_format($rule->price_per_100_chars, 2),
                    'min_price' => '$' . number_format($rule->min_price, 2),
                    'max_price' => '$' . number_format($rule->max_price, 2),
                    'active' => $rule->active,
                    'created_at' => $rule->created_at->format('Y-m-d H:i'),
                ];
            });

            return response()->json([
                'draw' => intval($request->draw),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching pricing rules: ' . $e->getMessage());
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Unable to fetch pricing rules'
            ], 500);
        }
    }

    /**
     * Get a single pricing rule
     */
    public function get(int $id): JsonResponse
    {
        try {
            $rule = LeadPricingRule::with('category')->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => $rule
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching pricing rule: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch pricing rule'
            ], 500);
        }
    }

    /**
     * Store a new pricing rule
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'category_id' => 'required|integer|exists:categories,id|unique:lead_pricing_rules,category_id',
            'base_price' => 'required|numeric|min:0',
            'price_per_field' => 'required|numeric|min:0',
            'price_per_100_chars' => 'required|numeric|min:0',
            'min_price' => 'required|numeric|min:0',
            'max_price' => 'required|numeric|min:0|gte:min_price',
            'active' => 'boolean',
        ]);

        try {
            $rule = LeadPricingRule::create([
                'category_id' => $request->category_id,
                'base_price' => $request->base_price,
                'price_per_field' => $request->price_per_field,
                'price_per_100_chars' => $request->price_per_100_chars,
                'min_price' => $request->min_price,
                'max_price' => $request->max_price,
                'active' => $request->active ?? true,
                'created_by_admin_id' => Auth::id(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pricing rule created successfully',
                'data' => $rule
            ], 201);
        } catch (\Exception $e) {
            Log::error('Error creating pricing rule: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to create pricing rule'
            ], 500);
        }
    }

    /**
     * Update an existing pricing rule
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $request->validate([
            'category_id' => 'required|integer|exists:categories,id|unique:lead_pricing_rules,category_id,' . $id,
            'base_price' => 'required|numeric|min:0',
            'price_per_field' => 'required|numeric|min:0',
            'price_per_100_chars' => 'required|numeric|min:0',
            'min_price' => 'required|numeric|min:0',
            'max_price' => 'required|numeric|min:0|gte:min_price',
            'active' => 'boolean',
        ]);

        try {
            $rule = LeadPricingRule::findOrFail($id);
            $rule->update($request->only([
                'category_id',
                'base_price',
                'price_per_field',
                'price_per_100_chars',
                'min_price',
                'max_price',
                'active'
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Pricing rule updated successfully',
                'data' => $rule
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating pricing rule: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to update pricing rule'
            ], 500);
        }
    }

    /**
     * Delete a pricing rule
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $rule = LeadPricingRule::findOrFail($id);
            $rule->delete();

            return response()->json([
                'success' => true,
                'message' => 'Pricing rule deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error deleting pricing rule: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to delete pricing rule'
            ], 500);
        }
    }

    /**
     * Toggle active status
     */
    public function toggleStatus(int $id): JsonResponse
    {
        try {
            $rule = LeadPricingRule::findOrFail($id);
            $rule->active = !$rule->active;
            $rule->save();

            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully',
                'active' => $rule->active
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to update status'
            ], 500);
        }
    }

    /**
     * Get categories that don't have pricing rules yet
     */
    public function getAvailableCategories(): JsonResponse
    {
        try {
            $existingCategoryIds = LeadPricingRule::pluck('category_id')->toArray();
            
            $categories = Categories::where(function($query) {
                    $query->whereNull('parent_id')->orWhere('parent_id', 0);
                })
                ->where('status', 1)
                ->whereNotIn('id', $existingCategoryIds)
                ->get(['id', 'name']);

            return response()->json([
                'success' => true,
                'categories' => $categories
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching available categories: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch categories'
            ], 500);
        }
    }

    /**
     * Get categories that have pricing rules (for test calculator)
     */
    public function getCategoriesWithRules(): JsonResponse
    {
        try {
            $rules = LeadPricingRule::with('category')
                ->where('active', true)
                ->get();
            
            $categories = $rules->map(function($rule) {
                return [
                    'id' => $rule->category_id,
                    'name' => $rule->category->name ?? 'Unknown'
                ];
            })->unique('id')->values();

            return response()->json([
                'success' => true,
                'categories' => $categories
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching categories with rules: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch categories'
            ], 500);
        }
    }

    /**
     * Apply pricing template to multiple categories
     */
    public function applyTemplate(Request $request): JsonResponse
    {
        $request->validate([
            'template' => 'required|in:low,medium,high',
            'category_ids' => 'required|array|min:1',
            'category_ids.*' => 'integer|exists:categories,id',
        ]);

        $templates = [
            'low' => [
                'base_price' => 5.00,
                'price_per_field' => 0.50,
                'price_per_100_chars' => 0.10,
                'min_price' => 5.00,
                'max_price' => 50.00
            ],
            'medium' => [
                'base_price' => 15.00,
                'price_per_field' => 1.00,
                'price_per_100_chars' => 0.20,
                'min_price' => 15.00,
                'max_price' => 100.00
            ],
            'high' => [
                'base_price' => 30.00,
                'price_per_field' => 2.00,
                'price_per_100_chars' => 0.50,
                'min_price' => 30.00,
                'max_price' => 250.00
            ],
        ];

        try {
            $template = $templates[$request->template];
            $created = 0;
            $skipped = 0;

            foreach ($request->category_ids as $categoryId) {
                // Check if rule already exists
                if (LeadPricingRule::where('category_id', $categoryId)->exists()) {
                    $skipped++;
                    continue;
                }

                LeadPricingRule::create(array_merge($template, [
                    'category_id' => $categoryId,
                    'active' => true,
                    'created_by_admin_id' => Auth::id(),
                ]));
                $created++;
            }

            return response()->json([
                'success' => true,
                'message' => "Template applied: {$created} created, {$skipped} skipped",
                'created' => $created,
                'skipped' => $skipped
            ]);
        } catch (\Exception $e) {
            Log::error('Error applying template: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to apply template'
            ], 500);
        }
    }

    /**
     * Test price calculation
     */
    public function testCalculation(Request $request): JsonResponse
    {
        $request->validate([
            'category_id' => 'required|integer|exists:categories,id',
            'fields_filled' => 'required|integer|min:0',
            'message_chars' => 'required|integer|min:0',
        ]);

        try {
            $price = LeadPricingRule::calculateUnlockPrice(
                $request->category_id,
                $request->fields_filled,
                $request->message_chars
            );

            $rule = LeadPricingRule::where('category_id', $request->category_id)->first();

            return response()->json([
                'success' => true,
                'price' => number_format($price, 2),
                'breakdown' => [
                    'base_price' => number_format($rule->base_price, 2),
                    'field_price' => number_format($request->fields_filled * $rule->price_per_field, 2),
                    'char_price' => number_format(ceil($request->message_chars / 100) * $rule->price_per_100_chars, 2),
                    'min_price' => number_format($rule->min_price, 2),
                    'max_price' => number_format($rule->max_price, 2),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error testing calculation: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to calculate price'
            ], 500);
        }
    }
}
