<?php

namespace Modules\Leads\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Modules\Leads\app\Models\UserFormInput;
use Modules\Leads\app\Models\LeadUnlock;
use App\Services\MarketplaceNotificationService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class ClientLeadController extends Controller
{
    /**
     * Display client submitted requests page
     */
    public function index()
    {
        $data['page'] = 'user.submitted-requests';
        $data['title'] = __('Submitted Requests');
        $data['asset'] = 'submitted-requests';
        
        return view('leads::client.submitted-requests.index', $data);
    }

    /**
     * Get client's submitted requests
     */
    public function getMyRequests(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();

            $query = UserFormInput::with(['category:id,name', 'service:id,source_name'])
                ->where('user_id', $userId)
                ->where('lead_source', 'marketplace')
                ->orderBy('created_at', 'desc');

            // Filter by status
            if ($request->filled('status')) {
                $query->where('marketplace_status', $request->status);
            }

            $leads = $query->paginate(20);

            // Get unlock count for each lead
            $leadIds = $leads->pluck('id')->toArray();
            $unlockCounts = LeadUnlock::whereIn('lead_id', $leadIds)
                ->selectRaw('lead_id, COUNT(*) as count')
                ->groupBy('lead_id')
                ->pluck('count', 'lead_id')
                ->toArray();

            $data = $leads->map(function($lead) use ($unlockCounts) {
                $unlockCount = $unlockCounts[$lead->id] ?? 0;
                $isExpired = $lead->expires_at && $lead->expires_at->isPast();
                
                return [
                    'id' => $lead->id,
                    'category' => [
                        'name' => $lead->category->name ?? 'General Request'
                    ],
                    'service' => [
                        'source_name' => $lead->service->source_name ?? 'General Service'
                    ],
                    'message_text' => $lead->message_text ?? 'No message provided',
                    'quality_score' => $lead->quality_score,
                    // Price removed - only providers see this when unlocking
                    'providers_interested' => $unlockCount,
                    'marketplace_status' => $lead->marketplace_status ?? 'new',
                    'created_at' => $lead->created_at->toISOString(),
                    'submitted_at' => $lead->created_at->format('Y-m-d H:i'),
                    'expires_at' => $lead->expires_at ? $lead->expires_at->format('Y-m-d H:i') : null,
                    'is_expired' => $isExpired,
                    'time_ago' => $lead->created_at->diffForHumans(),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $data,
                'pagination' => [
                    'current_page' => $leads->currentPage(),
                    'last_page' => $leads->lastPage(),
                    'per_page' => $leads->perPage(),
                    'total' => $leads->total(),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching client requests: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch your requests'
            ], 500);
        }
    }

    /**
     * Get providers interested in a specific lead
     */
    public function getInterestedProviders(Request $request, $leadId): JsonResponse
    {
        try {
            $userId = Auth::id();

            // Verify this lead belongs to the user
            $lead = UserFormInput::where('id', $leadId)
                ->where('user_id', $userId)
                ->firstOrFail();

            // Get providers who unlocked this lead
            $unlocks = LeadUnlock::with([
                    'provider:id,name,email,phone_number',
                    'provider.userDetails:user_id,profile_image'
                ])
                ->where('lead_id', $leadId)
                ->orderBy('unlocked_at', 'desc')
                ->get();

            $data = $unlocks->map(function($unlock) use ($leadId) {
                return [
                    'unlock_id' => $unlock->id,
                    'provider_id' => $unlock->provider_id,
                    'provider_name' => $unlock->provider->name ?? 'Unknown',
                    'provider_email' => $unlock->provider->email ?? '',
                    'provider_phone' => $unlock->provider->phone_number ?? '',
                    'provider_profile_image' => $unlock->provider->userDetails->profile_image 
                        ? asset('storage/profile/' . $unlock->provider->userDetails->profile_image) 
                        : null,
                    'unlocked_at' => $unlock->unlocked_at->diffForHumans(),
                    'connection_status' => $unlock->connection_status,
                    'lead_id' => $leadId,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'providers' => $data,
                    'lead' => [
                        'category' => $lead->category->name ?? 'N/A',
                        'service' => $lead->service->source_name ?? 'Not specified',
                        'message_preview' => substr($lead->message_text, 0, 200),
                        // Price removed - only providers see pricing
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching interested providers: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch interested providers'
            ], 500);
        }
    }

    /**
     * Accept provider connection request
     */
    public function acceptConnection(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();
            $unlockId = $request->unlock_id;

            Log::info('Accept connection request received', [
                'user_id' => $userId,
                'unlock_id' => $unlockId,
                'all_request_data' => $request->all()
            ]);

            $unlock = LeadUnlock::with('lead')
                ->where('id', $unlockId)
                ->firstOrFail();

            // Verify lead belongs to user
            if ($unlock->lead->user_id !== $userId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            // Allow re-accepting even if already accepted (client can change their mind)
            if ($unlock->connection_status === 'accepted') {
                return response()->json([
                    'success' => true,
                    'message' => 'This provider is already connected'
                ]);
            }

            // Decline any currently accepted providers for this lead (only one can be active)
            LeadUnlock::where('lead_id', $unlock->lead_id)
                ->where('connection_status', 'accepted')
                ->where('id', '!=', $unlockId)
                ->update([
                    'connection_status' => 'declined',
                    'connection_responded_at' => now()
                ]);

            $unlock->acceptConnection();

            // Send notification to provider about connection acceptance
            $notificationService = new MarketplaceNotificationService();
            $notificationService->notifyConnectionAccepted(
                $unlock->provider_id,
                $userId,
                $unlock->lead_id
            );

            Log::info('Connection request accepted', [
                'unlock_id' => $unlockId,
                'client_id' => $userId,
                'provider_id' => $unlock->provider_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Connection accepted! Provider can now contact you.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error accepting connection: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred'
            ], 500);
        }
    }

    /**
     * Decline provider connection request
     */
    public function declineConnection(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();
            $unlockId = $request->unlock_id;

            $unlock = LeadUnlock::with('lead')
                ->where('id', $unlockId)
                ->firstOrFail();

            // Verify lead belongs to user
            if ($unlock->lead->user_id !== $userId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            // Allow declining even if already declined or accepted (client can change their mind)
            if ($unlock->connection_status === 'declined') {
                return response()->json([
                    'success' => true,
                    'message' => 'Connection already declined'
                ]);
            }

            $unlock->declineConnection();

            Log::info('Connection request declined', [
                'unlock_id' => $unlockId,
                'client_id' => $userId,
                'provider_id' => $unlock->provider_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Connection declined.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error declining connection: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred'
            ], 500);
        }
    }

    /**
     * Cancel a submitted request
     */
    public function cancelRequest(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();
            $leadId = $request->lead_id;

            $lead = UserFormInput::where('id', $leadId)
                ->where('user_id', $userId)
                ->firstOrFail();

            if ($lead->marketplace_status === 'cancelled') {
                return response()->json([
                    'success' => false,
                    'message' => 'Request already cancelled'
                ], 400);
            }

            $lead->marketplace_status = 'cancelled';
            $lead->save();

            Log::info('Lead request cancelled', [
                'lead_id' => $leadId,
                'client_id' => $userId
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Request cancelled successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error cancelling request: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred'
            ], 500);
        }
    }

    /**
     * Display linked providers page
     */
    public function linkedProviders()
    {
        $data['page'] = 'user.linked-providers';
        $data['title'] = __('Linked Providers');
        $data['asset'] = 'linked-providers';
        
        return view('leads::client.linked-providers.index', $data);
    }

    /**
     * Get all linked providers for current user
     */
    public function getLinkedProvidersList(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();

            // Get all accepted connections for this user's leads
            $linkedProviders = LeadUnlock::with([
                    'provider:id,name,email,phone_number',
                    'provider.userDetails:user_id,profile_image',
                    'lead:id,category_id,service_id,message_text',
                    'lead.category:id,name',
                    'lead.service:id,source_name'
                ])
                ->whereHas('lead', function($q) use ($userId) {
                    $q->where('user_id', $userId);
                })
                ->where('connection_status', 'accepted')
                ->orderBy('connection_responded_at', 'desc')
                ->get();

            $data = $linkedProviders->map(function($unlock) {
                return [
                    'unlock_id' => $unlock->id,
                    'provider_id' => $unlock->provider_id,
                    'provider_name' => $unlock->provider->name ?? 'Unknown',
                    'provider_email' => $unlock->provider->email ?? '',
                    'provider_phone' => $unlock->provider->phone_number ?? '',
                    'provider_profile_image' => $unlock->provider->userDetails->profile_image 
                        ? asset('storage/profile/' . $unlock->provider->userDetails->profile_image) 
                        : null,
                    'request_id' => $unlock->lead_id,
                    'request_category' => $unlock->lead->category->name ?? 'N/A',
                    'request_service' => $unlock->lead->service->source_name ?? null,
                    'connected_at' => $unlock->connection_responded_at->diffForHumans(),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching linked providers: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch linked providers'
            ], 500);
        }
    }

    /**
     * Delink a provider (switch to another provider)
     */
    public function delinkProvider(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();
            $unlockId = $request->unlock_id;

            $unlock = LeadUnlock::with('lead')
                ->where('id', $unlockId)
                ->firstOrFail();

            // Verify lead belongs to user
            if ($unlock->lead->user_id !== $userId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            // Can only delink accepted connections
            if ($unlock->connection_status !== 'accepted') {
                return response()->json([
                    'success' => false,
                    'message' => 'Can only delink accepted connections'
                ], 400);
            }

            // Delink by setting status to declined
            $unlock->connection_status = 'declined';
            $unlock->connection_responded_at = now();
            $unlock->save();

            Log::info('Provider delinked', [
                'unlock_id' => $unlockId,
                'client_id' => $userId,
                'provider_id' => $unlock->provider_id,
                'lead_id' => $unlock->lead_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Provider delinked successfully. You can now connect with a different provider.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error delinking provider: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred'
            ], 500);
        }
    }
}

