<?php

namespace Modules\Leads\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Modules\Leads\app\Models\UserFormInput;
use Modules\Leads\app\Models\LeadUnlock;
use Modules\Leads\app\Models\LeadPricingRule;
use Modules\Leads\app\Models\UserHelpFormResponse;
use Modules\Categories\app\Models\Categories;
use App\Models\WalletHistory;
use App\Services\MarketplaceWalletService;
use App\Services\MarketplaceNotificationService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ProviderMarketplaceController extends Controller
{
    /**
     * Display provider marketplace page
     */
    public function index()
    {
        $data['page'] = 'provider.marketplace';
        $data['title'] = __('Marketplace Leads');
        $data['asset'] = 'marketplace-leads';
        
        return view('leads::provider.marketplace.index', $data);
    }

    /**
     * Get marketplace leads for provider
     */
    public function getLeads(Request $request): JsonResponse
    {
        try {
            $providerId = Auth::id();
            
            // Get provider's category
            $provider = Auth::user();
            $providerCategoryId = $provider->category_id;

            // Build query for marketplace leads
            // Show leads that are: marketplace source, not expired, and NOT unlocked by THIS provider
            $query = UserFormInput::with(['category:id,name', 'service:id,source_name', 'user:id,name,email,phone_number'])
                ->where('lead_source', 'marketplace')
                ->whereIn('marketplace_status', ['pending', 'unlocked'])  // Show both statuses
                ->where(function($q) {
                    $q->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
                })
                ->whereNotExists(function($subquery) use ($providerId) {
                    $subquery->select(DB::raw(1))
                             ->from('lead_unlocks')
                             ->whereColumn('lead_unlocks.lead_id', 'user_form_inputs.id')
                             ->where('lead_unlocks.provider_id', $providerId);
                });

            // Filter by provider's category (optional - remove if you want to show all categories)
            // if ($providerCategoryId) {
            //     $query->where('category_id', $providerCategoryId);
            // }

            // Apply filters
            if ($request->filled('category_id')) {
                $query->where('category_id', $request->category_id);
            }

            if ($request->filled('min_quality')) {
                $query->where('quality_score', '>=', $request->min_quality);
            }

            if ($request->filled('max_price')) {
                $query->where('calculated_unlock_price', '<=', $request->max_price);
            }

            if ($request->filled('location')) {
                $location = $request->location;
                // Sanitize location input to prevent SQL injection
                $location = preg_replace('/[^a-zA-Z0-9\s\-]/', '', $location);
                $location = trim($location);
                
                if (!empty($location)) {
                    $query->where(function($q) use ($location) {
                        $q->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(form_inputs, '$.city')) LIKE ?", ["%{$location}%"])
                          ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(form_inputs, '$.state')) LIKE ?", ["%{$location}%"]);
                    });
                }
            }

            // Order by quality score and date
            $query->orderBy('quality_score', 'desc')
                  ->orderBy('created_at', 'desc');

            $leads = $query->paginate(20);

            // Check which leads the provider has already unlocked
            $unlockedLeadIds = LeadUnlock::where('provider_id', $providerId)
                ->pluck('lead_id')
                ->toArray();

            // Transform data for frontend
            $data = $leads->map(function($lead) use ($unlockedLeadIds) {
                $formInputs = is_array($lead->form_inputs) ? $lead->form_inputs : json_decode($lead->form_inputs, true);
                $alreadyUnlocked = in_array($lead->id, $unlockedLeadIds);
                
                // Get question response count
                $questionsCount = UserHelpFormResponse::where('lead_id', $lead->id)
                    ->distinct('question_id')
                    ->count('question_id');
                
                // Get question responses for display
                $questionResponses = UserHelpFormResponse::with(['question', 'answer'])
                    ->where('lead_id', $lead->id)
                    ->get()
                    ->map(function($response) {
                        return [
                            'question' => $response->question->question_text ?? 'N/A',
                            'answer' => $response->answer->answer_text ?? $response->answer_text ?? 'N/A'
                        ];
                    });
                
                // Get user details - use form inputs for name if available, otherwise user table
                $userEmail = $lead->user->email ?? 'N/A';
                $userPhone = $lead->user->phone_number ?? 'N/A';
                
                // Try to get name from form inputs first (more accurate)
                $userName = 'N/A';
                if (isset($formInputs['first_name']) && isset($formInputs['last_name'])) {
                    $userName = trim($formInputs['first_name'] . ' ' . $formInputs['last_name']);
                } elseif (isset($formInputs['first_name'])) {
                    $userName = $formInputs['first_name'];
                } elseif ($lead->user->name) {
                    $userName = $lead->user->name;
                }
                
                if (!$alreadyUnlocked) {
                    // Only blur email and phone, keep full name visible
                    $userEmail = $this->blurEmail($userEmail);
                    $userPhone = $this->blurPhone($userPhone);
                }
                
                return [
                    'id' => $lead->id,
                    'category' => $lead->category->name ?? 'N/A',
                    'service' => $lead->service->source_name ?? null,
                    'message_text' => $lead->message_text ?? '',
                    'message_preview' => $this->getBlurredPreview($lead->message_text, 100),
                    'quality_score' => $lead->quality_score,
                    'unlock_price' => number_format((float)$lead->calculated_unlock_price, 2),
                    'calculated_unlock_price' => (float)$lead->calculated_unlock_price,
                    'fields_filled' => $lead->form_fields_filled_count,
                    'message_length' => $lead->message_char_count,
                    'location' => ($formInputs['city'] ?? '') . ', ' . ($formInputs['state'] ?? ''),
                    'country' => $formInputs['country'] ?? '',
                    'posted_at' => $lead->created_at->diffForHumans(),
                    'created_at' => $lead->created_at->toISOString(),
                    'is_new' => $lead->created_at->gt(now()->subHours(24)),
                    'is_high_quality' => $lead->quality_score >= 80,
                    'already_unlocked' => $alreadyUnlocked,
                    'questions_answered' => $questionsCount,
                    'question_responses' => $questionResponses,
                    'user' => [
                        'name' => $userName,
                        'email' => $userEmail,
                        'phone' => $userPhone
                    ],
                    'form_inputs' => $formInputs,
                ];
            });

            // Get stats for Quick Stats display
            $totalAvailable = UserFormInput::where('lead_source', 'marketplace')
                ->whereIn('marketplace_status', ['pending', 'unlocked'])
                ->where(function($q) {
                    $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
                })
                ->whereNotExists(function($subquery) use ($providerId) {
                    $subquery->select(DB::raw(1))
                             ->from('lead_unlocks')
                             ->whereColumn('lead_unlocks.lead_id', 'user_form_inputs.id')
                             ->where('lead_unlocks.provider_id', $providerId);
                })
                ->count();
                
            $unlockedCount = LeadUnlock::where('provider_id', $providerId)->count();
            $pendingCount = LeadUnlock::where('provider_id', $providerId)
                ->where('connection_status', 'pending')
                ->count();

            return response()->json([
                'success' => true,
                'data' => $data,
                'stats' => [
                    'total' => $totalAvailable,
                    'unlocked' => $unlockedCount,
                    'pending' => $pendingCount,
                ],
                'pagination' => [
                    'current_page' => $leads->currentPage(),
                    'last_page' => $leads->lastPage(),
                    'per_page' => $leads->perPage(),
                    'total' => $leads->total(),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching marketplace leads: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch marketplace leads'
            ], 500);
        }
    }

    /**
     * Get wallet balance
     */
    public function getWalletBalance(): JsonResponse
    {
        try {
            $providerId = Auth::id();
            $walletService = new MarketplaceWalletService();
            $balance = $walletService->getBalance($providerId);

            return response()->json([
                'success' => true,
                'balance' => number_format($balance, 2),
                'balance_raw' => $balance
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching wallet balance: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch wallet balance'
            ], 500);
        }
    }

    /**
     * Unlock lead (debit wallet and reveal details)
     */
    public function unlockLead(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'lead_id' => 'required|exists:user_form_inputs,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $providerId = Auth::id();
            $leadId = $request->lead_id;

            Log::info('Lead unlock attempt', ['provider_id' => $providerId, 'lead_id' => $leadId]);

            // Get lead
            $lead = UserFormInput::findOrFail($leadId);

            // Check if already unlocked by this provider
            $existingUnlock = LeadUnlock::where('lead_id', $leadId)
                ->where('provider_id', $providerId)
                ->first();

            if ($existingUnlock) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'You have already unlocked this lead'
                ], 400);
            }

            // Check if lead is still available (allow multiple providers to unlock)
            // Only block if lead is explicitly declined or cancelled by client
            if (in_array($lead->marketplace_status, ['declined', 'cancelled', 'completed'])) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'This lead is no longer available'
                ], 400);
            }

            // Check if lead has expired
            if ($lead->expires_at && $lead->expires_at->isPast()) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'This lead has expired'
                ], 400);
            }

            // Check if provider has sufficient balance using marketplace wallet service
            $walletService = new MarketplaceWalletService();
            $unlockPrice = (float)$lead->calculated_unlock_price;
            
            if (!$walletService->hasSufficientBalance($providerId, $unlockPrice)) {
                DB::rollBack();
                $balance = $walletService->getBalance($providerId);
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient wallet balance',
                    'required' => number_format($unlockPrice, 2),
                    'current' => number_format($balance, 2),
                    'shortfall' => number_format($unlockPrice - $balance, 2)
                ], 400);
            }

            // Debit wallet using marketplace wallet service
            $categoryName = $lead->category ? $lead->category->name : 'Lead';
            $description = "Marketplace Lead Unlock - {$categoryName} #{$lead->id}";
            $walletTransaction = $walletService->deductProviderWallet($providerId, $unlockPrice, $leadId, $description);
            
            if (!$walletTransaction) {
                DB::rollBack();
                Log::error('Wallet deduction failed', ['provider_id' => $providerId, 'lead_id' => $leadId, 'amount' => $unlockPrice]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to process wallet payment'
                ], 500);
            }

            // Create lead unlock record
            $leadUnlock = LeadUnlock::create([
                'lead_id' => $leadId,
                'provider_id' => $providerId,
                'credits_spent' => $unlockPrice,
                'wallet_transaction_id' => $walletTransaction->id,
                'connection_status' => 'pending',
            ]);

            // Update lead status (first unlock)
            if (!$lead->is_unlocked) {
                $lead->is_unlocked = true;
                $lead->unlocked_by_provider_id = $providerId;
                $lead->unlocked_at = now();
                $lead->marketplace_status = 'unlocked';
                $lead->save();
            }

            // Send notification to client about lead unlock
            try {
                $notificationService = new MarketplaceNotificationService();
                $notificationService->notifyLeadUnlocked(
                    $lead->user_id,
                    $providerId,
                    $leadId,
                    $lead->category->name ?? 'Lead'
                );
            } catch (\Exception $e) {
                // Log but don't fail the unlock if notification fails
                Log::warning('Failed to send lead unlock notification', [
                    'error' => $e->getMessage(),
                    'provider_id' => $providerId,
                    'lead_id' => $leadId
                ]);
            }

            // Get full lead details
            $formInputs = is_array($lead->form_inputs) ? $lead->form_inputs : json_decode($lead->form_inputs, true);
            
            $leadDetails = [
                'id' => $lead->id,
                'category' => $lead->category->name ?? 'N/A',
                'service' => $lead->service->source_name ?? 'Not specified',
                'client_name' => ($formInputs['first_name'] ?? '') . ' ' . ($formInputs['last_name'] ?? ''),
                'client_email' => $formInputs['email'] ?? '',
                'client_phone' => $formInputs['phone'] ?? '',
                'client_gender' => $formInputs['gender'] ?? '',
                'client_dob' => $formInputs['dob'] ?? '',
                'address' => $formInputs['address'] ?? '',
                'city' => $formInputs['city'] ?? '',
                'state' => $formInputs['state'] ?? '',
                'country' => $formInputs['country'] ?? '',
                'postal_code' => $formInputs['postal_code'] ?? '',
                'message_full' => $lead->message_text,
                'quality_score' => $lead->quality_score,
                'fields_filled' => $lead->form_fields_filled_count,
                'posted_at' => $lead->created_at->format('Y-m-d H:i'),
                'unlock_id' => $leadUnlock->id,
            ];

            DB::commit();

            Log::info('Lead unlocked successfully', [
                'lead_id' => $leadId,
                'provider_id' => $providerId,
                'credits_spent' => $unlockPrice
            ]);

            // Get new balance after unlock
            $newBalance = $walletService->getBalance($providerId);

            return response()->json([
                'success' => true,
                'message' => 'Lead unlocked successfully!',
                'lead_details' => $leadDetails,
                'new_balance' => number_format($newBalance, 2)
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Lead unlock failed', [
                'lead_id' => $request->lead_id ?? 'unknown',
                'provider_id' => Auth::id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while unlocking the lead: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get unlocked leads for provider
     */
    public function getUnlockedLeads(Request $request): JsonResponse
    {
        try {
            $providerId = Auth::id();

            $query = LeadUnlock::with(['lead.category', 'lead.service', 'lead.user'])
                ->where('provider_id', $providerId)
                ->orderBy('unlocked_at', 'desc');

            // Filter by connection status
            if ($request->filled('status')) {
                $query->where('connection_status', $request->status);
            }

            $unlocks = $query->paginate(20);

            $data = $unlocks->map(function($unlock) {
                $lead = $unlock->lead;
                $formInputs = is_array($lead->form_inputs) ? $lead->form_inputs : json_decode($lead->form_inputs, true);
                
                return [
                    'unlock_id' => $unlock->id,
                    'lead_id' => $lead->id,
                    'category' => $lead->category->name ?? 'N/A',
                    'service' => $lead->service->source_name ?? 'Not specified',
                    'client_name' => ($formInputs['first_name'] ?? '') . ' ' . ($formInputs['last_name'] ?? ''),
                    'client_email' => $formInputs['email'] ?? '',
                    'client_phone' => $formInputs['phone'] ?? '',
                    'location' => ($formInputs['city'] ?? '') . ', ' . ($formInputs['state'] ?? ''),
                    'message_preview' => substr($lead->message_text, 0, 150) . '...',
                    'unlocked_at' => $unlock->unlocked_at->diffForHumans(),
                    'credits_spent' => number_format($unlock->credits_spent, 2),
                    'connection_status' => $unlock->connection_status,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $data,
                'pagination' => [
                    'current_page' => $unlocks->currentPage(),
                    'last_page' => $unlocks->lastPage(),
                    'per_page' => $unlocks->perPage(),
                    'total' => $unlocks->total(),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching unlocked leads: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch unlocked leads'
            ], 500);
        }
    }

    /**
     * Send connection request to client
     */
    public function sendConnectionRequest(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'unlock_id' => 'required|exists:lead_unlocks,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $providerId = Auth::id();
            $unlock = LeadUnlock::where('id', $request->unlock_id)
                ->where('provider_id', $providerId)
                ->firstOrFail();

            if ($unlock->connection_status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Connection request already processed'
                ], 400);
            }

            // Send notification to client about connection request
            $notificationService = new MarketplaceNotificationService();
            $message = $request->message ?? '';
            $notificationService->notifyConnectionRequest(
                $unlock->lead->user_id,
                $providerId,
                $unlock->lead_id,
                $message
            );

            Log::info('Connection request sent', [
                'unlock_id' => $unlock->id,
                'provider_id' => $providerId,
                'lead_id' => $unlock->lead_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Connection request sent to client'
            ]);
        } catch (\Exception $e) {
            Log::error('Error sending connection request: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred'
            ], 500);
        }
    }

    /**
     * Get blurred preview of text
     */
    private function getBlurredPreview(string $text, int $length = 100): string
    {
        $preview = substr($text, 0, $length);
        $words = explode(' ', $preview);
        
        // Blur every other word
        $blurred = array_map(function($word, $index) {
            return $index % 2 === 1 ? str_repeat('█', strlen($word)) : $word;
        }, $words, array_keys($words));
        
        return implode(' ', $blurred) . '...';
    }

    /**
     * Blur email address
     */
    private function blurEmail(string $email): string
    {
        if (strpos($email, '@') === false) {
            return str_repeat('*', strlen($email));
        }
        
        $parts = explode('@', $email);
        $username = $parts[0];
        $domain = $parts[1];
        
        if (strlen($username) <= 2) {
            $blurredUsername = str_repeat('*', strlen($username));
        } else {
            $blurredUsername = substr($username, 0, 2) . str_repeat('*', strlen($username) - 2);
        }
        
        return $blurredUsername . '@' . $domain;
    }

    /**
     * Blur phone number
     */
    private function blurPhone(string $phone): string
    {
        if (strlen($phone) <= 4) {
            return str_repeat('*', strlen($phone));
        }
        
        return substr($phone, 0, 3) . str_repeat('*', strlen($phone) - 6) . substr($phone, -3);
    }
}

