<?php

namespace Modules\Leads\app\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Modules\Categories\app\Models\Categories;
use App\Models\User;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * LeadPricingRule Model
 * 
 * Manages dynamic pricing rules for marketplace leads.
 * Admin configures pricing per category.
 * 
 * @property int $id
 * @property int $category_id
 * @property float $base_price
 * @property float $price_per_field
 * @property float $price_per_100_chars
 * @property float $min_price
 * @property float $max_price
 * @property bool $active
 * @property int|null $created_by_admin_id
 * @property int|null $updated_by_admin_id
 * @property \Carbon\Carbon $created_at
 * @property \Carbon\Carbon $updated_at
 * @property \Carbon\Carbon|null $deleted_at
 */
class LeadPricingRule extends Model
{
    use HasFactory;

    protected $table = 'lead_pricing_rules';

    protected $fillable = [
        'category_id',
        'base_price',
        'price_per_field',
        'price_per_100_chars',
        'min_price',
        'max_price',
        'active',
        'created_by_admin_id',
        'updated_by_admin_id',
    ];

    protected $casts = [
        'base_price' => 'decimal:2',
        'price_per_field' => 'decimal:2',
        'price_per_100_chars' => 'decimal:2',
        'min_price' => 'decimal:2',
        'max_price' => 'decimal:2',
        'active' => 'boolean',
    ];

    /**
     * Get the category this pricing rule belongs to
     *
     * @return BelongsTo<Categories, LeadPricingRule>
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Categories::class, 'category_id');
    }

    /**
     * Get the admin who created this rule
     *
     * @return BelongsTo<User, LeadPricingRule>
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_admin_id');
    }

    /**
     * Get the admin who last updated this rule
     *
     * @return BelongsTo<User, LeadPricingRule>
     */
    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by_admin_id');
    }

    /**
     * Calculate unlock price for a lead
     * 
     * @param int $fieldsFilledCount Number of fields filled in the lead form
     * @param int $messageCharCount Number of characters in the message
     * @return float Calculated unlock price (capped by min/max)
     */
    public function calculateUnlockPrice(int $fieldsFilledCount, int $messageCharCount): float
    {
        $price = $this->base_price;
        $price += ($fieldsFilledCount * $this->price_per_field);
        $price += (ceil($messageCharCount / 100) * $this->price_per_100_chars);

        // Apply min/max caps
        $price = max($this->min_price, $price);
        $price = min($this->max_price, $price);

        return round($price, 2);
    }

    /**
     * Scope to get only active pricing rules
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    /**
     * Get pricing rule for a specific category
     *
     * @param int $categoryId
     * @return LeadPricingRule|null
     */
    public static function getForCategory(int $categoryId): ?LeadPricingRule
    {
        return static::where('category_id', $categoryId)
            ->where('active', true)
            ->first();
    }
}

