<?php

namespace Modules\Leads\app\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\User;
use App\Models\WalletHistory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * LeadUnlock Model
 * 
 * Tracks which providers have unlocked which leads
 * and manages the connection request flow.
 * 
 * @property int $id
 * @property int $lead_id
 * @property int $provider_id
 * @property float $credits_spent
 * @property int|null $wallet_transaction_id
 * @property string $connection_status
 * @property string|null $connection_message
 * @property \Carbon\Carbon|null $connection_requested_at
 * @property \Carbon\Carbon|null $connection_responded_at
 * @property \Carbon\Carbon $unlocked_at
 * @property \Carbon\Carbon $created_at
 * @property \Carbon\Carbon $updated_at
 * @property \Carbon\Carbon|null $deleted_at
 */
class LeadUnlock extends Model
{
    use HasFactory;

    protected $table = 'lead_unlocks';

    protected $fillable = [
        'lead_id',
        'provider_id',
        'credits_spent',
        'wallet_transaction_id',
        'connection_status',
        'connection_message',
        'connection_requested_at',
        'connection_responded_at',
        'unlocked_at',
    ];

    protected $casts = [
        'credits_spent' => 'decimal:2',
        'connection_requested_at' => 'datetime',
        'connection_responded_at' => 'datetime',
        'unlocked_at' => 'datetime',
    ];

    /**
     * The attributes that should be appended to the model
     *
     * @var array
     */
    protected $appends = ['is_connection_pending', 'is_connection_accepted', 'is_connection_declined'];

    /**
     * Get the lead that was unlocked
     *
     * @return BelongsTo<UserFormInput, LeadUnlock>
     */
    public function lead(): BelongsTo
    {
        return $this->belongsTo(UserFormInput::class, 'lead_id');
    }

    /**
     * Get the provider who unlocked the lead
     *
     * @return BelongsTo<User, LeadUnlock>
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(User::class, 'provider_id');
    }

    /**
     * Get the wallet transaction for this unlock
     *
     * @return BelongsTo<WalletHistory, LeadUnlock>
     */
    public function walletTransaction(): BelongsTo
    {
        return $this->belongsTo(WalletHistory::class, 'wallet_transaction_id');
    }

    /**
     * Check if connection request is pending
     *
     * @return bool
     */
    public function getIsConnectionPendingAttribute(): bool
    {
        return $this->connection_status === 'pending';
    }

    /**
     * Check if connection request is accepted
     *
     * @return bool
     */
    public function getIsConnectionAcceptedAttribute(): bool
    {
        return $this->connection_status === 'accepted';
    }

    /**
     * Check if connection request is declined
     *
     * @return bool
     */
    public function getIsConnectionDeclinedAttribute(): bool
    {
        return $this->connection_status === 'declined';
    }

    /**
     * Scope to get unlocks by provider
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $providerId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByProvider($query, int $providerId)
    {
        return $query->where('provider_id', $providerId);
    }

    /**
     * Scope to get unlocks by lead
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $leadId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByLead($query, int $leadId)
    {
        return $query->where('lead_id', $leadId);
    }

    /**
     * Scope to get pending connections
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePendingConnections($query)
    {
        return $query->where('connection_status', 'pending');
    }

    /**
     * Scope to get accepted connections
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAcceptedConnections($query)
    {
        return $query->where('connection_status', 'accepted');
    }

    /**
     * Send connection request to client
     *
     * @param string $message
     * @return bool
     */
    public function sendConnectionRequest(string $message): bool
    {
        $this->connection_message = $message;
        $this->connection_requested_at = now();
        $this->connection_status = 'pending';
        
        return $this->save();
    }

    /**
     * Accept connection request
     *
     * @return bool
     */
    public function acceptConnection(): bool
    {
        $this->connection_status = 'accepted';
        $this->connection_responded_at = now();
        
        return $this->save();
    }

    /**
     * Decline connection request
     *
     * @return bool
     */
    public function declineConnection(): bool
    {
        $this->connection_status = 'declined';
        $this->connection_responded_at = now();
        
        return $this->save();
    }
}

