<?php

namespace Modules\Leads\app\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Modules\Categories\app\Models\Categories;
use Modules\Categories\app\Models\CategoryFormInput;
use Modules\Product\app\Models\Product;
use App\Models\User;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * @property int $user_id
 * @property int|null $category_id
 * @property int|null $sub_category_id
 * @property array<int, array{id: int, value: mixed}>|null $form_inputs
 * @property string $status
 * @property string $lead_source
 * @property int|null $service_id
 * @property string|null $message_text
 * @property int $message_char_count
 * @property int $form_fields_filled_count
 * @property float $calculated_unlock_price
 * @property int $quality_score
 * @property bool $is_unlocked
 * @property int|null $unlocked_by_provider_id
 * @property \Carbon\Carbon|null $unlocked_at
 * @property bool $blur_details
 * @property string $marketplace_status
 * @property \Carbon\Carbon|null $expires_at
 * @property-read Categories|null $category
 * @property-read Categories|null $subCategory
 * @property-read User|null $user
 * @property-read array<int, array{title: string|null, description: string|null, option: mixed|null}>|null $form_inputs_details
 */
class UserFormInput extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'category_id',
        'sub_category_id',
        'form_inputs',
        'status',
        // Marketplace fields
        'lead_source',
        'service_id',
        'message_text',
        'message_char_count',
        'form_fields_filled_count',
        'calculated_unlock_price',
        'quality_score',
        'is_unlocked',
        'unlocked_by_provider_id',
        'unlocked_at',
        'blur_details',
        'marketplace_status',
        'expires_at',
    ];

    protected $casts = [
        'form_inputs' => 'array',
        'calculated_unlock_price' => 'decimal:2',
        'is_unlocked' => 'boolean',
        'blur_details' => 'boolean',
        'unlocked_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    /**
     * Define a relationship with the Categories model.
     *
     * @return BelongsTo<Categories, UserFormInput>
     */
    public function category(): BelongsTo
    {
        /** @var BelongsTo<Categories, UserFormInput> */
        return $this->belongsTo(Categories::class, 'category_id');
    }

    /**
     * Define a relationship with the sub-category (Categories model).
     *
     * @return BelongsTo<Categories, UserFormInput>
     */
    public function subCategory(): BelongsTo
    {
        /** @var BelongsTo<Categories, UserFormInput> */
        return $this->belongsTo(Categories::class, 'sub_category_id')->whereNotNull('parent_id');
    }

    /**
     * Define a relationship with the User model.
     *
     * @return BelongsTo<User, UserFormInput>
     */
    public function user(): BelongsTo
    {
        /** @var BelongsTo<User, UserFormInput> */
        return $this->belongsTo(User::class);
    }

    /**
     * Get question responses for this lead
     *
     * @return HasMany<UserHelpFormResponse, UserFormInput>
     */
    public function questionResponses(): HasMany
    {
        /** @var HasMany<UserHelpFormResponse, UserFormInput> */
        return $this->hasMany(UserHelpFormResponse::class, 'lead_id');
    }

    /**
     * Accessor to get detailed form inputs.
     *
     * @return array<int, array{title: string|null, description: string|null, option: mixed|null}>|null
     */
    public function getFormInputsDetailsAttribute(): ?array
    {
        $formInputs = $this->attributes['form_inputs'];
        
        // If it's already an array, return it
        if (is_array($formInputs)) {
            return $formInputs;
        }
        
        // If it's a string, decode it
        if (is_string($formInputs)) {
            $decoded = json_decode($formInputs, true);
            
            // For marketplace leads, the form_inputs is already a simple JSON object
            // For regular form leads, it might have category form input details
            if (is_array($decoded)) {
                // Check if this is a marketplace lead (has simple structure)
                if (isset($decoded['first_name']) || isset($decoded['email'])) {
                    return $decoded; // Marketplace lead - return as is
                }
                
                // Regular form lead - add category form input details
                foreach ($decoded as &$input) {
                    if (isset($input['id'])) {
                        $categoryFormInput = CategoryFormInput::find($input['id']);
                        if ($categoryFormInput) {
                            $input['details'] = [
                                'title' => $categoryFormInput->label ?? null,
                                'description' => $categoryFormInput->name ?? null,
                                'option' => $categoryFormInput->options ?? null,
                            ];
                        }
                    }
                }
                return $decoded;
            }
        }

        return null;
    }

    /**
     * Define a relationship with the ProviderFormsInput model.
     *
     * @return HasMany<ProviderFormsInput, UserFormInput>
     */
    public function providerFormsInputs(): HasMany
    {
        /** @var HasMany<ProviderFormsInput, UserFormInput> */
        return $this->hasMany(ProviderFormsInput::class, 'user_form_inputs_id', 'id');
    }

    // ============================================================
    // MARKETPLACE RELATIONSHIPS (Added for marketplace feature)
    // ============================================================

    /**
     * Get the service/product for this lead (optional)
     *
     * @return BelongsTo<Product, UserFormInput>
     */
    public function service(): BelongsTo
    {
        /** @var BelongsTo<Product, UserFormInput> */
        return $this->belongsTo(Product::class, 'service_id');
    }

    /**
     * Get the provider who first unlocked this lead
     *
     * @return BelongsTo<User, UserFormInput>
     */
    public function unlockedByProvider(): BelongsTo
    {
        /** @var BelongsTo<User, UserFormInput> */
        return $this->belongsTo(User::class, 'unlocked_by_provider_id');
    }

    /**
     * Get all unlocks for this lead (all providers who purchased it)
     *
     * @return HasMany<LeadUnlock, UserFormInput>
     */
    public function unlocks(): HasMany
    {
        /** @var HasMany<LeadUnlock, UserFormInput> */
        return $this->hasMany(LeadUnlock::class, 'lead_id');
    }

    // ============================================================
    // MARKETPLACE SCOPES
    // ============================================================

    /**
     * Scope to get only marketplace leads
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeMarketplace($query)
    {
        return $query->where('lead_source', 'marketplace');
    }

    /**
     * Scope to get only traditional form leads
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForm($query)
    {
        return $query->where('lead_source', 'form');
    }

    /**
     * Scope to get unlocked leads
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeUnlocked($query)
    {
        return $query->where('is_unlocked', true);
    }

    /**
     * Scope to get pending (not yet unlocked) leads
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePending($query)
    {
        return $query->where('is_unlocked', false)
            ->where('marketplace_status', 'pending');
    }

    /**
     * Scope to get non-expired marketplace leads
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('expires_at')
              ->orWhere('expires_at', '>', now());
        });
    }

    // ============================================================
    // MARKETPLACE HELPER METHODS
    // ============================================================

    /**
     * Check if this is a marketplace lead
     *
     * @return bool
     */
    public function isMarketplaceLead(): bool
    {
        return $this->lead_source === 'marketplace';
    }

    /**
     * Check if lead has been unlocked by any provider
     *
     * @return bool
     */
    public function isUnlocked(): bool
    {
        return $this->is_unlocked === true;
    }

    /**
     * Check if lead is expired
     *
     * @return bool
     */
    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Get the count of providers who unlocked this lead
     *
     * @return int
     */
    public function getUnlockCount(): int
    {
        return $this->unlocks()->count();
    }

    /**
     * Check if a specific provider has unlocked this lead
     *
     * @param int $providerId
     * @return bool
     */
    public function isUnlockedByProvider(int $providerId): bool
    {
        return $this->unlocks()->where('provider_id', $providerId)->exists();
    }
}
