<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * Adds marketplace-specific fields to the existing user_form_inputs table
     * to support the pay-per-lead marketplace functionality.
     *
     * @return void
     */
    public function up(): void
    {
        Schema::table('user_form_inputs', function (Blueprint $table) {
            // Source tracking
            $table->enum('lead_source', ['form', 'booking', 'marketplace'])->default('form')->after('category_id');
            
            // Marketplace-specific data
            $table->unsignedBigInteger('service_id')->nullable()->after('category_id')->comment('Optional service/product selection');
            $table->text('message_text')->nullable()->after('form_inputs')->comment('Client description of what they need');
            $table->integer('message_char_count')->default(0)->after('message_text')->comment('For pricing calculation');
            $table->integer('form_fields_filled_count')->default(0)->after('message_char_count')->comment('For pricing calculation');
            
            // Pricing
            $table->decimal('calculated_unlock_price', 10, 2)->default(0)->after('form_fields_filled_count')->comment('Auto-calculated based on pricing rules');
            $table->integer('quality_score')->default(0)->after('calculated_unlock_price')->comment('0-100 based on completeness');
            
            // Unlock tracking
            $table->boolean('is_unlocked')->default(false)->after('quality_score')->comment('Has any provider unlocked this lead');
            $table->unsignedBigInteger('unlocked_by_provider_id')->nullable()->after('is_unlocked')->comment('First provider who unlocked');
            $table->timestamp('unlocked_at')->nullable()->after('unlocked_by_provider_id');
            $table->boolean('blur_details')->default(true)->after('unlocked_at')->comment('Should details be blurred for providers');
            
            // Lead lifecycle
            $table->enum('marketplace_status', ['marketplace', 'pending', 'unlocked', 'connected', 'expired', 'declined'])->default('pending')->after('status');
            $table->timestamp('expires_at')->nullable()->after('marketplace_status')->comment('Auto-expire after 30 days');
            
            // Foreign keys (only for unlocked_by_provider_id, service_id can be null)
            $table->foreign('unlocked_by_provider_id')->references('id')->on('users')->onDelete('set null');
            
            // Indexes for performance
            $table->index('service_id'); // Index but no foreign key constraint
            $table->index('lead_source');
            $table->index('marketplace_status');
            $table->index('is_unlocked');
            $table->index('expires_at');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down(): void
    {
        Schema::table('user_form_inputs', function (Blueprint $table) {
            // Drop foreign keys first
            $table->dropForeign(['unlocked_by_provider_id']);
            
            // Drop indexes
            $table->dropIndex(['service_id']);
            $table->dropIndex(['lead_source']);
            $table->dropIndex(['marketplace_status']);
            $table->dropIndex(['is_unlocked']);
            $table->dropIndex(['expires_at']);
            
            // Drop columns
            $table->dropColumn([
                'lead_source',
                'service_id',
                'message_text',
                'message_char_count',
                'form_fields_filled_count',
                'calculated_unlock_price',
                'quality_score',
                'is_unlocked',
                'unlocked_by_provider_id',
                'unlocked_at',
                'blur_details',
                'marketplace_status',
                'expires_at'
            ]);
        });
    }
};

