<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * Creates the lead_unlocks table to track which providers
     * have purchased/unlocked which leads, including payment details.
     *
     * @return void
     */
    public function up(): void
    {
        Schema::create('lead_unlocks', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('lead_id')->comment('Reference to user_form_inputs.id');
            $table->unsignedBigInteger('provider_id')->comment('Provider who unlocked the lead');
            
            // Payment tracking
            $table->decimal('credits_spent', 10, 2)->comment('Amount debited from wallet');
            $table->unsignedBigInteger('wallet_transaction_id')->nullable()->comment('Reference to wallet transaction');
            
            // Connection status
            $table->enum('connection_status', ['pending', 'accepted', 'declined', 'expired'])->default('pending')->comment('Client response to connection request');
            $table->text('connection_message')->nullable()->comment('Provider message to client');
            $table->timestamp('connection_requested_at')->nullable();
            $table->timestamp('connection_responded_at')->nullable();
            
            // Unlock timestamp
            $table->timestamp('unlocked_at')->useCurrent();
            
            $table->timestamps();
            $table->softDeletes();
            
            // Foreign keys
            $table->foreign('lead_id')->references('id')->on('user_form_inputs')->onDelete('cascade');
            $table->foreign('provider_id')->references('id')->on('users')->onDelete('cascade');
            
            // Prevent duplicate unlocks by same provider
            $table->unique(['lead_id', 'provider_id'], 'unique_provider_lead_unlock');
            
            // Indexes for queries
            $table->index('unlocked_at');
            $table->index('connection_status');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down(): void
    {
        Schema::dropIfExists('lead_unlocks');
    }
};

