<?php

namespace Modules\Leads\Database\Seeders;

use Illuminate\Database\Seeder;
use Modules\Leads\app\Models\LeadPricingRule;
use Modules\Categories\app\Models\Categories;

class LeadPricingRulesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * Creates default pricing rules for all categories.
     * Categories are grouped into low/medium/high value tiers.
     */
    public function run(): void
    {
        // Get all top-level categories (parent_id is NULL or 0)
        $categories = Categories::where(function($query) {
            $query->whereNull('parent_id')->orWhere('parent_id', 0);
        })->get();

        if ($categories->isEmpty()) {
            $this->command->warn('No categories found. Please seed categories first.');
            return;
        }

        $this->command->info('Creating default pricing rules for ' . $categories->count() . ' categories...');

        // Define pricing tiers based on typical service values
        $pricingTiers = [
            'low' => [
                'base_price' => 5.00,
                'price_per_field' => 0.50,
                'price_per_100_chars' => 0.10,
                'min_price' => 5.00,
                'max_price' => 25.00,
                'categories' => ['cleaning', 'tutoring', 'pet', 'delivery', 'moving', 'gardening'],
            ],
            'medium' => [
                'base_price' => 15.00,
                'price_per_field' => 1.00,
                'price_per_100_chars' => 0.20,
                'min_price' => 10.00,
                'max_price' => 50.00,
                'categories' => ['plumbing', 'electrical', 'hvac', 'painting', 'carpentry', 'beauty', 'fitness', 'photography'],
            ],
            'high' => [
                'base_price' => 30.00,
                'price_per_field' => 2.00,
                'price_per_100_chars' => 0.50,
                'min_price' => 20.00,
                'max_price' => 100.00,
                'categories' => ['construction', 'renovation', 'legal', 'accounting', 'architect', 'engineering', 'wedding'],
            ],
        ];

        $created = 0;
        $skipped = 0;

        foreach ($categories as $category) {
            // Check if pricing rule already exists
            if (LeadPricingRule::where('category_id', $category->id)->exists()) {
                $this->command->warn("Pricing rule already exists for category: {$category->name}");
                $skipped++;
                continue;
            }

            // Determine tier based on category name (case-insensitive matching)
            $tier = 'medium'; // Default to medium
            $categoryNameLower = strtolower($category->name);

            foreach ($pricingTiers as $tierName => $tierData) {
                foreach ($tierData['categories'] as $keyword) {
                    if (str_contains($categoryNameLower, $keyword)) {
                        $tier = $tierName;
                        break 2;
                    }
                }
            }

            // Create pricing rule
            LeadPricingRule::create([
                'category_id' => $category->id,
                'base_price' => $pricingTiers[$tier]['base_price'],
                'price_per_field' => $pricingTiers[$tier]['price_per_field'],
                'price_per_100_chars' => $pricingTiers[$tier]['price_per_100_chars'],
                'min_price' => $pricingTiers[$tier]['min_price'],
                'max_price' => $pricingTiers[$tier]['max_price'],
                'active' => true,
                'created_by_admin_id' => null, // System-generated
            ]);

            $this->command->info("Created {$tier} tier pricing for: {$category->name}");
            $created++;
        }

        $this->command->info("✅ Seeding complete! Created: {$created}, Skipped: {$skipped}");
    }
}

