<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Modules\Leads\app\Models\UserFormInput;
use Modules\Leads\app\Models\LeadUnlock;
use App\Services\MarketplaceNotificationService;
use App\Services\MarketplaceWalletService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

/**
 * CleanupExpiredMarketplaceLeads
 * 
 * Command to automatically clean up expired marketplace leads.
 * This command does NOT affect existing traditional leads.
 */
class CleanupExpiredMarketplaceLeads extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'marketplace:cleanup-expired-leads 
                            {--dry-run : Show what would be cleaned up without making changes}
                            {--days=30 : Number of days after expiration to wait before cleanup}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean up expired marketplace leads and refund providers if applicable';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        $daysAfterExpiration = (int) $this->option('days');
        
        $this->info("Starting marketplace lead cleanup...");
        $this->info("Dry run: " . ($dryRun ? 'Yes' : 'No'));
        $this->info("Cleanup delay: {$daysAfterExpiration} days after expiration");

        try {
            // Find expired marketplace leads that are eligible for cleanup
            $cleanupDate = now()->subDays($daysAfterExpiration);
            
            $expiredLeads = UserFormInput::where('lead_source', 'marketplace')
                ->whereNotNull('expires_at')
                ->where('expires_at', '<', $cleanupDate)
                ->where('marketplace_status', '!=', 'cancelled')
                ->with(['unlocks.provider'])
                ->get();

            if ($expiredLeads->isEmpty()) {
                $this->info('No expired marketplace leads found for cleanup.');
                return 0;
            }

            $this->info("Found {$expiredLeads->count()} expired marketplace leads for cleanup.");

            $totalRefunds = 0;
            $totalLeadsProcessed = 0;

            foreach ($expiredLeads as $lead) {
                $this->processExpiredLead($lead, $dryRun, $totalRefunds, $totalLeadsProcessed);
            }

            if (!$dryRun) {
                $this->info("Cleanup completed successfully!");
                $this->info("Total leads processed: {$totalLeadsProcessed}");
                $this->info("Total refunds processed: {$totalRefunds}");
                
                Log::info('Marketplace lead cleanup completed', [
                    'leads_processed' => $totalLeadsProcessed,
                    'refunds_processed' => $totalRefunds,
                    'cleanup_date' => $cleanupDate->toDateString()
                ]);
            } else {
                $this->info("Dry run completed. {$totalLeadsProcessed} leads would be processed.");
                $this->info("{$totalRefunds} refunds would be processed.");
            }

            return 0;

        } catch (\Exception $e) {
            $this->error('Error during cleanup: ' . $e->getMessage());
            Log::error('Marketplace lead cleanup failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return 1;
        }
    }

    /**
     * Process a single expired lead
     *
     * @param UserFormInput $lead
     * @param bool $dryRun
     * @param int &$totalRefunds
     * @param int &$totalLeadsProcessed
     * @return void
     */
    private function processExpiredLead(UserFormInput $lead, bool $dryRun, int &$totalRefunds, int &$totalLeadsProcessed)
    {
        $this->line("Processing lead #{$lead->id} (expired: {$lead->expires_at->toDateString()})");

        if (!$dryRun) {
            DB::beginTransaction();
            try {
                // Get providers who unlocked this lead
                $unlocks = $lead->unlocks;
                $providerIds = $unlocks->pluck('provider_id')->toArray();

                // Send notifications to providers about cancellation
                if (!empty($providerIds)) {
                    $notificationService = new MarketplaceNotificationService();
                    $notificationService->notifyRequestCancelled(
                        $lead->id,
                        $providerIds,
                        'Lead expired and has been automatically cancelled'
                    );
                }

                // Refund providers who unlocked this lead
                $walletService = new MarketplaceWalletService();
                foreach ($unlocks as $unlock) {
                    if ($unlock->credits_spent > 0) {
                        $refundSuccess = $walletService->refundProviderWallet(
                            $unlock->provider_id,
                            $unlock->credits_spent,
                            $lead->id,
                            "Refund for expired marketplace lead #{$lead->id}"
                        );

                        if ($refundSuccess) {
                            $totalRefunds++;
                            $this->line("  - Refunded {$unlock->credits_spent} credits to provider #{$unlock->provider_id}");
                        } else {
                            $this->warn("  - Failed to refund provider #{$unlock->provider_id}");
                        }
                    }
                }

                // Update lead status
                $lead->update([
                    'marketplace_status' => 'expired',
                    'is_unlocked' => false
                ]);

                // Update unlock records
                $unlocks->each(function ($unlock) {
                    $unlock->update([
                        'connection_status' => 'expired'
                    ]);
                });

                DB::commit();
                $totalLeadsProcessed++;

            } catch (\Exception $e) {
                DB::rollBack();
                $this->error("  - Failed to process lead #{$lead->id}: " . $e->getMessage());
                Log::error('Failed to process expired lead', [
                    'lead_id' => $lead->id,
                    'error' => $e->getMessage()
                ]);
            }
        } else {
            // Dry run - just show what would happen
            $unlocks = $lead->unlocks;
            $refundAmount = $unlocks->sum('credits_spent');
            
            $this->line("  - Would update status to 'expired'");
            $this->line("  - Would refund {$unlocks->count()} providers");
            $this->line("  - Total refund amount: {$refundAmount} credits");
            
            $totalRefunds += $unlocks->count();
            $totalLeadsProcessed++;
        }
    }
}
