<?php

namespace App\Helpers;

use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\GlobalSetting\app\Models\InvoiceTemplate;
use Modules\Leads\app\Models\UserFormInput;
use App\Models\UserDetail;
use App\Models\User;
use App\Models\Bookings;
use Modules\GlobalSetting\app\Models\Templates;
use Modules\Communication\app\Http\Controllers\EmailController;
use Dompdf\Dompdf;
use Dompdf\Options;
use App\Models\PayoutHistory;
use Modules\Categories\app\Models\Categories;
use Modules\Product\app\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;


class InvoiceHelper
{
    public static function generateInvoice($providerFormInputId, $amount, $type, $providerid)
    {
        // Validate input parameters
        if (empty($providerFormInputId) || !is_numeric($providerFormInputId)) {
            throw new \InvalidArgumentException('Invalid provider form input ID');
        }
        
        if (!is_numeric($amount) || $amount < 0) {
            throw new \InvalidArgumentException('Invalid amount value');
        }
        
        if (!in_array($type, [1, 2, 3, 4, 5])) {
            throw new \InvalidArgumentException('Invalid invoice type. Must be 1 (Leads), 2 (Booking), 3 (Payout), 4 (Wallet), or 5 (Paradigm)');
        }
        
        if (empty($providerid) || !is_numeric($providerid)) {
            throw new \InvalidArgumentException('Invalid provider ID');
        }
        
        if($type == 1){
            $providerFormInput = UserFormInput::find($providerFormInputId);
            $defaultInvoiceTemplate = InvoiceTemplate::where('invoice_type', 'Leads')->first();
            $category = $providerFormInput ? Categories::where('id', $providerFormInput->category_id)->value('name') : null;
        } else if($type == 2){
            $providerFormInput = Bookings::find($providerFormInputId);
            $defaultInvoiceTemplate = InvoiceTemplate::where('invoice_type', 'Booking')->first();
            $category = $providerFormInput ? Product::where('id', $providerFormInput->product_id)->value('source_name') : null;
            $ratedetails = $providerFormInput ? Product::where('id', $providerFormInput->product_id)->select('source_price', 'duration')->first() : null;
        }
        else if($type == 3){
            $providerFormInput = PayoutHistory::find($providerFormInputId);
            $defaultInvoiceTemplate = InvoiceTemplate::where('invoice_type', 'Payout')->first();
        }
        else if($type == 4){
            // Wallet Top-up
            $providerFormInput = \App\Models\WalletHistory::find($providerFormInputId);
            $defaultInvoiceTemplate = InvoiceTemplate::where('invoice_type', 'Wallet Top-up')->first();
        }
        else if($type == 5){
            // Paradigm Purchase
            $providerFormInput = \App\Models\ParadigmPayment::find($providerFormInputId);
            $defaultInvoiceTemplate = InvoiceTemplate::where('invoice_type', 'Paradigm Purchase')->first();
        }
        
        if (!$providerFormInput) {
            throw new \Exception('Provider form input not found.');
        }
        if(!$defaultInvoiceTemplate){
            $defaultInvoiceTemplate = InvoiceTemplate::where('is_default', true)->first();
        }

        if (!$defaultInvoiceTemplate) {
            throw new \Exception('Invoice template not found.');
        }

        // Fetch global settings values or use default values
        $companyName = GlobalSetting::where('key', 'company_name')->value('value') ?? 'ISH';
        $companyAddress = GlobalSetting::where('key', 'site_address')->value('value') ?? '589 5th Ave, NY 10024, USA';
        $companyEmail = GlobalSetting::where('key', 'site_email')->value('value') ?? 'contact@ish.com';
        $companyPhone = GlobalSetting::where('key', 'phone_no')->value('value') ?? '(123) 456-7890';
        $privderImageStatus = GlobalSetting::where('key', 'providerlogo')->value('value');
        $companyImage = GlobalSetting::where('key', 'invoice_company_logo')->value('value');
        $invoicePrefix = GlobalSetting::where('key', 'invoice_prefix')->value('value') ?? '';
        $invoiceStarts = GlobalSetting::where('key', 'invoice_starts')->value('value') ?? '';
        $companyWebsite = GlobalSetting::where('key', 'website')->value('value') ?? '';
        $contact = $companyEmail . ' | ' . $companyPhone;
        $invoicePrefix = $invoicePrefix . $invoiceStarts;
        $orderId = str_pad($providerFormInputId ?? 0, 4, '0', STR_PAD_LEFT);
        $invoiceId = $invoicePrefix . $orderId;

        // Fetch provider details
        $providerDetails = UserDetail::where('user_id', $providerid)->first();
        $providerInfo = User::where('id', $providerid)->first();

        // Check if provider-specific values are enabled and available
        if ($privderImageStatus == 1 && $providerDetails && $providerInfo) {
            $companyName = $providerDetails->company_name ?? $companyName;
            $companyAddress = $providerDetails->company_address ?? $companyAddress;
            $companyEmail = $providerInfo->email ?? $companyEmail;
            $companyPhone = $providerInfo->phone_number ?? $companyPhone;

            // Set provider image URL if available
            if (!empty($providerDetails->company_image)) {
                $imageUrl = public_path('storage/company-image/' . $providerDetails->company_image);
            } else {
                $imageUrl = public_path('/front/img/logo.svg');
            }
        } elseif ($companyImage) {
            $imageUrl = public_path('storage/' . $companyImage);
        } else {
            $imageUrl = public_path('/front/img/logo.svg');
        }

        // Override image URL for a specific type
        if ($type == 3 && $companyImage) {
            $imageUrl = public_path('storage/' . $companyImage);
            $companyName = GlobalSetting::where('key', 'company_name')->value('value') ?? 'ISH';
            $companyAddress = GlobalSetting::where('key', 'site_address')->value('value') ?? '589 5th Ave, NY 10024, USA';
            $companyEmail = GlobalSetting::where('key', 'site_email')->value('value') ?? 'contact@ish.com';
            $companyPhone = GlobalSetting::where('key', 'phone_no')->value('value') ?? '(123) 456-7890';
        }

        $imageData = base64_encode(file_get_contents($imageUrl));
        $imageSrc = 'data:image;base64,' . $imageData;

        $userDetails = UserDetail::select(
            'first_name',
            'last_name',
            'address',
            'postal_code'
        )->where('user_id', $providerFormInput->user_id)
          ->first();

        $userName = $userDetails->first_name . ' ' . $userDetails->last_name ?? 'Demo User';
        $userAddress = $userDetails->address ?? 'No Address';
        $userCity = $userDetails->city ?? 'No City';
        $userState = $userDetails->state ?? 'No State';
        $userPostal = $userDetails->postal_code ?? 'No Postal Code';
        $user_email = User::select('email','phone_number')->where('id',$providerFormInput->user_id)->first();

        switch ($type) {
            case 1:
                $description = 'Leads';
                break;
            case 2:
                $description = 'Booking';
                break;
            case 3:
                $description = 'Payout';
                break;
            case 4:
                $description = 'Wallet Top-up';
                break;
            case 5:
                $description = 'Paradigm Assessment Purchase';
                break;
            default:
                $description = 'Unknown';
                break;
        }

        $service_amount = $amount - ($providerFormInput->amount_tax ?? 0);
        
        // Additional variables for Type 4 & 5
        $transactionId = $providerFormInput->transaction_id ?? '-';
        $paymentMethod = $providerFormInput->payment_type ?? 'Wallet';
        $paymentId = ($type == 5 && isset($providerFormInput->id)) ? $providerFormInput->id : '-';
        $templateName = ($type == 5 && isset($providerFormInput->template)) ? $providerFormInput->template->name : '-';
        $clientName = ($type == 5 && isset($providerFormInput->client_name)) ? $providerFormInput->client_name : '-';
        $accessToken = ($type == 5 && $providerFormInput->accessTokens->isNotEmpty()) ? $providerFormInput->accessTokens->first()->token : '-';
        
        $invoiceContent = str_replace(
            ['{{company_name}}', '{{company_address}}', '{{company_email}}', '{{company_phone}}',
             '{{user_name}}', '{{user_address}}', '{{user_city}}', '{{user_state}}', '{{user_postal}}',
             '{{user_email}}', '{{user_phone}}', '{{quote}}', '{{description}}', '{{logo}}','{{category}}',
             '{{service_amount}}', '{{amount_tax}}' , '{{id}}' , '{{date}}' ,'{{ratehour}}', '{{first_name}}', '{{last_name}}', '{{website_link}}', '{{contact}}',
             '{{amount}}', '{{transaction_id}}', '{{payment_method}}', '{{payment_id}}', '{{template_name}}', '{{client_name}}', '{{access_token}}'],
            [
                $companyName,
                $companyAddress,
                $companyEmail,
                $companyPhone,
                $userName,
                $userAddress,
                $userCity,
                $userState,
                $userPostal,
                $user_email->email,
                $user_email->phone_number,
                $amount,
                $description,
                $imageSrc,
                $category ?? '-',
                $service_amount,
                $providerFormInput->amount_tax ?? '-',
                $invoiceId ?? '-',
                $providerFormInput->created_at ?? '-',
                ($ratedetails->duration ?? null) ? ($ratedetails->source_price ?? '-') . '/' . ($ratedetails->duration ?? '- ') : '-',
                $userDetails->first_name ?? '', 
                $userDetails->last_name ?? '',
                $companyWebsite,
                $contact,
                $amount,
                $transactionId,
                $paymentMethod,
                $paymentId,
                $templateName,
                $clientName,
                $accessToken
            ],
            $defaultInvoiceTemplate->template_content
        );

        $dompdf = new Dompdf();
        $dompdf->loadHtml($invoiceContent);

        $options = new Options();
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isPhpEnabled', true);
        $dompdf->setOptions($options);

        $dompdf->render();

        $pdfPath = 'invoices/invoice_' . $invoiceId . '.pdf';

        if (!\Storage::disk('public')->exists('invoices')) {
            \Storage::disk('public')->makeDirectory('invoices');
        }

        \Storage::disk('public')->put($pdfPath, $dompdf->output());

        $pdfUrl = \Storage::disk('public')->url($pdfPath);

        if($type == 1){
            $notificationType = 30;
            $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->first();
            $subject = $template->subject;
            $content = $template->content;
            $user_email = User::select('email','name')->where('id',$providerFormInput->user_id)->first();
            $content = str_replace('{{user_name}}', $user_email->name, $content);
            $content = str_replace('{{company_name}}',  $companyName, $content);
        }else if($type == 2){
            $notificationType = 28;
            $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->first();
            $subject = $template->subject;
            $content = $template->content;
            $user_email = User::select('email','name')->where('id',$providerFormInput->user_id)->first();
            $content = str_replace('{{user_name}}', $user_email->name, $content);
            $content = str_replace('{{company_name}}',  $companyName, $content);
        }
        else if($type == 3){
            $notificationType = 29;
            $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->first();
            $subject = $template->subject;
            $content = $template->content;
            $user_email = User::select('email','name')->where('id',$providerFormInput->user_id)->first();
            $content = str_replace('{{user_name}}', $user_email->name, $content);
            $content = str_replace('{{company_name}}',  $companyName, $content);
        }
        else if($type == 4){
            // Wallet Top-up Invoice - Use Template (notification_type = 7)
            $notificationType = 7; // Wallet Top-up Success
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', $notificationType)
                ->where('status', 1)
                ->first();
            
            if ($template) {
                $subject = $template->subject;
                $content = $template->content;
            } else {
                // Fallback if template not found
                $subject = 'Wallet Top-up Invoice - ' . $invoiceId;
                $content = '<p>Dear {{user_name}},</p>
                            <p>Thank you for topping up your wallet with {{company_name}}.</p>
                            <p>Amount: $' . $amount . '</p>
                            <p>Your invoice is attached to this email.</p>
                            <p>Best regards,<br>{{company_name}}</p>';
            }
            $user_email = User::select('email','name')->where('id',$providerFormInput->user_id)->first();
            $content = str_replace('{{user_name}}', $user_email->name, $content);
            $content = str_replace('{{company_name}}',  $companyName, $content);
            $content = str_replace('{{amount}}',  '$' . $amount, $content);
            $content = str_replace('{{invoice_id}}',  $invoiceId, $content);
        }
        else if($type == 5){
            // Paradigm Purchase Invoice - Use Template (notification_type = 13)
            $notificationType = 13; // Self-Purchase Confirmation
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', $notificationType)
                ->where('status', 1)
                ->first();
            
            if ($template) {
                $subject = $template->subject;
                $content = $template->content;
            } else {
                // Fallback if template not found
                $subject = 'Paradigm Assessment Invoice - ' . $invoiceId;
                $content = '<p>Dear {{user_name}},</p>
                            <p>Thank you for purchasing a Paradigm Assessment with {{company_name}}.</p>
                            <p>Amount: $' . $amount . '</p>
                            <p>Your invoice is attached to this email.</p>
                            <p>Best regards,<br>{{company_name}}</p>';
            }
            $user_email = User::select('email','name')->where('id',$providerFormInput->user_id)->first();
            $content = str_replace('{{user_name}}', $user_email->name, $content);
            $content = str_replace('{{company_name}}',  $companyName, $content);
            $content = str_replace('{{amount}}',  '$' . $amount, $content);
            $content = str_replace('{{payment_id}}',  $invoiceId, $content);
            $content = str_replace('{{template_name}}',  $templateName ?? 'Paradigm Assessment', $content);
        }
        $emailData = [
            'to_email' => $user_email->email,
            'subject' => $subject,
            'content' => $content,
            'attachment' => public_path('storage/' . $pdfPath),
        ];

        $emailRequest = new Request($emailData);
        $emailController = new EmailController();
        $emailController->sendEmail($emailRequest);

        return $pdfPath;
    }
}
