<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Branches;
use App\Models\BranchDocument;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Modules\GlobalSetting\app\Models\Templates;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Communication\app\Http\Controllers\EmailController;
use Illuminate\Support\Facades\Log;

class BranchVerificationController extends Controller
{
    public function index()
    {
        return view('admin.branch-verification');
    }

    public function list(Request $request)
    {
        if ($request->ajax()) {
            $branches = Branches::with(['creator', 'documents'])
                ->where(function($query) {
                    $query->whereIn('verification_status', ['pending', 'approved', 'rejected'])
                          ->orWhereNull('verification_status');
                })
                ->orderBy('created_at', 'desc')
                ->get();

            $data = $branches->map(function($branch, $index) {
                $badges = [
                    'pending' => '<span class="badge badge-warning">Pending</span>',
                    'approved' => '<span class="badge badge-success">Approved</span>',
                    'rejected' => '<span class="badge badge-danger">Rejected</span>',
                ];
                
                $status = $branch->verification_status ?? 'approved'; // Default to approved if NULL
                
                return [
                    'DT_RowIndex' => $index + 1,
                    'branch_name' => $branch->branch_name,
                    'provider_name' => $branch->creator ? $branch->creator->name : 'N/A',
                    'documents_count' => $branch->documents->count(),
                    'status_badge' => $badges[$status] ?? '<span class="badge badge-secondary">Unknown</span>',
                    'created_at' => $branch->created_at->format('Y-m-d H:i:s'),
                    'action' => '<button class="btn btn-sm btn-primary view-branch-btn" data-id="'.$branch->id.'">View & Verify</button>',
                ];
            });

            return response()->json([
                'draw' => (int) $request->input('draw', 1),
                'recordsTotal' => $branches->count(),
                'recordsFiltered' => $branches->count(),
                'data' => $data
            ]);
        }
        
        return response()->json(['error' => 'Invalid request'], 400);
    }

    public function verify(Request $request)
    {
        $request->validate([
            'branch_id' => 'required|exists:branches,id',
            'action' => 'required|in:approve,reject',
            'rejection_reason' => 'required_if:action,reject|nullable|string',
        ]);

        $branch = Branches::findOrFail($request->branch_id);
        
        if ($request->action === 'approve') {
            $branch->update([
                'verification_status' => 'approved',
                'verified_by' => Auth::id(),
                'verified_at' => now(),
                'rejection_reason' => null,
            ]);
            
            // Send branch verified email (Type 56)
            $this->sendBranchVerifiedEmail($branch);
            
            return response()->json([
                'success' => true,
                'message' => 'Branch approved successfully.'
            ]);
        } else {
            $branch->update([
                'verification_status' => 'rejected',
                'verified_by' => Auth::id(),
                'verified_at' => now(),
                'rejection_reason' => $request->rejection_reason,
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Branch rejected successfully.'
            ]);
        }
    }

    public function details(Request $request, $id)
    {
        try {
            $branch = Branches::with(['creator', 'documents'])->findOrFail($id);
            
            // Get country, state, city names
            $countryName = 'N/A';
            $stateName = 'N/A';
            $cityName = 'N/A';
            
            try {
                if ($branch->branch_country) {
                    $countriesPath = public_path('countries.json');
                    if (file_exists($countriesPath)) {
                        $countriesJson = file_get_contents($countriesPath);
                        $countriesData = json_decode($countriesJson, true);
                        $countries = $countriesData['countries'] ?? [];
                        $country = collect($countries)->firstWhere('id', (int)$branch->branch_country);
                        $countryName = $country['name'] ?? 'N/A';
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Failed to load country name: ' . $e->getMessage());
            }
            
            try {
                if ($branch->branch_state) {
                    $statesPath = public_path('states.json');
                    if (file_exists($statesPath)) {
                        $statesJson = file_get_contents($statesPath);
                        $statesData = json_decode($statesJson, true);
                        $states = $statesData['states'] ?? [];
                        $state = collect($states)->firstWhere('id', (int)$branch->branch_state);
                        $stateName = $state['name'] ?? 'N/A';
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Failed to load state name: ' . $e->getMessage());
            }
            
            try {
                if ($branch->branch_city) {
                    $citiesPath = public_path('cities.json');
                    if (file_exists($citiesPath)) {
                        $citiesJson = file_get_contents($citiesPath);
                        $citiesData = json_decode($citiesJson, true);
                        $cities = $citiesData['cities'] ?? [];
                        $city = collect($cities)->firstWhere('id', (int)$branch->branch_city);
                        $cityName = $city['name'] ?? 'N/A';
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Failed to load city name: ' . $e->getMessage());
            }
            
            return response()->json([
                'branch' => [
                    'id' => $branch->id,
                    'branch_name' => $branch->branch_name,
                    'branch_email' => $branch->branch_email,
                    'branch_mobile' => $branch->branch_mobile,
                    'branch_address' => $branch->branch_address,
                    'branch_zip' => $branch->branch_zip, // Correct column name
                    'start_hour' => $branch->branch_startworkhour, // Map to expected name
                    'end_hour' => $branch->branch_endworkhour, // Map to expected name
                    'branch_workingday' => $branch->branch_workingday,
                    'branch_holiday' => $branch->branch_holiday,
                    'creator' => $branch->creator,
                ],
                'country_name' => $countryName,
                'state_name' => $stateName,
                'city_name' => $cityName,
                'branch_image_url' => $branch->branch_image ? asset('storage/branch/' . $branch->branch_image) : null,
                'working_days' => $branch->branch_workingday ? explode(',', $branch->branch_workingday) : [],
                'holidays' => $branch->branch_holiday ? explode(',', $branch->branch_holiday) : [],
                'documents' => $branch->documents->map(function($doc) {
                    return [
                        'id' => $doc->id,
                        'type' => $doc->document_type,
                        'name' => $doc->file_name,
                        'url' => asset('storage/' . $doc->file_path),
                    ];
                }),
            ]);
        } catch (\Exception $e) {
            Log::error('Branch verification details error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load branch details: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send branch verified email to provider (Type 56)
     */
    private function sendBranchVerifiedEmail(Branches $branch)
    {
        try {
            $template = Templates::where('notification_type', 56)
                ->where('type', 2) // Email type
                ->where('status', 1)
                ->first();

            if (!$template) {
                Log::warning('Branch verified email template (Type 56) not found');
                return;
            }

            $provider = User::find($branch->created_by);
            if (!$provider) {
                Log::warning('Provider not found for branch', ['branch_id' => $branch->id]);
                return;
            }

            $companyName = GlobalSetting::where('key', 'company_name')->value('value') ?? 'I Seek Help International';

            $replacements = [
                '{{branch_name}}' => $branch->branch_name,
                '{{provider_name}}' => $provider->name,
                '{{branch_address}}' => $branch->branch_address ?? 'N/A',
                '{{company_name}}' => $companyName,
                '{{current_year}}' => date('Y'),
            ];

            $subject = str_replace(array_keys($replacements), array_values($replacements), $template->subject);
            $content = str_replace(array_keys($replacements), array_values($replacements), $template->content);

            $data = [
                'to_email' => $provider->email,
                'subject' => $subject,
                'content' => $content
            ];

            $emailRequest = new Request($data);
            $emailController = new EmailController();
            $emailController->sendEmail($emailRequest);
        } catch (\Exception $e) {
            Log::error('Failed to send branch verified email: ' . $e->getMessage());
        }
    }
}

