<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class EditMediaController extends Controller
{
    /**
     * Display the custom links management page
     */
    public function customLinks()
    {
        try {
            $data = [
                'title' => __('Edit Media - Custom Links'),
                'heading' => __('Custom Links Management'),
                'breadcrumb' => [
                    ['title' => __('Dashboard'), 'url' => route('admin.dashboard')],
                    ['title' => __('Edit Media'), 'url' => ''],
                    ['title' => __('Custom Links'), 'url' => ''],
                ]
            ];
            
            return view('admin.edit-media.custom-links', $data);
        } catch (\Exception $e) {
            Log::error('Error loading custom links page: ' . $e->getMessage());
            return redirect()->route('admin.dashboard')->with('error', 'Unable to load custom links page');
        }
    }

    /**
     * Get all custom links (DataTable)
     */
    public function list(Request $request)
    {
        try {
            $query = DB::table('edit_media_custom_links')
                ->orderBy('created_at', 'desc');

            // Search filter
            if ($request->has('search') && !empty($request->search['value'])) {
                $search = $request->search['value'];
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('url', 'like', "%{$search}%");
                });
            }

            $totalRecords = DB::table('edit_media_custom_links')->count();
            $filteredRecords = $query->count();

            // Pagination
            if ($request->has('start') && $request->has('length')) {
                $query->skip($request->start)->take($request->length);
            }

            $links = $query->get();

            $data = $links->map(function ($link) {
                return [
                    'id' => $link->id,
                    'name' => $link->name,
                    'url' => $link->url,
                    'active' => $link->active,
                    'created_at' => date('Y-m-d H:i', strtotime($link->created_at)),
                ];
            });

            return response()->json([
                'draw' => intval($request->draw),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching custom links: ' . $e->getMessage());
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Unable to fetch custom links'
            ], 500);
        }
    }

    /**
     * Store a new custom link
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url|max:500',
        ]);

        try {
            $linkId = DB::table('edit_media_custom_links')->insertGetId([
                'name' => $request->name,
                'url' => $request->url,
                'active' => $request->has('active') ? 1 : 0,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Custom link created successfully',
                'data' => ['id' => $linkId]
            ]);
        } catch (\Exception $e) {
            Log::error('Error creating custom link: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to create custom link'
            ], 500);
        }
    }

    /**
     * Update a custom link
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url|max:500',
        ]);

        try {
            DB::table('edit_media_custom_links')
                ->where('id', $id)
                ->update([
                    'name' => $request->name,
                    'url' => $request->url,
                    'active' => $request->has('active') ? 1 : 0,
                    'updated_at' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Custom link updated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating custom link: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to update custom link'
            ], 500);
        }
    }

    /**
     * Delete a custom link
     */
    public function destroy($id)
    {
        try {
            DB::table('edit_media_custom_links')->where('id', $id)->delete();

            return response()->json([
                'success' => true,
                'message' => 'Custom link deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error deleting custom link: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to delete custom link'
            ], 500);
        }
    }

    /**
     * Toggle link status
     */
    public function toggleStatus($id)
    {
        try {
            $link = DB::table('edit_media_custom_links')->where('id', $id)->first();
            if (!$link) {
                return response()->json([
                    'success' => false,
                    'message' => 'Link not found'
                ], 404);
            }

            DB::table('edit_media_custom_links')
                ->where('id', $id)
                ->update([
                    'active' => $link->active ? 0 : 1,
                    'updated_at' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Link status updated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling link status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to update link status'
            ], 500);
        }
    }
}
