<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ParadigmFormTemplate;
use App\Models\ParadigmPersonalField;
use App\Models\ParadigmQuestion;
use App\Models\ParadigmBadgeTitle;
use App\Models\ParadigmPdfPage;
use App\Models\ParadigmPdfElement;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;

class ParadigmFormTemplateController extends Controller
{
    /**
     * Display a listing of Paradigm Profile templates
     */
    public function index(): View
    {
        $templates = ParadigmFormTemplate::with(['personalFields', 'questions', 'badgeTitles', 'pdfPages'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('admin.paradigm.templates.index', compact('templates'));
    }

    /**
     * Get templates as JSON (for API/AJAX requests)
     */
    public function getTemplatesJson(): JsonResponse
    {
        $templates = ParadigmFormTemplate::where('is_active', true)
            ->select('id', 'name', 'description', 'is_active')
            ->orderBy('name', 'asc')
            ->get();

        return response()->json($templates);
    }

    /**
     * Show the form for creating a new template
     */
    public function create(): View
    {
        return view('admin.paradigm.templates.create');
    }

    /**
     * Store a newly created template
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'settings' => 'nullable|array'
        ]);

        $template = ParadigmFormTemplate::create([
            'name' => $request->name,
            'description' => $request->description,
            'is_active' => $request->boolean('is_active', true),
            'settings' => $request->settings ?? []
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Paradigm Profile template created successfully',
            'template' => $template
        ]);
    }

    /**
     * Display the specified template
     */
    public function show(ParadigmFormTemplate $paradigmFormTemplate): View
    {
        $template = $paradigmFormTemplate->load([
            'personalFields',
            'questions' => function($query) {
                $query->orderBy('assessment_module')->orderBy('display_order');
            },
            'badgeTitles',
            'pdfPages.pdfElements'
        ]);

        return view('admin.paradigm.templates.show', compact('template'));
    }

    /**
     * Preview the form
     */
    public function preview(ParadigmFormTemplate $paradigmFormTemplate): View
    {
        $template = $paradigmFormTemplate->load([
            'personalFields' => function($query) {
                $query->orderBy('display_order');
            },
            'questions' => function($query) {
                $query->orderBy('assessment_module')->orderBy('display_order');
            },
            'badgeTitles',
            'pdfPages.pdfElements' => function($query) {
                $query->orderBy('display_order');
            }
        ]);
        
        return view('admin.paradigm.templates.preview', compact('template'));
    }

    /**
     * Generate PDF preview for template
     */
    public function generatePdfPreview(Request $request, $templateId = null)
    {
        try {
            // Get template ID from route parameter or request
            $templateId = $templateId ?? $request->route('paradigmFormTemplate') ?? $request->route('templateId');
            
            \Log::info('Paradigm PDF Preview Generation Request', [
                'template_id' => $templateId,
                'route_params' => $request->route() ? $request->route()->parameters() : [],
                'all_params' => $request->all(),
                'personal_data_raw' => $request->input('personal_data'),
                'scores_raw' => $request->input('scores'),
                'special_raw' => $request->input('special'),
                'result_raw' => $request->input('result')
            ]);
            
            // Find template
            $template = ParadigmFormTemplate::find($templateId);
            if (!$template) {
                \Log::error('Paradigm template not found for PDF preview generation', ['template_id' => $templateId]);
                return response()->json(['error' => 'Template not found'], 404);
            }
            
            // Load template with PDF elements
            $template = $template->load([
                'pdfPages.pdfElements' => function($query) {
                    $query->orderBy('display_order');
                }
            ]);
            
            // Get data from request (sent from JavaScript)
            $personalData = $request->input('personal_data', []);
            $scores = $request->input('scores', []);
            $special = $request->input('special', []);
            $result = $request->input('result', []);
            
            // Use actual form data or fallback to defaults
            $personalData = is_string($personalData) ? json_decode($personalData, true) : $personalData;
            $scores = is_string($scores) ? json_decode($scores, true) : $scores;
            $special = is_string($special) ? json_decode($special, true) : $special;
            $result = is_string($result) ? json_decode($result, true) : $result;
            
            \Log::info('Paradigm PDF Preview Data After Processing', [
                'personal_data' => $personalData,
                'scores' => $scores,
                'special' => $special,
                'result' => $result,
                'personal_data_type' => gettype($personalData),
                'scores_type' => gettype($scores),
                'special_type' => gettype($special),
                'result_type' => gettype($result)
            ]);
            
            // Create assessment result with actual form data (matching backup structure but with individual fields)
            $sampleResult = new \App\Models\ParadigmAssessmentResult([
                'template_id' => $template->id,
                'session_id' => $result['session_id'] ?? 'SES-123456789',
                'name' => $personalData['name'] ?? 'John',
                'surname' => $personalData['surname'] ?? 'Doe',
                'gender' => $personalData['gender'] ?? 'Male',
                'age_range' => $personalData['age'] ?? '25-34',
                'phone' => $personalData['phone'] ?? '555-1234',
                'email' => $personalData['email'] ?? 'john@example.com',
                'scores_json' => $scores, // Already an array from JSON decode
                'paradigm_type' => $special['paradigm_type'] ?? 'The Whole-Brain Thinker',
                'dominance_category_short' => $special['dominance_category_short'] ?? 'All Frame',
                'answers_json' => [], // Empty for preview
                'completed_at' => isset($result['completed_at']) ? \Carbon\Carbon::parse($result['completed_at']) : now()
            ]);
            
            // Set the template relationship so PDF service can access PDF builder elements
            $sampleResult->setRelation('template', $template);
            
            // Generate PDF using the PDF service
            $pdfService = new \App\Services\ParadigmPdfServiceDomPDF();
            $pdfPath = $pdfService->generatePdf($sampleResult);
            
            // Create user-friendly filename
            $userName = trim(($personalData['name'] ?? 'John') . ' ' . ($personalData['surname'] ?? 'Doe'));
            $userName = preg_replace('/[^a-zA-Z0-9\s\-_]/', '', $userName); // Remove special characters
            $userName = preg_replace('/\s+/', '_', $userName); // Replace spaces with underscores
            $fileName = $userName . '_' . ($special['paradigm_type'] ?? 'Whole-Brain-Thinker') . '_' . date('Y-m-d_H-i-s') . '.pdf';
            
            // Save PDF to public storage with user name
            $publicPath = 'paradigm-results/' . $fileName;
            $fullPublicPath = public_path('storage/' . $publicPath);
            
            // Ensure directory exists
            $directory = dirname($fullPublicPath);
            if (!is_dir($directory)) {
                mkdir($directory, 0755, true);
            }
            
            // Copy PDF to public storage
            copy($pdfPath, $fullPublicPath);
            
            // Read the PDF content for download
            $pdfContent = file_get_contents($pdfPath);
            
            return response($pdfContent, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="' . $fileName . '"'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Paradigm PDF Preview Generation Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate PDF preview'], 500);
        }
    }

    /**
     * Show the form for editing the specified template
     */
    public function edit(ParadigmFormTemplate $paradigmFormTemplate): View
    {
        $template = $paradigmFormTemplate->load([
            'personalFields',
            'questions' => function($query) {
                $query->orderBy('assessment_module')->orderBy('display_order');
            },
            'badgeTitles',
            'pdfPages.pdfElements'
        ]);

        return view('admin.paradigm.templates.edit', compact('template'));
    }

    /**
     * Update a specific question
     */
    public function updateQuestion(Request $request, ParadigmQuestion $question): JsonResponse
    {
        $validated = $request->validate([
            'question_text' => 'required|string',
            'option_a_label' => 'required|string',
            'option_b_label' => 'required|string',
            'option_c_label' => 'nullable|string',
            'option_d_label' => 'nullable|string',
            'a_side' => 'required|string',
            'b_side' => 'required|string',
            'c_side' => 'nullable|string',
            'd_side' => 'nullable|string',
        ]);

        try {
            $question->update($validated);
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            \Log::error('Error updating question: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Server error while updating question.'], 500);
        }
    }

    /**
     * Update the specified template
     */
    public function update(Request $request, ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'settings' => 'nullable|array',
            'personal_fields' => 'nullable|array',
            'questions' => 'nullable|array',
            'badge_titles' => 'nullable|array'
        ]);

        // Update template basic info
        $paradigmFormTemplate->update([
            'name' => $request->name,
            'description' => $request->description,
            'is_active' => $request->boolean('is_active', true),
            'settings' => $request->settings ?? $paradigmFormTemplate->settings
        ]);

        // Update personal fields
        if ($request->has('personal_fields')) {
            foreach ($request->personal_fields as $fieldId => $fieldData) {
                $field = ParadigmPersonalField::find($fieldId);
                if ($field && $field->template_id == $paradigmFormTemplate->id) {
                    // Process options for select fields
                    $options = null;
                    if ($fieldData['field_type'] === 'select' && !empty($fieldData['options'])) {
                        $options = array_filter(array_map('trim', explode("\n", $fieldData['options'])));
                    }
                    
                    $field->update([
                        'field_label' => $fieldData['field_label'],
                        'field_type' => $fieldData['field_type'],
                        'is_required' => $fieldData['is_required'],
                        'display_order' => $fieldData['display_order'],
                        'options' => $options
                    ]);
                }
            }
        }

        // Update questions
        if ($request->has('questions')) {
            foreach ($request->questions as $questionId => $questionData) {
                $question = ParadigmQuestion::find($questionId);
                if ($question && $question->template_id == $paradigmFormTemplate->id) {
                    $question->update([
                        'question_text' => $questionData['question_text'],
                        'option_a_label' => $questionData['option_a_label'],
                        'option_b_label' => $questionData['option_b_label'],
                        'option_c_label' => $questionData['option_c_label'] ?? null,
                        'option_d_label' => $questionData['option_d_label'] ?? null,
                        'option_e_label' => $questionData['option_e_label'] ?? null,
                        'assessment_module' => $questionData['assessment_module'],
                        'question_type' => $questionData['question_type'],
                        'rating_scale_min' => $questionData['rating_scale_min'] ?? null,
                        'rating_scale_max' => $questionData['rating_scale_max'] ?? null,
                        'is_required' => $questionData['is_required'],
                        'display_order' => $questionData['display_order']
                    ]);
                }
            }
        }

        // Update badge titles
        if ($request->has('badge_titles')) {
            foreach ($request->badge_titles as $badgeId => $badgeData) {
                $badge = ParadigmBadgeTitle::find($badgeId);
                if ($badge && $badge->template_id == $paradigmFormTemplate->id) {
                    $badge->update([
                        'badge_title' => $badgeData['badge_title'],
                        'description' => $badgeData['description'],
                        'category' => $badgeData['category'] ?? null,
                        'preference_level' => $badgeData['preference_level'] ?? null
                    ]);
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Paradigm Profile template updated successfully',
            'template' => $paradigmFormTemplate
        ]);
    }

    /**
     * Remove the specified template
     */
    public function destroy(ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $paradigmFormTemplate->delete();

        return response()->json([
            'success' => true,
            'message' => 'Paradigm Profile template deleted successfully'
        ]);
    }

    /**
     * Toggle template active status
     */
    public function toggleStatus(ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $paradigmFormTemplate->update([
            'is_active' => !$paradigmFormTemplate->is_active
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Template status updated successfully',
            'is_active' => $paradigmFormTemplate->is_active
        ]);
    }

    /**
     * Get template statistics
     */
    public function getStats(ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $stats = [
            'personal_fields_count' => $paradigmFormTemplate->personalFields()->count(),
            'questions_count' => $paradigmFormTemplate->questions()->count(),
            'questions_by_module' => [
                'balance_profile' => $paradigmFormTemplate->getQuestionsByModule('balance_profile')->count(),
                'conflict_styles' => $paradigmFormTemplate->getQuestionsByModule('conflict_styles')->count(),
                'connection_styles' => $paradigmFormTemplate->getQuestionsByModule('connection_styles')->count(),
                'personality_blueprint' => $paradigmFormTemplate->getQuestionsByModule('personality_blueprint')->count(),
                'serq' => $paradigmFormTemplate->getQuestionsByModule('serq')->count(),
                'brain_frames' => $paradigmFormTemplate->getQuestionsByModule('brain_frames')->count(),
                'brain_fields' => $paradigmFormTemplate->getQuestionsByModule('brain_fields')->count(),
                'type_classification' => $paradigmFormTemplate->getQuestionsByModule('type_classification')->count(),
            ],
            'badge_titles_count' => $paradigmFormTemplate->badgeTitles()->count(),
            'pdf_pages_count' => $paradigmFormTemplate->pdfPages()->count(),
            'assessment_results_count' => $paradigmFormTemplate->assessmentResults()->count(),
        ];

        return response()->json([
            'success' => true,
            'stats' => $stats
        ]);
    }

    /**
     * Duplicate template
     */
    public function duplicate(ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $newTemplate = $paradigmFormTemplate->replicate();
        $newTemplate->name = $paradigmFormTemplate->name . ' (Copy)';
        $newTemplate->save();

        // Duplicate related data
        $this->duplicateTemplateData($paradigmFormTemplate, $newTemplate);

        return response()->json([
            'success' => true,
            'message' => 'Template duplicated successfully',
            'template' => $newTemplate
        ]);
    }

    /**
     * Add personal field to template
     */
    public function addPersonalField(Request $request, ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $request->validate([
            'field_key' => 'required|string|max:255',
            'field_label' => 'required|string|max:255',
            'field_type' => 'required|in:text,email,select,textarea,number,date,tel,radio,checkbox,time,datetime-local',
            'is_required' => 'boolean',
            'display_order' => 'integer|min:0',
            'options' => 'nullable|string'
        ]);

        $options = null;
        if (in_array($request->field_type, ['select', 'radio', 'checkbox']) && !empty($request->options)) {
            $options = array_filter(array_map('trim', explode("\n", $request->options)));
        }

        $field = ParadigmPersonalField::create([
            'template_id' => $paradigmFormTemplate->id,
            'field_key' => $request->field_key,
            'field_label' => $request->field_label,
            'field_type' => $request->field_type,
            'is_required' => $request->boolean('is_required', false),
            'display_order' => $request->display_order ?? 0,
            'options' => $options
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Personal field added successfully',
            'field' => $field
        ]);
    }

    /**
     * Add question to template
     */
    public function addQuestion(Request $request, ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $request->validate([
            'question_text' => 'required|string',
            'option_a_label' => 'required|string|max:255',
            'option_b_label' => 'required|string|max:255',
            'option_c_label' => 'nullable|string|max:255',
            'option_d_label' => 'nullable|string|max:255',
            'option_e_label' => 'nullable|string|max:255',
            'assessment_module' => 'required|in:balance_profile,conflict_styles,connection_styles,personality_blueprint,serq,brain_frames,brain_fields,type_classification',
            'question_type' => 'required|in:rating_scale,multiple_choice,ranking,binary_choice',
            'rating_scale_min' => 'nullable|integer|min:1',
            'rating_scale_max' => 'nullable|integer|min:1',
            'is_required' => 'boolean',
            'display_order' => 'integer|min:0'
        ]);

        $question = ParadigmQuestion::create([
            'template_id' => $paradigmFormTemplate->id,
            'question_text' => $request->question_text,
            'option_a_label' => $request->option_a_label,
            'option_b_label' => $request->option_b_label,
            'option_c_label' => $request->option_c_label,
            'option_d_label' => $request->option_d_label,
            'option_e_label' => $request->option_e_label,
            'assessment_module' => $request->assessment_module,
            'question_type' => $request->question_type,
            'rating_scale_min' => $request->rating_scale_min,
            'rating_scale_max' => $request->rating_scale_max,
            'is_required' => $request->boolean('is_required', true),
            'display_order' => $request->display_order ?? 0
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Question added successfully',
            'question' => $question
        ]);
    }

    /**
     * Add badge title to template
     */
    public function addBadgeTitle(Request $request, ParadigmFormTemplate $paradigmFormTemplate): JsonResponse
    {
        $request->validate([
            'type_code' => 'required|string|max:255',
            'badge_title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'nullable|string|max:255',
            'preference_level' => 'nullable|string|max:255'
        ]);

        $badge = ParadigmBadgeTitle::create([
            'template_id' => $paradigmFormTemplate->id,
            'type_code' => $request->type_code,
            'badge_title' => $request->badge_title,
            'description' => $request->description,
            'category' => $request->category,
            'preference_level' => $request->preference_level
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Badge title added successfully',
            'badge' => $badge
        ]);
    }

    /**
     * Duplicate template related data
     */
    private function duplicateTemplateData(ParadigmFormTemplate $original, ParadigmFormTemplate $new): void
    {
        // Duplicate personal fields
        foreach ($original->personalFields as $field) {
            $field->replicate()->fill(['template_id' => $new->id])->save();
        }

        // Duplicate questions
        foreach ($original->questions as $question) {
            $question->replicate()->fill(['template_id' => $new->id])->save();
        }

        // Duplicate badge titles
        foreach ($original->badgeTitles as $badge) {
            $badge->replicate()->fill(['template_id' => $new->id])->save();
        }

        // Duplicate PDF pages and elements
        foreach ($original->pdfPages as $page) {
            $newPage = $page->replicate()->fill(['template_id' => $new->id])->save();
            
            foreach ($page->pdfElements as $element) {
                $element->replicate()->fill([
                    'template_id' => $new->id,
                    'page_id' => $newPage->id
                ])->save();
            }
        }
    }
    
    /**
     * Get page elements
     */
    public function getPageElements(int $pageId): JsonResponse
    {
        $page = \App\Models\ParadigmPdfPage::with('pdfElements')->findOrFail($pageId);
        
        return response()->json([
            'success' => true,
            'elements' => $page->pdfElements->map(function($element) {
                return [
                    'id' => $element->id,
                    'element_type' => $element->element_type,
                    'element_key' => $element->element_key,
                    'text_template' => $element->text_template,
                    'x_pct' => $element->x_pct,
                    'y_pct' => $element->y_pct,
                    'w_pct' => $element->w_pct,
                    'font_px' => $element->font_px,
                    'font_weight' => $element->font_weight,
                    'text_align' => $element->text_align,
                    'color' => $element->color,
                    'bg_color' => $element->bg_color,
                    'fg_color' => $element->fg_color,
                ];
            })
        ]);
    }
    
    /**
     * Create new element
     */
    public function createElement(Request $request): JsonResponse
    {
        $request->validate([
            'page_id' => 'required|exists:paradigm_pdf_pages,id',
            'element_type' => 'required|in:text,number,pill,image',
            'element_key' => 'required|string|max:50',
            'x_pct' => 'required|numeric|min:0|max:100',
            'y_pct' => 'required|numeric|min:0|max:100',
            'w_pct' => 'required|numeric|min:1|max:100',
            'font_px' => 'required|integer|min:8|max:72',
            'font_weight' => 'required|integer|min:100|max:900',
            'text_align' => 'required|in:left,center,right',
            'color' => 'required|string|max:7',
            'bg_color' => 'nullable|string|max:7',
            'fg_color' => 'nullable|string|max:7',
            'text_template' => 'nullable|string',
        ]);
        
        $element = \App\Models\ParadigmPdfElement::create($request->all());
        
        return response()->json([
            'success' => true,
            'element' => $element
        ]);
    }
    
    /**
     * Update element
     */
    public function updateElement(Request $request, \App\Models\ParadigmPdfElement $element): JsonResponse
    {
        $request->validate([
            'element_type' => 'sometimes|in:text,number,pill,image',
            'element_key' => 'sometimes|string|max:50',
            'x_pct' => 'sometimes|numeric|min:0|max:100',
            'y_pct' => 'sometimes|numeric|min:0|max:100',
            'w_pct' => 'sometimes|numeric|min:1|max:100',
            'font_px' => 'sometimes|integer|min:8|max:72',
            'font_weight' => 'sometimes|integer|min:100|max:900',
            'text_align' => 'sometimes|in:left,center,right',
            'color' => 'sometimes|string|max:7',
            'bg_color' => 'nullable|string|max:7',
            'fg_color' => 'nullable|string|max:7',
            'text_template' => 'nullable|string',
        ]);
        
        $element->update($request->all());
        
        return response()->json([
            'success' => true,
            'element' => $element
        ]);
    }
    
    /**
     * Delete element
     */
    public function deleteElement(\App\Models\ParadigmPdfElement $element): JsonResponse
    {
        $element->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Element deleted successfully'
        ]);
    }
    
    /**
     * Create new PDF page
     */
    public function createPage(Request $request): JsonResponse
    {
        // Get template ID from route parameter
        $templateId = $request->route('paradigmFormTemplate');
        $paradigmFormTemplate = ParadigmFormTemplate::find($templateId);
        
        if (!$paradigmFormTemplate) {
            return response()->json([
                'success' => false,
                'message' => 'Template not found'
            ], 404);
        }
        
        // Debug: Log the incoming request data
        \Log::info('Create Paradigm Page Request Data:', $request->all());
        \Log::info('Template ID:', $paradigmFormTemplate->id);
        
        try {
            $request->validate([
                'page_title' => 'required|string|max:255',
                'page_number' => 'required|integer|min:1',
                'is_content_page' => 'boolean'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Paradigm Validation failed:', $e->errors());
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }
        
        // Check if page number already exists
        $existingPage = $paradigmFormTemplate->pdfPages()->where('page_number', $request->page_number)->first();
        if ($existingPage) {
            return response()->json([
                'success' => false,
                'message' => 'Page number already exists'
            ], 422);
        }
        
        $maxOrder = $paradigmFormTemplate->pdfPages()->max('display_order') ?? 0;
        
        $page = $paradigmFormTemplate->pdfPages()->create([
            'page_title' => $request->page_title,
            'page_number' => $request->page_number,
            'is_content_page' => $request->boolean('is_content_page', true),
            'display_order' => $maxOrder + 1
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'PDF page created successfully',
            'page' => $page
        ]);
    }
    
    /**
     * Update PDF page
     */
    public function updatePage(Request $request, \App\Models\ParadigmPdfPage $page): JsonResponse
    {
        $request->validate([
            'page_title' => 'sometimes|string|max:255',
            'page_number' => 'sometimes|integer|min:1',
            'is_content_page' => 'sometimes|boolean',
            'background_image_path' => 'sometimes|nullable|string'
        ]);
        
        // Check if page number already exists (excluding current page)
        if ($request->has('page_number')) {
            $existingPage = $page->template->pdfPages()
                ->where('page_number', $request->page_number)
                ->where('id', '!=', $page->id)
                ->first();
            if ($existingPage) {
                return response()->json([
                    'success' => false,
                    'message' => 'Page number already exists'
                ], 422);
            }
        }
        
        $page->update($request->only([
            'page_title', 'page_number', 'is_content_page', 'background_image_path'
        ]));
        
        return response()->json([
            'success' => true,
            'message' => 'PDF page updated successfully',
            'page' => $page
        ]);
    }
    
    /**
     * Delete PDF page
     */
    public function deletePage(\App\Models\ParadigmPdfPage $page): JsonResponse
    {
        // Delete all elements first
        $page->pdfElements()->delete();
        
        // Delete the page
        $page->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'PDF page deleted successfully'
        ]);
    }
}
