<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ParadigmFormTemplate;
use App\Models\ParadigmPricing;
use App\Models\ParadigmPayment;
use App\Models\CreditPackage;
use App\Models\ParadigmMarketplacePricingRule;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Validator;

class ParadigmPricingController extends Controller
{
    /**
     * Display pricing management page
     */
    public function index(): View
    {
        try {
            $templates = ParadigmFormTemplate::with('pricing')->get();
            
            // Get the Paradigm Profile package for user self-purchase
            $package = CreditPackage::where('is_paradigm_package', true)->first();
            
            // Get ONLY self-purchased Paradigm Payment transactions from PayFast
            // Must have payment_method = 'payfast' OR be linked to PayFast wallet transaction
            $transactions = ParadigmPayment::with(['template', 'client.userDetail', 'walletTransaction'])
                ->where('self_purchased', 1)
                ->whereNotNull('self_purchased')
                ->whereIn('status', ['paid', 'free'])
                ->where(function($query) {
                    $query->where('payment_method', 'payfast')
                          ->orWhereHas('walletTransaction', function($q) {
                              $q->where('payment_type', 'payfast')
                                ->where('status', 'Completed');
                          });
                })
                ->orderBy('created_at', 'desc')
                ->paginate(20);
            
            // Get marketplace pricing rules
            $marketplaceRules = ParadigmMarketplacePricingRule::orderBy('sort_order')->orderBy('id')->get();
            
            return view('admin.paradigm.pricing.index', compact('templates', 'package', 'transactions', 'marketplaceRules'));
        } catch (\Exception $e) {
            // If there's an error, return empty collection
            $templates = collect();
            $package = null;
            $transactions = collect();
            $marketplaceRules = collect();
            return view('admin.paradigm.pricing.index', compact('templates', 'package', 'transactions', 'marketplaceRules'));
        }
    }

    /**
     * Create or update pricing for a template
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'template_id' => 'required|exists:paradigm_form_templates,id',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|max:3',
            'description' => 'nullable|string|max:255',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Use updateOrCreate to avoid constraint violations
            // This will update existing record or create new one if doesn't exist
            $pricing = ParadigmPricing::updateOrCreate(
                ['template_id' => $request->template_id], // Find by template_id
                [
                    'price' => $request->price,
                    'currency' => $request->currency,
                    'description' => $request->description,
                    'is_active' => $request->boolean('is_active', true)
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Pricing updated successfully',
                'data' => $pricing
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update pricing: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get pricing for a specific template
     */
    public function show(int $templateId): JsonResponse
    {
        try {
            $pricing = ParadigmPricing::where('template_id', $templateId)
                ->where('is_active', true)
                ->first();

            if (!$pricing) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active pricing found for this template'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $pricing
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve pricing: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle pricing status
     */
    public function toggleStatus(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'template_id' => 'required|exists:paradigm_form_templates,id',
            'is_active' => 'required|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $pricing = ParadigmPricing::where('template_id', $request->template_id)
                ->where('is_active', true)
                ->first();

            if (!$pricing) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active pricing found for this template'
                ], 404);
            }

            $pricing->update(['is_active' => $request->boolean('is_active')]);

            return response()->json([
                'success' => true,
                'message' => 'Pricing status updated successfully',
                'data' => $pricing
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update pricing status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get marketplace pricing rules (not used - rules are loaded in index)
     */
    public function marketplacePricing(): View
    {
        // This method is not used - marketplace pricing is shown in the index tab
        return redirect()->route('admin.paradigm.pricing.index');
    }

    /**
     * Store marketplace pricing rule
     */
    public function storeMarketplacePricing(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'rule_name' => 'required|string|max:255',
            'paradigm_type' => 'nullable|string|max:10',
            'min_score' => 'nullable|numeric|min:0',
            'max_score' => 'nullable|numeric|min:0|gte:min_score',
            'price_credits' => 'required|numeric|min:0.01',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $rule = ParadigmMarketplacePricingRule::create([
                'rule_name' => $request->rule_name,
                'paradigm_type' => $request->paradigm_type,
                'min_score' => $request->min_score,
                'max_score' => $request->max_score,
                'price_credits' => $request->price_credits,
                'is_active' => $request->boolean('is_active', true),
                'sort_order' => $request->sort_order ?? 0,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pricing rule created successfully',
                'data' => $rule
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create pricing rule: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete marketplace pricing rule
     */
    public function deleteMarketplacePricing($ruleId): JsonResponse
    {
        try {
            $rule = ParadigmMarketplacePricingRule::findOrFail($ruleId);
            $rule->delete();

            return response()->json([
                'success' => true,
                'message' => 'Pricing rule deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete pricing rule: ' . $e->getMessage()
            ], 500);
        }
    }
}