<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ParadigmAssessmentResult;
use App\Models\ParadigmFormTemplate;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ParadigmResultsController extends Controller
{
    /**
     * Display a listing of assessment results
     */
    public function index(Request $request): View
    {
        // Fetch completed assessment results with relationships
        // Order by created_at descending (newest first)
        $results = ParadigmAssessmentResult::with(['provider', 'template'])
            ->whereNotNull('paradigm_type') // Only show completed assessments
            ->orderBy('created_at', 'desc')
            ->get();
        
        return view('admin.paradigm.results.index', compact('results'));
    }
    
    /**
     * Display the specified assessment result
     */
    public function show(ParadigmAssessmentResult $result): View
    {
        $result->load('template');
        return view('admin.paradigm.results.show', compact('result'));
    }
    
    /**
     * Delete a single assessment result
     */
    public function delete(Request $request)
    {
        $request->validate([
            'result_id' => 'required|exists:paradigm_assessment_results,id'
        ]);

        try {
            $result = ParadigmAssessmentResult::findOrFail($request->result_id);
            $clientName = $result->name ?? 'Unknown';
            
            // Delete the result (cascade will handle related data)
            $result->delete();
            
            // Log the deletion
            \Log::info('Admin deleted paradigm assessment result', [
                'result_id' => $request->result_id,
                'client_name' => $clientName,
                'admin_id' => auth()->id(),
                'deleted_at' => now()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => __('Assessment result deleted successfully')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to delete paradigm assessment result', [
                'result_id' => $request->result_id,
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to delete assessment result')
            ], 500);
        }
    }
    
    /**
     * Delete all assessment results
     */
    public function deleteAll(Request $request)
    {
        try {
            $count = ParadigmAssessmentResult::count();
            
            if ($count === 0) {
                return response()->json([
                    'success' => false,
                    'message' => __('No assessment results to delete')
                ]);
            }
            
            // Log before deletion
            \Log::info('Admin deleting all paradigm assessment results', [
                'count' => $count,
                'admin_id' => auth()->id(),
                'deleted_at' => now()
            ]);
            
            // Delete all results (handle foreign key constraints)
            \DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            ParadigmAssessmentResult::truncate();
            \DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            
            return response()->json([
                'success' => true,
                'message' => __('All assessment results deleted successfully')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to delete all paradigm assessment results', [
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to delete all assessment results')
            ], 500);
        }
    }
    
    /**
     * Export results to CSV
     */
    public function export(Request $request)
    {
        $query = ParadigmAssessmentResult::with('template');
        
        // Apply same filters as index
        if ($request->filled('template_id')) {
            $query->where('template_id', $request->template_id);
        }
        
        if ($request->filled('date_from')) {
            $query->whereDate('completed_at', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('completed_at', '<=', $request->date_to);
        }
        
        $results = $query->orderBy('completed_at', 'desc')->get();
        
        $filename = 'paradigm-results-' . date('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($results) {
            $file = fopen('php://output', 'w');
            
            // CSV headers for all 74 Paradigm Profile elements
            fputcsv($file, [
                'ID',
                'Name',
                'Surname',
                'Gender',
                'Age Range',
                'Email',
                'Phone',
                'Template',
                'Paradigm Type',
                'Dominance Category',
                // Balance Profile (9 elements)
                'Emotional Balance %',
                'Spiritual Balance %',
                'Physical Balance %',
                'Social Balance %',
                'Financial Balance %',
                'Occupational Balance %',
                'Intellectual Balance %',
                'Environmental Balance %',
                'Overall Wellness Score',
                // Conflict Styles (15 elements)
                'Conflict Assertive Challenger Score',
                'Conflict Integrative Bridge Builder Score',
                'Conflict Fair Negotiator Score',
                'Conflict Quiet Protector Score',
                'Conflict Peacekeeper Score',
                'Conflict Assertive Challenger Rank',
                'Conflict Integrative Bridge Builder Rank',
                'Conflict Fair Negotiator Rank',
                'Conflict Quiet Protector Rank',
                'Conflict Peacekeeper Rank',
                'Conflict Rank 1 Name',
                'Conflict Rank 2 Name',
                'Conflict Rank 3 Name',
                'Conflict Rank 4 Name',
                'Conflict Rank 5 Name',
                // Connection Styles (15 elements)
                'Connection Verbal Validator Score',
                'Connection Presence Seeker Score',
                'Connection Symbolic Appreciator Score',
                'Connection Practical Supporter Score',
                'Connection Grounded Connector Score',
                'Connection Verbal Validator Rank',
                'Connection Presence Seeker Rank',
                'Connection Symbolic Appreciator Rank',
                'Connection Practical Supporter Rank',
                'Connection Grounded Connector Rank',
                'Connection Rank 1 Name',
                'Connection Rank 2 Name',
                'Connection Rank 3 Name',
                'Connection Rank 4 Name',
                'Connection Rank 5 Name',
                // Personality Blueprint (2 elements)
                'Extroversion %',
                'Introversion %',
                // SERQ (3 elements)
                'SQ Level',
                'EQ Level',
                'RQ Level',
                // Brain Profile - FRAMES (4 elements)
                'LU %',
                'LL %',
                'RU %',
                'RL %',
                'LU Preference Level',
                'LL Preference Level',
                'RU Preference Level',
                'RL Preference Level',
                // Brain Profile - FIELDS (8 elements)
                'Fact Finder %',
                'Question Asker %',
                'Rule Keeper %',
                'Task Manager %',
                'Creator %',
                'Big Thinker %',
                'Empath %',
                'Friend Maker %',
                // Conditional Booleans (4 elements)
                'Is Single Frame',
                'Is Dual Frame',
                'Is Triple Frame',
                'Is All Frame',
                'Completed At'
            ]);
            
            // CSV data
            foreach ($results as $result) {
                $scores = $result->getScores();
                fputcsv($file, [
                    $result->id,
                    $result->name,
                    $result->surname,
                    $result->gender,
                    $result->age_range,
                    $result->email,
                    $result->phone,
                    $result->template->name,
                    $result->paradigm_type,
                    $result->dominance_category_short,
                    // Balance Profile
                    $scores['emotional_balance_percentage'] ?? 0,
                    $scores['spiritual_balance_percentage'] ?? 0,
                    $scores['physical_balance_percentage'] ?? 0,
                    $scores['social_balance_percentage'] ?? 0,
                    $scores['financial_balance_percentage'] ?? 0,
                    $scores['occupational_balance_percentage'] ?? 0,
                    $scores['intellectual_balance_percentage'] ?? 0,
                    $scores['environmental_balance_percentage'] ?? 0,
                    $scores['overall_wellness_score'] ?? 0,
                    // Conflict Styles
                    $scores['conflict_style_assertive_challenger_score'] ?? 0,
                    $scores['conflict_style_integrative_bridge_builder_score'] ?? 0,
                    $scores['conflict_style_fair_negotiator_score'] ?? 0,
                    $scores['conflict_style_quiet_protector_score'] ?? 0,
                    $scores['conflict_style_peacekeeper_score'] ?? 0,
                    $scores['conflict_style_assertive_challenger_rank'] ?? 0,
                    $scores['conflict_style_integrative_bridge_builder_rank'] ?? 0,
                    $scores['conflict_style_fair_negotiator_rank'] ?? 0,
                    $scores['conflict_style_quiet_protector_rank'] ?? 0,
                    $scores['conflict_style_peacekeeper_rank'] ?? 0,
                    $scores['conflict_style_rank_1_name'] ?? '',
                    $scores['conflict_style_rank_2_name'] ?? '',
                    $scores['conflict_style_rank_3_name'] ?? '',
                    $scores['conflict_style_rank_4_name'] ?? '',
                    $scores['conflict_style_rank_5_name'] ?? '',
                    // Connection Styles
                    $scores['connection_style_verbal_validator_score'] ?? 0,
                    $scores['connection_style_presence_seeker_score'] ?? 0,
                    $scores['connection_style_symbolic_appreciator_score'] ?? 0,
                    $scores['connection_style_practical_supporter_score'] ?? 0,
                    $scores['connection_style_grounded_connector_score'] ?? 0,
                    $scores['connection_style_verbal_validator_rank'] ?? 0,
                    $scores['connection_style_presence_seeker_rank'] ?? 0,
                    $scores['connection_style_symbolic_appreciator_rank'] ?? 0,
                    $scores['connection_style_practical_supporter_rank'] ?? 0,
                    $scores['connection_style_grounded_connector_rank'] ?? 0,
                    $scores['connection_style_rank_1_name'] ?? '',
                    $scores['connection_style_rank_2_name'] ?? '',
                    $scores['connection_style_rank_3_name'] ?? '',
                    $scores['connection_style_rank_4_name'] ?? '',
                    $scores['connection_style_rank_5_name'] ?? '',
                    // Personality Blueprint
                    $scores['extroversion_percentage'] ?? 0,
                    $scores['introversion_percentage'] ?? 0,
                    // SERQ
                    $scores['sq_level'] ?? 'Low',
                    $scores['eq_level'] ?? 'Low',
                    $scores['rq_level'] ?? 'Low',
                    // Brain Profile - FRAMES
                    $scores['lu_percentage'] ?? 0,
                    $scores['ll_percentage'] ?? 0,
                    $scores['ru_percentage'] ?? 0,
                    $scores['rl_percentage'] ?? 0,
                    $scores['lu_preference_level'] ?? 'Low',
                    $scores['ll_preference_level'] ?? 'Low',
                    $scores['ru_preference_level'] ?? 'Low',
                    $scores['rl_preference_level'] ?? 'Low',
                    // Brain Profile - FIELDS
                    $scores['fact_finder_percentage'] ?? 0,
                    $scores['question_asker_percentage'] ?? 0,
                    $scores['rule_keeper_percentage'] ?? 0,
                    $scores['task_manager_percentage'] ?? 0,
                    $scores['creator_percentage'] ?? 0,
                    $scores['big_thinker_percentage'] ?? 0,
                    $scores['empath_percentage'] ?? 0,
                    $scores['friend_maker_percentage'] ?? 0,
                    // Conditional Booleans
                    $scores['is_single_frame'] ?? false,
                    $scores['is_dual_frame'] ?? false,
                    $scores['is_triple_frame'] ?? false,
                    $scores['is_all_frame'] ?? false,
                    $result->completed_at?->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    /**
     * Download a specific PDF file
     */
    public function downloadPdfFile(Request $request, $filename)
    {
        // Set memory limit to 2GB for PDF generation with high-quality images
        ini_set('memory_limit', '2048M');
        
        // SECURITY FIX: Validate filename to prevent path traversal
        // Only allow alphanumeric, dash, underscore, and dot
        if (!preg_match('/^[a-zA-Z0-9_\-\.]+$/', $filename)) {
            abort(400, 'Invalid filename format');
        }
        
        // Ensure it ends with .pdf
        if (!str_ends_with(strtolower($filename), '.pdf')) {
            abort(400, 'Invalid file type - must be PDF');
        }
        
        // Use basename to prevent directory traversal
        $safeFilename = basename($filename);
        $filePath = public_path('storage/paradigm-results/' . $safeFilename);
        
        // Verify the resolved path is still in the allowed directory
        $realPath = realpath($filePath);
        $allowedDir = realpath(public_path('storage/paradigm-results'));
        
        if (!$realPath || !str_starts_with($realPath, $allowedDir . DIRECTORY_SEPARATOR)) {
            abort(403, 'Unauthorized file access attempt detected');
        }
        
        if (!file_exists($filePath)) {
            return redirect()->route('admin.paradigm.results.index')
                ->with('error', 'PDF file not found.');
        }
        
        return response()->download($filePath, $safeFilename);
    }
    
    /**
     * Delete a PDF file
     */
    public function deletePdfFile(Request $request)
    {
        try {
            $filename = $request->input('filename');
            $filePath = public_path('storage/paradigm-results/' . $filename);
            
            if (!file_exists($filePath)) {
                return redirect()->route('admin.paradigm.results.index')
                    ->with('error', 'PDF file not found.');
            }
            
            unlink($filePath);
            
            return redirect()->route('admin.paradigm.results.index')
                ->with('success', "PDF file '{$filename}' has been deleted successfully.");
                
        } catch (\Exception $e) {
            return redirect()->route('admin.paradigm.results.index')
                ->with('error', 'Failed to delete PDF file: ' . $e->getMessage());
        }
    }
    
    /**
     * Format file size in human readable format
     */
    private function formatFileSize($bytes)
    {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }
    
    /**
     * Get results statistics
     */
    public function getStats(Request $request)
    {
        $query = ParadigmAssessmentResult::query();
        
        // Apply filters
        if ($request->filled('template_id')) {
            $query->where('template_id', $request->template_id);
        }
        
        if ($request->filled('date_from')) {
            $query->whereDate('completed_at', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('completed_at', '<=', $request->date_to);
        }
        
        $totalResults = $query->count();
        $recentResults = $query->where('completed_at', '>=', now()->subDays(7))->count();
        
        // Get paradigm type distribution
        $typeDistribution = $query->selectRaw('paradigm_type, COUNT(*) as count')
            ->groupBy('paradigm_type')
            ->orderBy('count', 'desc')
            ->get();
        
        // Get dominance category distribution
        $dominanceDistribution = $query->selectRaw('dominance_category_short, COUNT(*) as count')
            ->groupBy('dominance_category_short')
            ->orderBy('count', 'desc')
            ->get();
        
        // Get average scores for key elements
        $avgScores = [];
        $keyElements = [
            'emotional_balance_percentage', 'spiritual_balance_percentage', 'physical_balance_percentage',
            'social_balance_percentage', 'financial_balance_percentage', 'occupational_balance_percentage',
            'intellectual_balance_percentage', 'environmental_balance_percentage', 'overall_wellness_score',
            'lu_percentage', 'll_percentage', 'ru_percentage', 'rl_percentage',
            'extroversion_percentage', 'introversion_percentage'
        ];
        
        foreach ($keyElements as $element) {
            $avgScores[$element] = $query->selectRaw("AVG(JSON_EXTRACT(scores_json, '$.{$element}')) as avg_score")
                ->value('avg_score') ?? 0;
        }
        
        return response()->json([
            'success' => true,
            'stats' => [
                'total_results' => $totalResults,
                'recent_results' => $recentResults,
                'type_distribution' => $typeDistribution,
                'dominance_distribution' => $dominanceDistribution,
                'average_scores' => $avgScores
            ]
        ]);
    }
}
