<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\StaffDocument;
use Illuminate\Support\Facades\Auth;
use Modules\GlobalSetting\app\Models\Templates;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Communication\app\Http\Controllers\EmailController;
use Illuminate\Support\Facades\Log;

class StaffVerificationController extends Controller
{
    public function index()
    {
        return view('admin.staff-verification');
    }

    public function list(Request $request)
    {
        if ($request->ajax()) {
            $staff = User::with(['staffDocuments', 'category'])
                ->where('user_type', 4) // Staff only
                ->where(function($query) {
                    $query->whereIn('staff_verification_status', ['pending', 'approved', 'rejected'])
                          ->orWhereNull('staff_verification_status');
                })
                ->orderBy('created_at', 'desc')
                ->get();

            $data = $staff->map(function($staffMember, $index) {
                $badges = [
                    'pending' => '<span class="badge badge-warning">Pending</span>',
                    'approved' => '<span class="badge badge-success">Approved</span>',
                    'rejected' => '<span class="badge badge-danger">Rejected</span>',
                ];
                
                $status = $staffMember->staff_verification_status ?? 'pending'; // Default to pending if NULL
                
                return [
                    'DT_RowIndex' => $index + 1,
                    'name' => $staffMember->name,
                    'email' => $staffMember->email,
                    'category_name' => $staffMember->category ? $staffMember->category->name : 'N/A',
                    'documents_count' => $staffMember->staffDocuments->count(),
                    'status_badge' => $badges[$status] ?? '<span class="badge badge-secondary">Unknown</span>',
                    'created_at' => $staffMember->created_at->format('Y-m-d H:i:s'),
                    'action' => '<button class="btn btn-sm btn-primary view-staff-btn" data-id="'.$staffMember->id.'">View & Verify</button>',
                ];
            });

            return response()->json([
                'draw' => (int) $request->input('draw', 1),
                'recordsTotal' => $staff->count(),
                'recordsFiltered' => $staff->count(),
                'data' => $data
            ]);
        }
        
        return response()->json(['error' => 'Invalid request'], 400);
    }

    public function verify(Request $request)
    {
        $request->validate([
            'staff_id' => 'required|exists:users,id',
            'action' => 'required|in:approve,reject',
            'rejection_reason' => 'required_if:action,reject|nullable|string',
        ]);

        $staff = User::findOrFail($request->staff_id);
        
        if ($request->action === 'approve') {
            $staff->update([
                'staff_verification_status' => 'approved',
                'staff_verified_by' => Auth::id(),
                'staff_verified_at' => now(),
                'staff_rejection_reason' => null,
            ]);
            
            // Send staff verified email (Type 55)
            $this->sendStaffVerifiedEmail($staff);
            
            return response()->json([
                'success' => true,
                'message' => 'Staff approved successfully.'
            ]);
        } else {
            $staff->update([
                'staff_verification_status' => 'rejected',
                'staff_verified_by' => Auth::id(),
                'staff_verified_at' => now(),
                'staff_rejection_reason' => $request->rejection_reason,
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Staff rejected successfully.'
            ]);
        }
    }

    public function details(Request $request, $id)
    {
        $staff = User::with(['staffDocuments', 'category', 'userDetail'])->findOrFail($id);
        
        Log::info("Fetching staff details for ID: {$id}");
        Log::info("Staff documents count: " . $staff->staffDocuments->count());
        
        // Get location names from JSON files
        $cityName = 'N/A';
        $stateName = 'N/A';
        $countryName = 'N/A';
        
        if ($staff->userDetail) {
            if ($staff->userDetail->country) {
                $countriesPath = public_path('countries.json');
                $countries = json_decode(file_get_contents($countriesPath), true);
                $countryIndex = array_search($staff->userDetail->country, array_column($countries['countries'], 'id'));
                if ($countryIndex !== false) {
                    $countryName = $countries['countries'][$countryIndex]['name'];
                }
            }
            
            if ($staff->userDetail->state) {
                $statesPath = public_path('states.json');
                $states = json_decode(file_get_contents($statesPath), true);
                $stateIndex = array_search($staff->userDetail->state, array_column($states['states'], 'id'));
                if ($stateIndex !== false) {
                    $stateName = $states['states'][$stateIndex]['name'];
                }
            }
            
            if ($staff->userDetail->city) {
                $citiesPath = public_path('cities.json');
                $cities = json_decode(file_get_contents($citiesPath), true);
                $cityIndex = array_search($staff->userDetail->city, array_column($cities['cities'], 'id'));
                if ($cityIndex !== false) {
                    $cityName = $cities['cities'][$cityIndex]['name'];
                }
            }
        }
        
        return response()->json([
            'staff' => [
                'id' => $staff->id,
                'name' => $staff->name,
                'email' => $staff->email,
                'phone' => $staff->phone_number,
                'address' => $staff->userDetail->address ?? 'N/A',
                'city' => $cityName,
                'state' => $stateName,
                'country' => $countryName,
                'postal_code' => $staff->userDetail->postal_code ?? 'N/A',
                'bio' => $staff->userDetail->bio ?? 'N/A',
                'category' => $staff->category ? $staff->category->name : 'N/A',
                'status' => $staff->staff_verification_status ?? 'pending',
                'verification_status' => $staff->staff_verification_status ?? 'pending',
                'created_at' => $staff->created_at->format('Y-m-d H:i:s'),
            ],
            'documents' => $staff->staffDocuments->map(function($doc) {
                return [
                    'id' => $doc->id,
                    'type' => $doc->document_type,
                    'name' => $doc->file_name,
                    'url' => asset('storage/' . $doc->file_path),
                    'is_image' => in_array(strtolower(pathinfo($doc->file_name, PATHINFO_EXTENSION)), ['jpg', 'jpeg', 'png', 'gif', 'webp']),
                ];
            }),
        ]);
    }

    /**
     * Send staff verified email (Type 55)
     */
    private function sendStaffVerifiedEmail(User $staff)
    {
        try {
            $template = Templates::where('notification_type', 55)
                ->where('type', 2) // Email type
                ->where('status', 1)
                ->first();

            if (!$template) {
                Log::warning('Staff verified email template (Type 55) not found');
                return;
            }

            $provider = User::find($staff->parent_id ?? Auth::id());
            $companyName = GlobalSetting::where('key', 'company_name')->value('value') ?? 'I Seek Help International';

            $replacements = [
                '{{staff_name}}' => $staff->name,
                '{{company_name}}' => $companyName,
                '{{current_year}}' => date('Y'),
            ];

            $subject = str_replace(array_keys($replacements), array_values($replacements), $template->subject);
            $content = str_replace(array_keys($replacements), array_values($replacements), $template->content);

            $data = [
                'to_email' => $staff->email,
                'subject' => $subject,
                'content' => $content
            ];

            $emailRequest = new Request($data);
            $emailController = new EmailController();
            $emailController->sendEmail($emailRequest);
        } catch (\Exception $e) {
            Log::error('Failed to send staff verified email: ' . $e->getMessage());
        }
    }
}

