<?php

namespace App\Http\Controllers;

use App\Models\BranchDocument;
use App\Models\Branches;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class BranchDocumentController extends Controller
{
    /**
     * Upload branch documents (4 separate file inputs)
     */
    public function uploadDocuments(Request $request): JsonResponse
    {
        // Validate that at least the 2 mandatory documents are present
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'document_1' => 'required|file|mimes:jpg,jpeg,png,gif,pdf,application/pdf|max:5120', // Company Registration - Mandatory
            'document_2' => 'required|file|mimes:jpg,jpeg,png,gif,pdf,application/pdf|max:5120', // Director ID - Mandatory
            'document_3' => 'nullable|file|mimes:jpg,jpeg,png,gif,pdf,application/pdf|max:5120', // Proof of Address - Optional
            'document_4' => 'nullable|file|mimes:jpg,jpeg,png,gif,pdf,application/pdf|max:5120', // Additional Document - Optional
        ], [
            'document_1.required' => 'Company Registration Document is required.',
            'document_1.mimes' => 'Company Registration must be JPG, PNG, GIF, or PDF.',
            'document_1.max' => 'Company Registration must not exceed 5MB.',
            'document_2.required' => 'Director ID Document is required.',
            'document_2.mimes' => 'Director ID must be JPG, PNG, GIF, or PDF.',
            'document_2.max' => 'Director ID must not exceed 5MB.',
            'document_3.mimes' => 'Proof of Address must be JPG, PNG, GIF, or PDF.',
            'document_3.max' => 'Proof of Address must not exceed 5MB.',
            'document_4.mimes' => 'Additional Document must be JPG, PNG, GIF, or PDF.',
            'document_4.max' => 'Additional Document must not exceed 5MB.',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        try {
            $branchId = $request->branch_id;
            $uploadedFiles = [];
            
            // Document type mapping
            $documentTypes = [
                'document_1' => 'company_registration',
                'document_2' => 'director_id',
                'document_3' => 'proof_of_address',
                'document_4' => 'additional_document',
            ];
            
            // Process each document
            for ($i = 1; $i <= 4; $i++) {
                $fieldName = "document_{$i}";
                
                if ($request->hasFile($fieldName)) {
                    $file = $request->file($fieldName);
                    
                    // Store the file
                    $filename = Str::uuid() . '_' . time() . '_' . rand(1000, 9999) . '.' . $file->getClientOriginalExtension();
                    $path = $file->storeAs('branch-documents', $filename, 'public');
                    
                    // Create document record
                    $document = BranchDocument::create([
                        'branch_id' => $branchId,
                        'document_type' => $documentTypes[$fieldName],
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $path,
                        'file_size' => $file->getSize(),
                        'uploaded_by' => Auth::id(),
                    ]);
                    
                    $uploadedFiles[] = [
                        'id' => $document->id,
                        'type' => $documentTypes[$fieldName],
                        'filename' => $filename,
                        'url' => asset('storage/' . $path)
                    ];
                }
            }
            
            // Update branch verification status to pending
            Branches::where('id', $branchId)->update(['verification_status' => 'pending']);
            
            return response()->json([
                'success' => true,
                'message' => count($uploadedFiles) . ' document(s) uploaded successfully',
                'data' => [
                    'files' => $uploadedFiles,
                    'count' => count($uploadedFiles)
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload documents: ' . $e->getMessage()
            ], 500);
        }
    }
}

