<?php

namespace App\Http\Controllers;

use App\Services\PaddleService;
use App\Services\CreditConversionService;
use App\Repositories\Contracts\WalletInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PaddleWalletController extends Controller
{
    protected $paddleService;
    protected $walletRepository;

    public function __construct(PaddleService $paddleService, WalletInterface $walletRepository)
    {
        $this->paddleService = $paddleService;
        $this->walletRepository = $walletRepository;
    }

    /**
     * Create Paddle checkout for wallet top-up
     */
    public function createCheckout(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:50',
            'userId' => 'required|integer'
        ]);

        try {
            // Check if Paddle is enabled
            if (!$this->paddleService->isEnabled()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Paddle payment is not configured. Please contact administrator.'
                ], 400);
            }

            // Create wallet record first (same pattern as existing system)
            $walletHistory = $this->walletRepository->addWalletAmount([
                'user_id' => $request->userId,
                'amount' => $request->amount,
                'credits' => $request->credits ?? $request->amount,
                'payment_method' => 'paddle',
                'status' => 'Pending',
                'type' => 1, // Credit
            ]);

            // Create Paddle checkout
            $checkout = $this->paddleService->createWalletCheckout(
                $request->amount,
                $request->userId
            );

            if ($checkout['success']) {
                return response()->json([
                    'success' => true,
                    'checkout_url' => $checkout['checkout_url'],
                    'transaction_id' => $checkout['transaction_id'],
                    'wallet_record_id' => $walletHistory->id,
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => $checkout['error'] ?? 'Failed to create Paddle checkout'
            ], 400);

        } catch (\Exception $e) {
            Log::error('Paddle wallet checkout error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error processing wallet top-up request'
            ], 500);
        }
    }

    /**
     * Handle successful wallet payment
     */
    public function success(Request $request)
    {
        try {
            $transactionId = $request->get('transaction_id');
            
            if (!$transactionId) {
                return redirect()->route('provider.wallet-topup')
                    ->with('error', 'Invalid payment response');
            }

            // Get transaction details from Paddle
            $transaction = $this->paddleService->getTransaction($transactionId);
            
            if ($transaction['success']) {
                $data = $transaction['data']['data'];
                
                // Update wallet record status
                $this->walletRepository->confirmTransaction($transactionId, 'Completed');
                
                return view('provider.wallet-payment-success', [
                    'amount' => $data['details']['totals']['total'],
                    'currency' => $data['currency_code'],
                    'transaction_id' => $transactionId,
                ]);
            }

            return redirect()->route('provider.wallet-topup')
                ->with('error', 'Payment verification failed');

        } catch (\Exception $e) {
            Log::error('Paddle wallet success error: ' . $e->getMessage());
            return redirect()->route('provider.wallet-topup')
                ->with('error', 'Error processing payment');
        }
    }

    /**
     * Handle cancelled wallet payment
     */
    public function cancel(Request $request)
    {
        return redirect()->route('provider.wallet-topup')
            ->with('error', 'Payment was cancelled');
    }

    /**
     * Handle Paddle webhook for wallet payments
     */
    public function webhook(Request $request)
    {
        try {
            $payload = $request->getContent();
            $signature = $request->header('Paddle-Signature');
            
            // Verify webhook signature
            if (!$this->paddleService->verifyWebhook($payload, $signature)) {
                Log::warning('Invalid Paddle webhook signature');
                return response()->json(['error' => 'Invalid signature'], 400);
            }

            $data = json_decode($payload, true);
            $eventType = $data['event_type'] ?? '';

            Log::info('Paddle wallet webhook received', [
                'event_type' => $eventType,
                'data' => $data
            ]);

            switch ($eventType) {
                case 'transaction.completed':
                    $this->handleTransactionCompleted($data);
                    break;
                    
                case 'transaction.updated':
                    $this->handleTransactionUpdated($data);
                    break;
                    
                default:
                    Log::info('Unhandled Paddle webhook event', ['event_type' => $eventType]);
            }

            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            Log::error('Paddle wallet webhook error: ' . $e->getMessage());
            return response()->json(['error' => 'Webhook processing failed'], 500);
        }
    }

    /**
     * Handle transaction completed webhook
     */
    private function handleTransactionCompleted($data)
    {
        $transaction = $data['data'];
        $customData = $transaction['custom_data'] ?? [];
        
        if (($customData['type'] ?? '') === 'wallet_topup') {
            $userId = $customData['user_id'] ?? null;
            $amount = $customData['amount'] ?? 0;
            $customerId = $transaction['customer_id'] ?? null;
            
            if ($userId && $amount > 0) {
                // Store Paddle customer ID if not already stored
                if ($customerId) {
                    $this->paddleService->storePaddleCustomerId($userId, $customerId);
                }
                
                // Update wallet record
                $this->walletRepository->confirmTransaction(
                    $transaction['id'], 
                    'Completed'
                );
                
                Log::info('Paddle wallet top-up completed', [
                    'user_id' => $userId,
                    'amount' => $amount,
                    'transaction_id' => $transaction['id'],
                    'customer_id' => $customerId
                ]);
            }
        }
    }

    /**
     * Handle transaction updated webhook
     */
    private function handleTransactionUpdated($data)
    {
        $transaction = $data['data'];
        $status = $transaction['status'] ?? '';
        
        // Update wallet record status based on Paddle status
        $walletStatus = $this->mapPaddleStatusToWallet($status);
        
        if ($walletStatus) {
            $this->walletRepository->confirmTransaction(
                $transaction['id'], 
                $walletStatus
            );
        }
    }

    /**
     * Map Paddle transaction status to wallet status
     */
    private function mapPaddleStatusToWallet($paddleStatus)
    {
        $statusMap = [
            'completed' => 'Completed',
            'failed' => 'Failed',
            'cancelled' => 'Cancelled',
            'pending' => 'Pending',
        ];

        return $statusMap[$paddleStatus] ?? null;
    }
}
