<?php

namespace App\Http\Controllers;

use App\Models\ParadigmFormTemplate;
use App\Models\ParadigmPricing;
use App\Models\ParadigmPayment;
use App\Models\ParadigmAccessToken;
use App\Models\ParadigmClientResult;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Modules\Communication\app\Helpers\MailConfigurator;
use Modules\Communication\app\Mail\Samplemail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Support\Facades\Cache;

class ParadigmPaymentController extends Controller
{
    /**
     * Default expiration hours for assignments (7 days)
     * Since we use dashboard-based access, this is just for record-keeping
     */
    private const DEFAULT_EXPIRATION_HOURS = 168;

    /**
     * Generate a payment link for a Paradigm template
     */
    public function generatePaymentLink(Request $request)
    {
        Log::info('generatePaymentLink request received', $request->all());

        $request->validate([
            'template_id' => 'required|exists:paradigm_form_templates,id',
            'payment_type' => 'required|in:free,provider,client',
            'client_email' => 'required|email',
            'client_name' => 'required|string|max:255',
            'client_surname' => 'required|string|max:255',
            'client_gender' => 'required|string',
            'client_dob' => 'required|date',
            'client_whatsapp' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $user = Auth::user();
            $providerId = ($user->user_type == 4 && $user->userDetail && $user->userDetail->parent_id) ? $user->userDetail->parent_id : $user->id;

            // Check if user already exists
            $clientUser = User::where('email', $request->client_email)->first();
            if (!$clientUser) {
                // Auto-generate a secure random password
                $autoGeneratedPassword = Str::random(16);
                
                // Create new user if not exists
                $clientUser = User::create([
                    'name' => $request->client_name . ' ' . $request->client_surname,
                    'email' => $request->client_email,
                    'password' => Hash::make($autoGeneratedPassword),
                    'phone_number' => $request->client_whatsapp, // Update User model phone_number
                    'user_type' => 3, // Client user type
                    'status' => 1,
                    'email_verified_at' => now(), // Auto-verify email
                ]);

                $clientUser->userDetail()->create([
                    'first_name' => $request->client_name,
                    'last_name' => $request->client_surname,
                    'gender' => $request->client_gender,
                    'dob' => $request->client_dob,
                    'mobile_number' => $request->client_whatsapp,
                ]);
            } else {
                // User exists - update User and userDetail
                $clientUser->update([
                    'phone_number' => $request->client_whatsapp, // Update User model phone_number
                ]);
                
                if ($clientUser->userDetail) {
                    // Update existing userDetail
                    $clientUser->userDetail->update([
                        'first_name' => $request->client_name,
                        'last_name' => $request->client_surname,
                        'gender' => $request->client_gender,
                        'dob' => $request->client_dob,
                        'mobile_number' => $request->client_whatsapp,
                    ]);
                } else {
                    // Create userDetail if it doesn't exist
                    $clientUser->userDetail()->create([
                        'first_name' => $request->client_name,
                        'last_name' => $request->client_surname,
                        'gender' => $request->client_gender,
                        'dob' => $request->client_dob,
                        'mobile_number' => $request->client_whatsapp,
                    ]);
                }
            }

            $clientId = $clientUser->id;
            $template = ParadigmFormTemplate::findOrFail($request->template_id);
            $paymentType = $request->payment_type;
            
            // Admin-generated links are always free
            if ($user->user_type == 1) {
                $amount = 0;
            } else {
                $pricing = ParadigmPricing::where('template_id', $template->id)
                    ->where('is_active', true)
                    ->first();
                $amount = $pricing ? $pricing->price : 0;
            }
            $currency = 'USD'; // Or get from settings

            // Check wallet balance before creating payment
            if ($paymentType === 'provider' && $amount > 0) {
                $walletService = new \App\Services\ParadigmWalletService();
                if (!$walletService->hasSufficientBalance($providerId, $amount)) {
                    throw new \Exception('Insufficient wallet balance.');
                }
            }

            // Create ParadigmPayment record first (needed for wallet deduction reference)
            $payment = ParadigmPayment::create([
                'provider_id' => $providerId,
                'client_id' => $clientId, // Link to the new/existing user
                'client_email' => $request->client_email,
                'client_name' => $request->client_name,
                'client_surname' => $request->client_surname,
                'client_gender' => $request->client_gender,
                'client_dob' => $request->client_dob,
                'client_whatsapp' => $request->client_whatsapp,
                'template_id' => $template->id,
                'amount' => $paymentType === 'client' ? $amount : 0,
                'paid_by' => $paymentType,
                'status' => ($paymentType === 'provider' || $paymentType === 'free') ? 'paid' : 'pending',
                'payment_link' => ParadigmPayment::generatePaymentLink(),
                'expires_at' => now()->addDays(30),
                'self_purchased' => 0,
            ]);

            // Handle wallet deduction after payment is created (so we have payment ID)
            if ($paymentType === 'provider' && $amount > 0) {
                $walletService = new \App\Services\ParadigmWalletService();
                $walletTransaction = $walletService->deductProviderWallet(
                    $providerId, 
                    $amount, 
                    $payment->id, 
                    "Paradigm assessment for client {$request->client_email}"
                );
                
                if (!$walletTransaction) {
                    throw new \Exception('Failed to deduct from wallet. Please try again.');
                }
            }

            // Handle free assessment: create access token and update provider's free assessment fields
            if ($paymentType === 'free') {
                // Create access token for the free assessment
                $accessToken = ParadigmAccessToken::createForPayment($payment, 168); // 7 days
                
                // If this is a provider creating a free assessment for themselves, update their free_assessment_used flag
                if ($user->email === $request->client_email && !$user->free_assessment_used) {
                    $user->update([
                        'free_assessment_used' => true,
                        'free_assessment_id' => $payment->id
                    ]);
                }
            }

            // Send notifications
            if ($request->send_email || $request->send_whatsapp) {
                $this->sendAssignmentEmail(
                    $request->client_email,
                    $template->name,
                    $paymentType,
                    $amount,
                    $currency,
                    $request->client_whatsapp,
                    $providerId,
                    $payment
                );
            }

            // Get app URL for assessment link
            $settings = \DB::table('general_settings')->pluck('value', 'key')->toArray();
            $appUrl = $settings['website'] ?? config('app.url', url('/'));

            DB::commit();
            return response()->json([
                'success' => true, 
                'message' => 'Assessment assigned successfully.',
                'data' => [
                    'is_free' => ($user->user_type == 1 || $amount == 0),
                    'payment_link' => $payment->payment_link,
                    'assessment_link' => $appUrl . '/paradigm/assessment-direct/' . $payment->payment_link
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error in generatePaymentLink', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['success' => false, 'message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Process payment completion (called by payment gateway webhooks)
     */
    public function processPaymentCompletion(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'payment_link' => 'required|exists:paradigm_payments,payment_link',
            'payment_reference' => 'required|string',
            'payment_method' => 'required|string',
            'client_id' => 'required|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $payment = ParadigmPayment::where('payment_link', $request->payment_link)->firstOrFail();

            // SECURITY FIX: Idempotency check - prevent duplicate processing
            if ($payment->status === 'paid' || $payment->status === 'completed') {
                // Payment already processed - return existing access token
                $existingToken = ParadigmAccessToken::where('payment_id', $payment->id)
                    ->where('status', 'active')
                    ->first();
                
                return response()->json([
                    'success' => true,
                    'message' => 'Payment already processed',
                    'data' => [
                        'access_token' => $existingToken ? $existingToken->token : null,
                        'expires_at' => $existingToken ? $existingToken->expires_at : null
                    ]
                ]);
            }

            // Update payment status
            $payment->update([
                'client_id' => $request->client_id,
                'status' => 'paid',
                'payment_reference' => $request->payment_reference,
                'payment_method' => $request->payment_method,
                'paid_at' => now()
            ]);

            // Create access token
            $accessToken = ParadigmAccessToken::createForPayment($payment, 24);

            // Create client result record
            ParadigmClientResult::create([
                'client_id' => $request->client_id,
                'provider_id' => $payment->provider_id,
                'template_id' => $payment->template_id,
                'payment_id' => $payment->id,
                'status' => 'in_progress',
                'expires_at' => now()->addDays(7) // 7 days to complete assessment
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment processed successfully',
                'data' => [
                    'access_token' => $accessToken->token,
                    'expires_at' => $accessToken->expires_at
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process payment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate access token and get assessment access
     */
    public function validateAccess(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'access_token' => 'required|string|exists:paradigm_access_tokens,token'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid access token'
            ], 401);
        }

        try {
            $accessToken = ParadigmAccessToken::where('token', $request->access_token)
                ->with(['template', 'payment'])
                ->firstOrFail();

            if (!$accessToken->isActive()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Access token has expired or been used'
                ], 401);
            }

            $payment = $accessToken->payment;
            
            // Check if payment is required and not completed
            $requiresPayment = ($payment->paid_by === 'client' && $payment->status !== 'completed');

            return response()->json([
                'success' => true,
                'message' => 'Access validated',
                'data' => [
                    'template' => $accessToken->template,
                    'payment_status' => $payment->status,
                    'paid_by' => $payment->paid_by,
                    'requires_payment' => $requiresPayment,
                    'amount' => $payment->amount,
                    'currency' => $payment->currency,
                    'payment_id' => $payment->id
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to validate access: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Process wallet payment for client
     * Uses ParadigmWalletService (safe wrapper)
     */
    public function processWalletPayment(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'access_token' => 'required|string|exists:paradigm_access_tokens,token'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid access token'
            ], 422);
        }

        try {
            $accessToken = ParadigmAccessToken::where('token', $request->access_token)
                ->with(['payment'])
                ->firstOrFail();

            $payment = $accessToken->payment;

            // Verify payment requires client payment
            if ($payment->paid_by !== 'client') {
                return response()->json([
                    'success' => false,
                    'message' => 'This assessment does not require client payment'
                ], 400);
            }

            // SECURITY FIX: Wrap in transaction to prevent race conditions
            \DB::beginTransaction();
            
            try {
                // Lock the payment record to prevent concurrent processing
                $payment = \App\Models\ParadigmPayment::where('id', $payment->id)
                    ->lockForUpdate()
                    ->first();

                // Check if already paid (with lock held)
                if ($payment->status === 'completed' || $payment->status === 'paid') {
                    \DB::rollBack();
                    return response()->json([
                        'success' => true,
                        'message' => 'Payment already completed'
                    ]);
                }

                $clientId = Auth::id();
                $walletService = new \App\Services\ParadigmWalletService();

                // Check if client has sufficient balance
                if (!$walletService->hasSufficientBalance($clientId, $payment->amount)) {
                    \DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Insufficient wallet balance'
                    ], 400);
                }

                // Deduct from client wallet using safe wrapper service
                $description = "Paradigm Assessment - " . $payment->template->name;
                $walletTransaction = $walletService->deductClientWallet(
                    $clientId,
                    $payment->amount,
                    $payment->id,
                    $description
                );

                if (!$walletTransaction) {
                    \DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to process wallet payment'
                    ], 500);
                }

                // Update payment record
                $payment->update([
                    'status' => 'paid',
                    'client_id' => $clientId,
                    'wallet_transaction_id' => $walletTransaction->id,
                    'payment_method' => 'client_wallet',
                    'payment_reference' => 'WALLET-CLIENT-' . $payment->id,
                    'paid_at' => now()
                ]);

                // Update access token with client_id
                $accessToken->update(['client_id' => $clientId]);

                // Generate invoice for Paradigm purchase (Type 5)
                try {
                    $pdfPath = \App\Helpers\InvoiceHelper::generateInvoice(
                        $payment->id,
                        $payment->amount,
                        5, // Type 5 = Paradigm Purchase
                        $clientId
                    );
                    Log::info('Paradigm purchase invoice generated (client wallet)', ['pdf_path' => $pdfPath, 'payment_id' => $payment->id]);
                } catch (\Exception $e) {
                    Log::error('Failed to generate Paradigm purchase invoice: ' . $e->getMessage());
                }
                
                // Send paradigm purchase confirmation email (Template 109)
                try {
                    $this->sendParadigmPurchaseEmail($payment, $clientId);
                } catch (\Exception $e) {
                    Log::error('Failed to send paradigm purchase email: ' . $e->getMessage());
                }

                // Commit transaction - all changes successful
                \DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'Payment successful'
                ]);

            } catch (\Exception $innerException) {
                // Rollback transaction on any error
                \DB::rollBack();
                throw $innerException; // Re-throw to be caught by outer catch
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process payment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send email notification to client when assessment is assigned
     */
    private function sendAssignmentEmail($clientEmail, $templateName, $paidBy, $amount = 0, $currency = 'USD', $clientWhatsApp = null, $overrideProviderId = null, $payment = null)
    {
        try {
            // Configure SMTP from admin settings (required for email to work)
            MailConfigurator::configureMail();
            
            // Get the email template from database (Template 106 - Paradigm Invitation)
            $template = \DB::table('templates')
                ->where('notification_type', 10)
                ->where('type', 1)
                ->where('status', 1)
                ->first();
            
            // Get company settings
            $settings = \DB::table('general_settings')
                ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no', 'company_address'])
                ->pluck('value', 'key');
            
            $companyName = $settings['company_name'] ?? config('app.name', 'System');
            $companyWebsite = $settings['website'] ?? config('app.url', url('/'));
            $companyEmail = $settings['site_email'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyAddress = $settings['company_address'] ?? '';
            
            // Build placeholders
            $appName = $companyName;
            $appUrl = $companyWebsite;
            $loginUrl = $appUrl . '/login';
            $registerUrl = $appUrl;
            $dashboardUrl = $appUrl . '/user/paradigm/assessments';
            
            // Use direct access link if payment has client details, otherwise use old flow
            // FIX: Force app.iseekhelpoffice.com domain
            $baseUrl = 'https://app.iseekhelpoffice.com';
            
            if ($payment && $payment->client_name && $payment->client_surname) {
                $assessmentLink = $baseUrl . '/paradigm/assessment-direct/' . $payment->payment_link;
            } else {
                $assessmentLink = $baseUrl . '/paradigm/assessment'; // Link to start assessment (old flow)
            }
            
            // Generate a unique access token (you can customize this)
            $accessToken = strtoupper(substr(md5($clientEmail . time()), 0, 12));
            
            // Get provider name from auth user
            $providerName = auth()->user()->name ?? 'Your Provider';
            
            // Build payment info based on type
            $paymentInfo = '';
            if ($paidBy === 'free') {
                $paymentInfo = 'This assessment is free of charge.';
            } elseif ($paidBy === 'provider') {
                $paymentInfo = 'This assessment has been paid for by your practitioner.';
            } elseif ($paidBy === 'client') {
                $paymentInfo = "Payment of {$currency} {$amount} will be deducted from your wallet when you start the assessment.";
            }
            
            if ($template) {
                // All possible placeholders for Paradigm Invitation
                $placeholders = [
                    '{{client_name}}', '{{client_email}}', '{{template_name}}', 
                    '{{provider_name}}', '{{access_token}}', '{{assessment_link}}',
                    '{{payment_info}}', '{{login_url}}', '{{register_url}}', 
                    '{{dashboard_url}}', '{{app_name}}', '{{company_name}}',
                    '{{company_website}}', '{{company_email}}', '{{company_phone}}',
                    '{{company_address}}'
                ];
                
                $replacements = [
                    'Client', $clientEmail, $templateName,
                    $providerName, $accessToken, $assessmentLink,
                    $paymentInfo, $loginUrl, $registerUrl,
                    $dashboardUrl, $appName, $companyName,
                    $companyWebsite, $companyEmail, $companyPhone,
                    $companyAddress
                ];
                
                $subject = str_replace($placeholders, $replacements, $template->subject);
                $emailContent = str_replace($placeholders, $replacements, $template->content);
                
                // Use Samplemail class (same as all other system emails)
                $data = [
                    'subject' => $subject,
                    'message' => $emailContent,
                    'attachment' => null
                ];
                
                Mail::to($clientEmail)->send(new Samplemail($data));
            } else {
                // Fallback to plain text if template not found
                $subject = "New Assessment Assigned: {$templateName}";
                $message = "Hello,\n\n";
                $message .= "You have been assigned a new Paradigm Profile assessment: {$templateName}\n\n";
                $message .= "{$paymentInfo}\n\n";
                $message .= "To start your assessment:\n";
                $message .= "1. Log in to your account: {$loginUrl}\n";
                $message .= "2. Go to your assessments dashboard: {$dashboardUrl}\n";
                $message .= "3. Click 'Start' on the assigned assessment\n\n";
                $message .= "If you don't have an account yet, please register using this email address.\n\n";
                $message .= "Best regards,\n{$appName}";
                
                $data = [
                    'subject' => $subject,
                    'message' => $message,
                    'attachment' => null
                ];
                
                Mail::to($clientEmail)->send(new Samplemail($data));
            }
            
            // Send Paradigm invitation WhatsApp
            try {
                $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
                
                // Use provided WhatsApp number if available, otherwise try to get from registered user
                $phoneNumber = $clientWhatsApp;
                
                if (empty($phoneNumber)) {
                    // Fallback: try to get phone from registered user
                    $clientUser = \App\Models\User::where('email', $clientEmail)->first();
                    $phoneNumber = $clientUser->phone_number ?? null;
                }
                
                if (!empty($phoneNumber)) {
                    // Get provider details for WhatsApp template
                    // Use override provider ID if provided (for free/self-purchased assessments), otherwise use current auth user
                    $providerId = $overrideProviderId ?? auth()->id();
                    $provider = \App\Models\User::find($providerId);
                    
                    if ($provider) {
                        // Get category name
                        $categoryName = 'Provider'; // Default
                        if ($provider->user_type == 2) {
                            $providerDetails = \App\Models\ProviderDetail::where('user_id', $provider->id)->first();
                            if ($providerDetails && !empty($providerDetails->category_id)) {
                                $category = \DB::table('categories')->where('id', $providerDetails->category_id)->first();
                                $categoryName = $category->name ?? 'Provider';
                            }
                        } elseif ($provider->user_type == 1) {
                            // For admin, use company name as category
                            $categoryName = 'Support Team';
                        }
                        
                        $providerName = $provider->name ?? 'Provider';
                        $providerEmail = $provider->email ?? '';
                        $providerPhone = $provider->phone_number ?? '';
                        
                        // Generate assessment link
                        $assessmentLink = null;
                        if ($payment && $payment->payment_link) {
                            // Use direct access link if client details provided, otherwise use old flow
                            if ($payment->client_name && $payment->client_surname) {
                                $assessmentLink = url('/paradigm/assessment-direct/' . $payment->payment_link);
                            } else {
                                $assessmentLink = url('/paradigm/assessment'); // Old flow - requires login
                            }
                        }
                        
                        // Get client name for WhatsApp template
                        $clientName = 'Client'; // Default
                        if ($payment) {
                            if ($payment->client_name && $payment->client_surname) {
                                $clientName = trim($payment->client_name . ' ' . $payment->client_surname);
                            } elseif ($payment->client_name) {
                                $clientName = $payment->client_name;
                            }
                        }
                        
                        // Send WhatsApp to client using client template (userpurchaseassessment)
                        // Template variables: {{1}} = Client Name, {{2}} = Assessment Link
                        $smsRepository->sendParadigmInvitationClientWhatsApp(
                            $phoneNumber,
                            $clientName,
                            $assessmentLink
                        );
                    }
                }
            } catch (\Exception $whatsappEx) {
                Log::warning('Failed to send paradigm invitation WhatsApp: ' . $whatsappEx->getMessage());
            }
            
        } catch (\Exception $e) {
            // Log error but don't fail the assignment
            Log::error('Failed to send assignment email: ' . $e->getMessage());
        }
    }
    
    /**
     * Send paradigm purchase confirmation email (Template 109)
     */
    private function sendParadigmPurchaseEmail($payment, $clientId)
    {
        try {
            // Configure mail
            \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
            
            // Get template (Paradigm Purchase Confirmation - notification_type 13)
            $template = \DB::table('templates')
                ->where('notification_type', 13) // Self-Purchase Confirmation notification type
                ->where('type', 1) // Email type
                ->where('status', 1) // Active
                ->first();
            
            if (!$template) {
                \Log::warning('Paradigm purchase email template not found', [
                    'notification_type' => 13,
                    'type' => 1
                ]);
                return;
            }
            
            // Get client details
            $client = \App\Models\User::with('userDetails')->find($clientId);
            if (!$client) {
                \Log::warning('Client not found for paradigm purchase email', ['client_id' => $clientId]);
                return;
            }
            
            // Get template details
            $payment->load('template');
            $templateName = $payment->template ? $payment->template->name : 'Paradigm Assessment';
            
            // Get company settings
            $settings = \DB::table('general_settings')
                ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no', 'address'])
                ->pluck('value', 'key');
            
            $companyName = $settings['company_name'] ?? config('app.name');
            $companyWebsite = $settings['website'] ?? config('app.url');
            $companyEmail = $settings['site_email'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyAddress = $settings['address'] ?? '';
            
            // Get currency
            $walletRepo = new \App\Repositories\Eloquent\WalletRepository();
            $currency = $walletRepo->getCurrencySymbol();
            $currencySymbol = $currency ? $currency->symbol : 'R';
            
            // Prepare replacements
            $clientName = $client->name ?? '';
            $firstName = $client->userDetails->first_name ?? '';
            $lastName = $client->userDetails->last_name ?? '';
            $amount = $payment->amount;
            $paymentRef = $payment->payment_reference ?? '';
            $accessUrl = url('/user/paradigm/assessments');
            
            // For self-purchase, provider details = admin/company details
            $providerName = $companyName;
            $providerEmail = $companyEmail;
            $providerPhone = $companyPhone;
            
            // Replace placeholders in subject
            $subject = str_replace(
                ['{{client_name}}', '{{first_name}}', '{{last_name}}', '{{user_name}}', '{{template_name}}', '{{amount}}', '{{company_name}}', '{{provider_name}}', '{{provider_email}}', '{{provider_phone}}'],
                [$clientName, $firstName, $lastName, $clientName, $templateName, $currencySymbol . $amount, $companyName, $providerName, $providerEmail, $providerPhone],
                $template->subject
            );
            
            // Replace placeholders in content
            $content = str_replace(
                ['{{client_name}}', '{{first_name}}', '{{last_name}}', '{{user_name}}', '{{template_name}}', '{{amount}}', '{{payment_reference}}', '{{access_url}}', '{{company_name}}', '{{website_link}}', '{{company_email}}', '{{company_phone}}', '{{company_address}}', '{{provider_name}}', '{{provider_email}}', '{{provider_phone}}'],
                [$clientName, $firstName, $lastName, $clientName, $templateName, $currencySymbol . $amount, $paymentRef, $accessUrl, $companyName, $companyWebsite, $companyEmail, $companyPhone, $companyAddress, $providerName, $providerEmail, $providerPhone],
                $template->content
            );
            
            // Send email
            $emailData = [
                'to_email' => $client->email,
                'subject' => $subject,
                'content' => $content
            ];
            
            \Log::info('Sending paradigm purchase confirmation email', [
                'to_email' => $client->email,
                'amount' => $amount,
                'template' => $templateName
            ]);
            
            $emailRequest = new \Illuminate\Http\Request($emailData);
            $emailController = new \Modules\Communication\app\Http\Controllers\EmailController();
            $emailController->sendEmail($emailRequest);
            
            \Log::info('Paradigm purchase confirmation email sent successfully');
            
        } catch (\Exception $e) {
            \Log::error('Failed to send paradigm purchase email: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Register user from chatbot (direct creation like provider payment links)
     */
    public function chatbotRegister(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'first_name' => 'required|string|max:100',
                'last_name' => 'required|string|max:100',
                'email' => 'required|email', // Don't check unique - we handle existing users
                'phone_number' => 'required|string',
                'gender' => 'nullable|string',
                'dob' => 'nullable|date',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'code' => 422,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }

            // Check if user already exists
            $user = User::where('email', $request->email)->first();
            
            if (!$user) {
                // Auto-generate a secure random password
                $autoGeneratedPassword = Str::random(16);
                
                // Generate unique username
                $firstName = strtolower(str_replace(' ', '', $request->first_name));
                $lastName = strtolower(str_replace(' ', '', $request->last_name));
                $randomSuffix = rand(1000, 9999);
                $uniqueName = $firstName . '_' . $lastName . '_' . $randomSuffix;
                
                // Create new user
                $user = User::create([
                    'name' => $uniqueName,
                    'email' => $request->email,
                    'password' => Hash::make($autoGeneratedPassword),
                    'phone_number' => $request->phone_number,
                    'user_type' => 3, // Client user type
                    'status' => 1,
                    'email_verified_at' => now(), // Auto-verify email
                ]);

                // Create userDetail
                $user->userDetail()->create([
                    'first_name' => $request->first_name,
                    'last_name' => $request->last_name,
                    'gender' => $request->gender,
                    'dob' => $request->dob,
                    'mobile_number' => $request->phone_number,
                ]);
            } else {
                // User exists - update if needed
                $user->update([
                    'phone_number' => $request->phone_number,
                ]);
                
                if ($user->userDetail) {
                    $user->userDetail->update([
                        'first_name' => $request->first_name,
                        'last_name' => $request->last_name,
                        'gender' => $request->gender ?? $user->userDetail->gender,
                        'dob' => $request->dob ?? $user->userDetail->dob,
                        'mobile_number' => $request->phone_number,
                    ]);
                } else {
                    $user->userDetail()->create([
                        'first_name' => $request->first_name,
                        'last_name' => $request->last_name,
                        'gender' => $request->gender,
                        'dob' => $request->dob,
                        'mobile_number' => $request->phone_number,
                    ]);
                }
            }

            // Auto-login the user
            Auth::login($user);
            session(['user_id' => $user->id]);
            Cache::forever('user_auth_id', $user->id);

            return response()->json([
                'code' => 200,
                'message' => 'Registration Successful',
                'user_id' => $user->id,
                'register_status' => '0', // No verification needed
            ]);

        } catch (\Exception $e) {
            Log::error('Chatbot registration error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'code' => 500,
                'message' => 'Registration failed: ' . $e->getMessage()
            ], 500);
        }
    }
}