<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Repositories\Eloquent\PayFastRepository;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PayFastController extends Controller
{
    private $payFastRepo;
    
    public function __construct()
    {
        $this->payFastRepo = new PayFastRepository();
    }
    
    /**
     * Initiate PayFast payment
     * Follows Stripe pattern exactly
     */
    public function payment(Request $request)
    {
        try {
            $amount = $request->input('amount');
            $userId = $request->input('user_id');
            $trxId = $request->input('trx_id');
            $type = $request->input('type', 'wallet');
            $userName = $request->input('usrname', '');
            
            // Get user email
            $user = Auth::user();
            $userEmail = $user ? $user->email : '';
            
            if (empty($amount) || empty($trxId)) {
                return redirect()->back()->with('error', 'Invalid payment parameters.');
            }
            
            Log::info('PayFast payment initiated', [
                'amount' => $amount,
                'trx_id' => $trxId,
                'user_id' => $userId,
                'type' => $type
            ]);
            
            $paymentData = $this->payFastRepo->createPayment($amount, $userId, $trxId, $type, $userName, $userEmail);
            
            // Return view with auto-submit form (like Stripe does)
            return view('payfast.redirect', [
                'action_url' => $paymentData['action_url'],
                'payment_data' => $paymentData['data']
            ]);
            
        } catch (\Exception $e) {
            Log::error('PayFast payment initiation failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Payment initiation failed. Please try again.');
        }
    }
    
    /**
     * Handle successful payment return - Provider
     */
    public function successProvider(Request $request)
    {
        Log::info('PayFast success - Provider', $request->all());
        
        return redirect()->route('provider.wallet')->with('success', 'Payment successful! Your wallet will be updated shortly.');
    }
    
    /**
     * Handle successful payment return - User
     */
    public function successUser(Request $request)
    {
        Log::info('PayFast success - User', $request->all());
        
        // Check payment status from PayFast
        $paymentStatus = $request->input('payment_status', '');
        if (strtolower($paymentStatus) === 'failed' || strtolower($paymentStatus) === 'cancelled') {
            Log::warning('PayFast success - Payment failed or cancelled', ['status' => $paymentStatus]);
            return redirect()->route('user.paradigm.buy-credits')->with('error', 'Payment was not successful. Please try again.');
        }
        
        // Check if this is a Paradigm profile purchase
        $mPaymentId = $request->input('m_payment_id');
        
        // Get authenticated user
        $user = Auth::user();
        if (!$user) {
            Log::warning('PayFast success - User not authenticated');
            return redirect()->route('user.wallet')->with('success', 'Payment successful! Your wallet will be updated shortly.');
        }
        
        // Get Paradigm package for comparison
        $paradigmPackage = \App\Models\CreditPackage::where('is_paradigm_package', true)
            ->where('is_active', true)
            ->first();
        
        if (!$paradigmPackage) {
            Log::info('PayFast success - No Paradigm package configured, normal wallet top-up');
            return redirect()->route('user.wallet')->with('success', 'Payment successful! Your wallet will be updated shortly.');
        }
        
        $requiredCredits = (float) $paradigmPackage->credits;
        
        // Try multiple methods to find the transaction
        $walletHistory = null;
        $transactionId = $mPaymentId;
        
        // Method 1: Find by m_payment_id from PayFast return
        if ($mPaymentId) {
            $walletHistory = \App\Models\WalletHistory::where('transaction_id', $mPaymentId)->first();
            if ($walletHistory) {
                Log::info('PayFast success - Found transaction by m_payment_id', ['transaction_id' => $mPaymentId]);
            }
        }
        
        // Method 2: Find by user's most recent PayFast transaction (for sandbox or if m_payment_id doesn't match)
        if (!$walletHistory) {
            $recentTransaction = \App\Models\WalletHistory::where('user_id', $user->id)
                ->where('payment_type', 'payfast')
                ->whereIn('status', ['pending', 'Pending', 'Completed', 'completed'])
                ->orderBy('created_at', 'desc')
                ->first();
            
            if ($recentTransaction) {
                // Check if credits match Paradigm package (within 1 second of creation for sandbox)
                $timeDiff = now()->diffInSeconds($recentTransaction->created_at);
                $creditsMatch = abs($recentTransaction->credits - $requiredCredits) < 0.01;
                
                if ($creditsMatch && $timeDiff < 300) { // Within 5 minutes
                    Log::info('PayFast success - Found matching recent transaction', [
                        'transaction_id' => $recentTransaction->transaction_id,
                        'credits' => $recentTransaction->credits,
                        'time_diff_seconds' => $timeDiff
                    ]);
                    $walletHistory = $recentTransaction;
                    $transactionId = $recentTransaction->transaction_id;
                }
            }
        }
        
        // Method 3: Check if user has any pending PayFast transaction matching Paradigm credits
        if (!$walletHistory) {
            $pendingTransaction = \App\Models\WalletHistory::where('user_id', $user->id)
                ->where('payment_type', 'payfast')
                ->where('status', 'pending')
                ->whereRaw('ABS(credits - ?) < 0.01', [$requiredCredits])
                ->orderBy('created_at', 'desc')
                ->first();
            
            if ($pendingTransaction) {
                $timeDiff = now()->diffInSeconds($pendingTransaction->created_at);
                if ($timeDiff < 300) { // Within 5 minutes
                    Log::info('PayFast success - Found pending matching transaction', [
                        'transaction_id' => $pendingTransaction->transaction_id,
                        'time_diff_seconds' => $timeDiff
                    ]);
                    $walletHistory = $pendingTransaction;
                    $transactionId = $pendingTransaction->transaction_id;
                }
            }
        }
        
        if ($walletHistory) {
            // Verify credits match Paradigm package
            $creditsMatch = abs($walletHistory->credits - $requiredCredits) < 0.01;
            
            Log::info('PayFast success - Checking Paradigm match', [
                'wallet_credits' => $walletHistory->credits,
                'package_credits' => $requiredCredits,
                'match' => $creditsMatch,
                'user_id' => $walletHistory->user_id,
                'transaction_id' => $transactionId,
                'status' => $walletHistory->status
            ]);
            
            if ($creditsMatch) {
                // This is a Paradigm profile purchase - redirect to success page
                Log::info('PayFast success - Paradigm purchase detected, redirecting to success page...', [
                    'user_id' => $walletHistory->user_id,
                    'transaction_id' => $transactionId
                ]);
                return redirect()->route('user.paradigm.payment-success', ['transaction_id' => $transactionId]);
            } else {
                Log::info('PayFast success - Credits do not match Paradigm package', [
                    'wallet_credits' => $walletHistory->credits,
                    'required_credits' => $requiredCredits
                ]);
            }
        } else {
            Log::warning('PayFast success - Wallet history not found after all attempts', [
                'm_payment_id' => $mPaymentId,
                'user_id' => $user->id,
                'all_request_params' => $request->all()
            ]);
        }
        
        return redirect()->route('user.wallet')->with('success', 'Payment successful! Your wallet will be updated shortly.');
    }
    
    /**
     * Handle Paradigm profile purchase after PayFast payment
     * @param int $userId
     * @param string $transactionId
     * @param bool $returnJson - If true, returns JSON response for API, otherwise redirects
     */
    private function handleParadigmProfilePurchase($userId, $transactionId, $returnJson = false)
    {
        try {
            // Get user
            $user = \App\Models\User::find($userId);
            if (!$user) {
                if ($returnJson) {
                    return response()->json([
                        'success' => false,
                        'message' => 'User not found.'
                    ], 404);
                }
                return redirect()->route('user.wallet')->with('error', 'User not found.');
            }
            
            // Get Paradigm package
            $package = \App\Models\CreditPackage::where('is_paradigm_package', true)
                ->where('is_active', true)
                ->first();
            
            if (!$package) {
                if ($returnJson) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Paradigm Profile package not configured.'
                    ], 404);
                }
                return redirect()->route('user.wallet')->with('error', 'Paradigm Profile package not configured.');
            }
            
            $requiredCredits = (float) $package->credits;
            $walletService = new \App\Services\ParadigmWalletService();
            
            // Get transaction credits (ITN should have already processed by the time this is called)
            $walletHistory = \App\Models\WalletHistory::where('transaction_id', $transactionId)
                ->where('user_id', $userId)
                ->first();
            
            // Use transaction credits if available, otherwise use required credits
            $transactionCredits = $walletHistory ? (float) ($walletHistory->credits ?? $walletHistory->amount ?? 0) : 0;
            $creditsToUse = $transactionCredits > 0 ? $transactionCredits : $requiredCredits;
            
            // Check if payment already exists for this transaction (prevent duplicates)
            $existingPayment = \App\Models\ParadigmPayment::where('payment_reference', $transactionId)
                ->where('self_purchased', 1)
                ->where('client_id', $userId)
                ->first();
            
            if ($existingPayment) {
                // Payment already created
                $assessmentLink = url('/paradigm/assessment-direct/' . $existingPayment->payment_link);
                if ($returnJson) {
                    return response()->json([
                        'success' => true,
                        'message' => 'Assessment already created',
                        'assessment_link' => $assessmentLink
                    ]);
                }
                return redirect($assessmentLink)->with('success', 'Your assessment is ready!');
            }
            
            // Get template_id from paradigm_pricing
            $pricing = \DB::table('paradigm_pricing')->first();
            $templateId = $pricing ? ($pricing->template_id ?? 1) : 1;
            
            // Get admin user ID for self-purchased assessments
            $adminUser = \App\Models\User::where('user_type', 1)->first();
            $adminProviderId = $adminUser ? $adminUser->id : $userId;
            
            // Get user details
            $userDetail = $user->userDetail;
            $clientName = $userDetail ? ($userDetail->first_name ?? '') : '';
            $clientSurname = $userDetail ? ($userDetail->last_name ?? '') : '';
            
            // If no userDetail, try to get from user name
            if (empty($clientName) && !empty($user->name)) {
                $nameParts = explode(' ', $user->name, 2);
                $clientName = $nameParts[0] ?? '';
                $clientSurname = $nameParts[1] ?? '';
            }
            
            \DB::beginTransaction();
            
            try {
                // Create ParadigmPayment record
                $payment = new \App\Models\ParadigmPayment();
                $payment->provider_id = $adminProviderId;
                $payment->client_email = $user->email;
                $payment->client_id = $userId;
                $payment->client_name = $clientName;
                $payment->client_surname = $clientSurname;
                $payment->client_gender = $userDetail ? ($userDetail->gender ?? null) : null;
                $payment->client_dob = $userDetail ? ($userDetail->dob ?? null) : null;
                $payment->client_whatsapp = $user->phone_number ?? null;
                $payment->template_id = $templateId;
                $payment->amount = $creditsToUse; // Use full transaction credits
                $payment->currency = 'USD';
                $payment->status = 'paid';
                $payment->payment_link = \App\Models\ParadigmPayment::generatePaymentLink();
                $payment->paid_at = now();
                $payment->self_purchased = 1;
                $payment->payment_method = 'payfast';
                $payment->payment_reference = $transactionId;
                $payment->save();
                
                // Create access token
                $accessToken = \App\Models\ParadigmAccessToken::create([
                    'payment_id' => $payment->id,
                    'token' => \Illuminate\Support\Str::random(64),
                    'template_id' => $payment->template_id,
                    'provider_id' => $adminProviderId,
                    'client_id' => $userId,
                    'status' => 'active',
                    'expires_at' => now()->addDays(30),
                ]);
                
                // Deduct from wallet (use full transaction credits)
                $walletDeduction = $walletService->deductClientWallet(
                    $userId,
                    $creditsToUse, // Use full transaction credits
                    $payment->id,
                    'Self-purchased Paradigm Assessment'
                );
                
                if (!$walletDeduction) {
                    \DB::rollBack();
                    $errorMsg = 'Failed to deduct credits from wallet. Please ensure you have sufficient credits.';
                    if ($returnJson) {
                        return response()->json([
                            'success' => false,
                            'message' => $errorMsg
                        ], 400);
                    }
                    return redirect()->route('user.wallet')->with('error', $errorMsg);
                }
                
                // Send email and WhatsApp notifications (same as UserParadigmController)
                try {
                    $template = \App\Models\ParadigmFormTemplate::find($templateId);
                    $templateName = $template ? $template->name : 'Paradigm Profile';
                    
                    // Use UserParadigmController's notification method via reflection
                    $userParadigmController = new \App\Http\Controllers\UserParadigmController();
                    $reflection = new \ReflectionClass($userParadigmController);
                    $method = $reflection->getMethod('sendParadigmInvitationNotifications');
                    $method->setAccessible(true);
                    $method->invoke($userParadigmController, $user->email, $templateName, 'client', $requiredCredits, 'USD', $user->phone_number, $adminProviderId, $payment);
                    
                    \Log::info('Paradigm invitation notifications sent after PayFast payment', [
                        'user_id' => $userId,
                        'email' => $user->email,
                        'phone' => $user->phone_number,
                        'payment_id' => $payment->id
                    ]);
                } catch (\Exception $e) {
                    \Log::warning('Failed to send Paradigm invitation notifications after PayFast: ' . $e->getMessage(), [
                        'trace' => $e->getTraceAsString()
                    ]);
                    // Don't fail the purchase if notifications fail
                }
                
                \DB::commit();
                
                // Return response based on context
                $assessmentLink = url('/paradigm/assessment-direct/' . $payment->payment_link);
                
                if ($returnJson) {
                    return response()->json([
                        'success' => true,
                        'message' => 'Assessment created successfully',
                        'assessment_link' => $assessmentLink,
                        'payment_id' => $payment->id
                    ]);
                }
                
                return redirect($assessmentLink)->with('success', 'Payment successful! Your assessment is ready.');
                
            } catch (\Exception $e) {
                \DB::rollBack();
                \Log::error('Failed to create Paradigm payment after PayFast: ' . $e->getMessage());
                
                if ($returnJson) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to create assessment: ' . $e->getMessage()
                    ], 500);
                }
                
                return redirect()->route('user.wallet')->with('error', 'Payment successful but failed to create assessment. Please contact support.');
            }
            
        } catch (\Exception $e) {
            \Log::error('Error handling Paradigm profile purchase: ' . $e->getMessage());
            
            if ($returnJson) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('user.wallet')->with('error', 'An error occurred. Please try again.');
        }
    }
    
    /**
     * Handle cancelled payment - Provider
     */
    public function cancelProvider(Request $request)
    {
        Log::info('PayFast cancelled - Provider', $request->all());
        
        return redirect()->route('provider.wallet.topup')->with('error', 'Payment was cancelled.');
    }
    
    /**
     * Handle cancelled payment - User
     */
    public function cancelUser(Request $request)
    {
        Log::info('PayFast cancelled - User', $request->all());
        
        // Check if this was a Paradigm purchase attempt
        $mPaymentId = $request->input('m_payment_id');
        if ($mPaymentId) {
            $walletHistory = \App\Models\WalletHistory::where('transaction_id', $mPaymentId)->first();
            if ($walletHistory) {
                $paradigmPackage = \App\Models\CreditPackage::where('is_paradigm_package', true)
                    ->where('is_active', true)
                    ->first();
                if ($paradigmPackage && abs($walletHistory->credits - $paradigmPackage->credits) < 0.01) {
                    return redirect()->route('user.paradigm.buy-credits')->with('error', 'Payment was cancelled. Please try again.');
                }
            }
        }
        
        return redirect()->route('user.wallet')->with('error', 'Payment was cancelled.');
    }
    
    /**
     * Show Paradigm payment success page
     */
    public function paradigmPaymentSuccess(Request $request)
    {
        $transactionId = $request->input('transaction_id');
        
        if (!$transactionId) {
            return redirect()->route('user.paradigm.buy-credits')->with('error', 'Invalid transaction.');
        }
        
        // Verify transaction belongs to current user
        $user = Auth::user();
        $walletHistory = \App\Models\WalletHistory::where('transaction_id', $transactionId)
            ->where('user_id', $user->id)
            ->first();
        
        if (!$walletHistory) {
            return redirect()->route('user.paradigm.buy-credits')->with('error', 'Transaction not found.');
        }
        
        return view('user.paradigm.payment-success', [
            'transactionId' => $transactionId
        ]);
    }
    
    /**
     * Create assessment after payment (API endpoint)
     */
    public function createAssessmentAfterPayment(Request $request)
    {
        try {
            // Check if user is authenticated
            if (!Auth::check()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authentication required. Please log in and try again.'
                ], 401);
            }
            
            $transactionId = $request->input('transaction_id');
            $userId = Auth::id();
            
            if (!$transactionId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction ID is required'
                ], 400);
            }
            
            // Verify transaction belongs to user
            $walletHistory = \App\Models\WalletHistory::where('transaction_id', $transactionId)
                ->where('user_id', $userId)
                ->first();
            
            if (!$walletHistory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction not found'
                ], 404);
            }
            
            // For local/sandbox testing, immediately confirm transaction (no waiting)
            // In production, ITN will be sent automatically
            $isLocalTesting = app()->environment(['local', 'testing']) || config('payfast.mode') === 'sandbox';
            
            if ($isLocalTesting && strtolower($walletHistory->status ?? '') === 'pending') {
                Log::info('Local/Sandbox testing - Immediately confirming transaction', [
                    'transaction_id' => $transactionId,
                    'current_status' => $walletHistory->status
                ]);
                
                try {
                    $walletRepo = new \App\Repositories\Eloquent\WalletRepository();
                    $result = $walletRepo->confirmTransaction($transactionId, 'Completed');
                    
                    if (!$result) {
                        throw new \Exception('confirmTransaction returned false');
                    }
                    
                    // Re-query wallet history after confirmation to get updated status
                    $walletHistory = \App\Models\WalletHistory::where('transaction_id', $transactionId)
                        ->where('user_id', $userId)
                        ->first();
                    
                    Log::info('Local/Sandbox - Transaction confirmed immediately', [
                        'transaction_id' => $transactionId,
                        'wallet_history_id' => $result,
                        'new_status' => $walletHistory->status ?? 'unknown',
                        'credits' => $walletHistory->credits ?? $walletHistory->amount
                    ]);
                } catch (\Exception $e) {
                    Log::error('Local/Sandbox - Failed to confirm transaction', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString(),
                        'transaction_id' => $transactionId
                    ]);
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to confirm transaction: ' . $e->getMessage()
                    ], 500);
                }
            } else {
                // For production, wait briefly for ITN (max 3 seconds, check every 0.5 seconds)
                $maxWait = 3;
                $waited = 0;
                
                while ($waited < $maxWait) {
                    $walletHistory = \App\Models\WalletHistory::where('transaction_id', $transactionId)
                        ->where('user_id', $userId)
                        ->first();
                    
                    if ($walletHistory) {
                        $status = strtolower($walletHistory->status ?? '');
                        if (in_array($status, ['completed', 'complete'])) {
                            Log::info('Transaction completed via ITN', [
                                'transaction_id' => $transactionId,
                                'status' => $walletHistory->status,
                                'waited_seconds' => $waited
                            ]);
                            break;
                        }
                    }
                    
                    usleep(500000); // Wait 0.5 seconds
                    $waited += 0.5;
                }
                
                // If still pending in production, return error
                if ($walletHistory && strtolower($walletHistory->status ?? '') === 'pending') {
                    return response()->json([
                        'success' => false,
                        'message' => 'Credits are still being processed. Please wait a moment and try again.'
                    ], 400);
                }
            }
            
            // Final check - ensure transaction is completed
            if (!$walletHistory || !in_array(strtolower($walletHistory->status ?? ''), ['completed', 'complete'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction is not completed. Status: ' . ($walletHistory->status ?? 'unknown')
                ], 400);
            }
            
            // Use the existing handleParadigmProfilePurchase logic
            return $this->handleParadigmProfilePurchase($userId, $transactionId, true);
            
        } catch (\Exception $e) {
            Log::error('Error creating assessment after payment', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'transaction_id' => $transactionId ?? 'unknown',
                'user_id' => $userId ?? 'unknown'
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to create assessment: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Handle ITN (Instant Transaction Notification)
     * This is the webhook callback from PayFast
     * THIS IS WHERE CREDITS ARE ALLOCATED (via WalletRepository->confirmTransaction)
     */
    public function notify(Request $request)
    {
        // Log the raw ITN request as soon as it arrives, before any processing.
        \Illuminate\Support\Facades\Log::channel('stderr')->info('RAW PAYFAST ITN RECEIVED: ', $request->all());
        
        // It's crucial to send a 200 OK header back to PayFast immediately
        // to prevent them from resending the ITN.
        // We can then process the data in the background.
        try {
            // Acknowledge the request immediately to prevent PayFast timeout.
            // We send a 200 OK response and then continue processing the ITN.
            if (ob_get_level() == 0) {
                ob_start();
            }
            // Use Laravel's response helpers to send headers and content.
            response('OK', 200)->send();
            
            // End buffering and flush output to the browser
            ob_end_flush();
            ob_flush();
            flush();
            
            // Close the connection to the client and continue script execution
            if (function_exists('fastcgi_finish_request')) {
                fastcgi_finish_request();
            }

            // --- PROCESSING CONTINUES IN THE BACKGROUND ---
            
            $postData = $request->all();
            
            Log::info('PayFast ITN received and acknowledged. Background processing started.', [
                'm_payment_id' => $postData['m_payment_id'] ?? '',
                'payment_status' => $postData['payment_status'] ?? '',
                'amount_gross' => $postData['amount_gross'] ?? ''
            ]);
            
            // Process ITN - This will allocate credits if payment is COMPLETE
            $result = $this->payFastRepo->processITN($postData);
            
            if ($result) {
                Log::info('PayFast ITN processed successfully (background)', [
                    'trx_id' => $postData['m_payment_id'] ?? '',
                    'wallet_history_id' => $result
                ]);
            } else {
                Log::error('PayFast ITN validation or processing failed (background)', [
                    'trx_id' => $postData['m_payment_id'] ?? ''
                ]);
            }
            
        } catch (\Exception $e) {
            // We can't return a response here as it's already sent, so we just log the error.
            Log::error('PayFast ITN background processing error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
    
    /**
     * Manual ITN trigger for local testing
     * This allows manually updating transaction status when testing locally
     * since ITNs are sent to production server, not localhost
     */
    public function manualITNTrigger(Request $request)
    {
        // Only allow in local/sandbox environment
        if (!app()->environment(['local', 'testing']) && config('payfast.mode') !== 'sandbox') {
            return response()->json([
                'success' => false,
                'message' => 'This endpoint is only available in local/sandbox environment'
            ], 403);
        }
        
        try {
            $transactionId = $request->input('transaction_id');
            
            if (!$transactionId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction ID is required'
                ], 400);
            }
            
            // Find the transaction
            $walletHistory = \App\Models\WalletHistory::where('transaction_id', $transactionId)->first();
            
            if (!$walletHistory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction not found'
                ], 404);
            }
            
            // Use WalletRepository to confirm transaction (same as production ITN)
            $walletRepo = new \App\Repositories\Eloquent\WalletRepository();
            $result = $walletRepo->confirmTransaction($transactionId, 'Completed');
            
            if ($result) {
                Log::info('Manual ITN trigger - Transaction confirmed', [
                    'transaction_id' => $transactionId,
                    'wallet_history_id' => $result
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Transaction status updated to completed',
                    'transaction_id' => $transactionId,
                    'wallet_history_id' => $result
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to confirm transaction'
                ], 500);
            }
            
        } catch (\Exception $e) {
            Log::error('Manual ITN trigger error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }
}

