<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Models\ParadigmAssessmentResult;
use App\Models\ParadigmMarketplacePurchase;
use App\Models\ParadigmLinkRequest;
use App\Models\ParadigmPayment;
use App\Models\User;
use App\Services\ParadigmWalletService;
use App\Services\ParadigmPdfServiceDomPDF;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ParadigmMarketplaceController extends Controller
{
    protected $walletService;
    protected $pdfService;

    public function __construct()
    {
        $this->walletService = new ParadigmWalletService();
        $this->pdfService = new ParadigmPdfServiceDomPDF();
    }

    /**
     * List available marketplace profiles
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Get available profiles (completed, self-purchased, not sold)
        // Self-purchased profiles have provider_id = admin (user_type = 1)
        $query = ParadigmAssessmentResult::with(['template'])
            ->where('is_available_in_marketplace', true)
            ->whereNotNull('completed_at')
            ->whereDoesntHave('marketplacePurchases', function($q) {
                $q->where('is_sold', true);
            })
            ->whereHas('provider', function($q) {
                // Provider should be admin (user_type = 1) for self-purchased profiles
                $q->where('user_type', 1);
            });

        // Filter by paradigm type
        if ($request->filled('type')) {
            $query->where('paradigm_type', $request->type);
        }

        // Search by name
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('surname', 'like', "%{$search}%");
            });
        }

        $profiles = $query->orderBy('marketplace_listed_at', 'desc')->paginate(20);

        // Calculate prices for each profile
        $profiles->getCollection()->transform(function($profile) {
            $profile->marketplace_price = $profile->getMarketplacePrice();
            return $profile;
        });

        return view('provider.paradigm.marketplace.index', compact('profiles'));
    }

    /**
     * Show profile details before purchase
     */
    public function show($resultId)
    {
        $result = ParadigmAssessmentResult::with(['template'])->findOrFail($resultId);
        
        // Check if already sold
        $isSold = ParadigmMarketplacePurchase::where('result_id', $resultId)
            ->where('is_sold', true)
            ->exists();
        
        if ($isSold) {
            return redirect()->route('provider.paradigm.marketplace')
                ->with('error', 'This profile has already been purchased.');
        }

        $price = $result->getMarketplacePrice();
        $user = Auth::user();
        $balance = $this->walletService->getBalance($user->id);

        return view('provider.paradigm.marketplace.show', compact('result', 'price', 'balance'));
    }

    /**
     * Purchase profile
     */
    public function purchase(Request $request, $resultId)
    {
        $user = Auth::user();
        $result = ParadigmAssessmentResult::findOrFail($resultId);

        // Check if already sold
        $isSold = ParadigmMarketplacePurchase::where('result_id', $resultId)
            ->where('is_sold', true)
            ->exists();
        
        if ($isSold) {
            return response()->json([
                'success' => false,
                'message' => 'This profile has already been purchased.'
            ], 400);
        }

        $price = $result->getMarketplacePrice();
        
        if ($price <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid price for this profile.'
            ], 400);
        }

        // Check wallet balance
        if (!$this->walletService->hasSufficientBalance($user->id, $price)) {
            $balance = $this->walletService->getBalance($user->id);
            return response()->json([
                'success' => false,
                'message' => 'Insufficient wallet balance.',
                'required' => $price,
                'current' => $balance
            ], 400);
        }

        DB::beginTransaction();
        try {
            // Deduct credits
            $walletTransaction = $this->walletService->deductProviderWallet(
                $user->id,
                $price,
                0, // No paradigm_payment_id for marketplace purchases
                "Marketplace Profile Purchase - {$result->name} {$result->surname}"
            );

            if (!$walletTransaction) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to process payment.'
                ], 500);
            }

            // Create purchase record
            $purchase = ParadigmMarketplacePurchase::create([
                'result_id' => $resultId,
                'purchaser_id' => $user->id,
                'purchase_price' => $price,
                'purchased_at' => now(),
                'wallet_transaction_id' => $walletTransaction->id,
                'is_sold' => true,
                'linked_practitioner_id' => $user->id,
            ]);

            // Find client from payment
            $payment = ParadigmPayment::where('template_id', $result->template_id)
                ->where('provider_id', $result->provider_id)
                ->where(function($q) use ($result) {
                    $q->where('client_email', $result->email);
                    if ($result->name && $result->surname) {
                        $q->orWhere(function($p) use ($result) {
                            $p->where('client_name', $result->name)
                              ->where('client_surname', $result->surname);
                        });
                    }
                })
                ->where('self_purchased', 1)
                ->where('created_at', '<=', $result->created_at)
                ->orderBy('created_at', 'desc')
                ->first();

            // Create link request if client found
            if ($payment && $payment->client_id) {
                ParadigmLinkRequest::create([
                    'result_id' => $resultId,
                    'marketplace_purchase_id' => $purchase->id,
                    'practitioner_id' => $user->id,
                    'client_id' => $payment->client_id,
                    'status' => 'pending',
                    'requested_at' => now(),
                ]);
            }

            // Update purchase count
            $result->increment('marketplace_purchase_count');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Profile purchased successfully!',
                'purchase_id' => $purchase->id,
                'redirect' => route('provider.paradigm.marketplace.my-purchases')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Marketplace purchase failed', [
                'user_id' => $user->id,
                'result_id' => $resultId,
                'error' => $e->getMessage()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Purchase failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download purchased profile PDF
     */
    public function download($resultId)
    {
        $user = Auth::user();
        
        // Check if user purchased this profile
        $purchase = ParadigmMarketplacePurchase::where('result_id', $resultId)
            ->where(function($q) use ($user) {
                $q->where('purchaser_id', $user->id)
                  ->orWhere('linked_practitioner_id', $user->id);
            })
            ->where('is_sold', true)
            ->first();

        if (!$purchase) {
            abort(403, 'You do not have access to this profile.');
        }

        $result = ParadigmAssessmentResult::with(['template.pdfPages.pdfElements'])->findOrFail($resultId);
        
        return $this->pdfService->generatePdf($result);
    }

    /**
     * List purchased profiles
     */
    public function myPurchases()
    {
        $user = Auth::user();
        
        $purchases = ParadigmMarketplacePurchase::with(['result.template', 'linkRequest'])
            ->where(function($q) use ($user) {
                $q->where('purchaser_id', $user->id)
                  ->orWhere('linked_practitioner_id', $user->id);
            })
            ->where('is_sold', true)
            ->orderBy('purchased_at', 'desc')
            ->paginate(20);

        return view('provider.paradigm.marketplace.my-purchases', compact('purchases'));
    }

    /**
     * Show transfer form
     */
    public function transferProfile($purchaseId)
    {
        $user = Auth::user();
        $purchase = ParadigmMarketplacePurchase::with(['result', 'linkedPractitioner'])->findOrFail($purchaseId);

        // Check ownership
        if ($purchase->linked_practitioner_id != $user->id) {
            abort(403, 'You do not own this profile.');
        }

        return view('provider.paradigm.marketplace.transfer', compact('purchase'));
    }

    /**
     * Process transfer
     */
    public function processTransfer(Request $request, $purchaseId)
    {
        $user = Auth::user();
        $purchase = ParadigmMarketplacePurchase::findOrFail($purchaseId);

        // Check ownership
        if ($purchase->linked_practitioner_id != $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'You do not own this profile.'
            ], 403);
        }

        $request->validate([
            'practitioner_email' => 'required|email|exists:users,email',
            'transfer_price' => 'required|numeric|min:0.01',
        ]);

        $newPractitioner = User::where('email', $request->practitioner_email)->first();
        
        // Verify new practitioner is a practitioner
        if ($newPractitioner->user_type != 2 || !$newPractitioner->category || $newPractitioner->category->slug !== 'practitioner') {
            return response()->json([
                'success' => false,
                'message' => 'The specified user is not a practitioner.'
            ], 400);
        }

        $transferPrice = (float) $request->transfer_price;

        // Check new practitioner wallet balance
        if (!$this->walletService->hasSufficientBalance($newPractitioner->id, $transferPrice)) {
            return response()->json([
                'success' => false,
                'message' => 'New practitioner has insufficient wallet balance.'
            ], 400);
        }

        DB::beginTransaction();
        try {
            // Deduct from new practitioner
            $walletTransactionTo = $this->walletService->deductProviderWallet(
                $newPractitioner->id,
                $transferPrice,
                0,
                "Profile Transfer - {$purchase->result->name} {$purchase->result->surname}"
            );

            if (!$walletTransactionTo) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to deduct credits from new practitioner.'
                ], 500);
            }

            // Credit old practitioner (use refundAssessment which adds credits)
            $walletTransactionFrom = $this->walletService->refundAssessment(
                $user->id,
                $transferPrice,
                0,
                "Profile Transfer Sale - {$purchase->result->name} {$purchase->result->surname}"
            );

            if (!$walletTransactionFrom) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to credit credits to your wallet.'
                ], 500);
            }

            // Create transfer record
            $transfer = \App\Models\ParadigmPractitionerTransfer::create([
                'marketplace_purchase_id' => $purchaseId,
                'from_practitioner_id' => $user->id,
                'to_practitioner_id' => $newPractitioner->id,
                'transfer_price' => $transferPrice,
                'transfer_status' => 'completed',
                'wallet_transaction_id_from' => $walletTransactionFrom->id,
                'wallet_transaction_id_to' => $walletTransactionTo->id,
                'transferred_at' => now(),
            ]);

            // Update purchase ownership
            $purchase->update([
                'linked_practitioner_id' => $newPractitioner->id,
            ]);

            // Update link request if exists
            $linkRequest = ParadigmLinkRequest::where('marketplace_purchase_id', $purchaseId)->first();
            if ($linkRequest) {
                $linkRequest->update([
                    'practitioner_id' => $newPractitioner->id,
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Profile transferred successfully!',
                'redirect' => route('provider.paradigm.marketplace.my-purchases')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Profile transfer failed', [
                'purchase_id' => $purchaseId,
                'error' => $e->getMessage()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Transfer failed: ' . $e->getMessage()
            ], 500);
        }
    }
}

