<?php

namespace App\Http\Controllers;
use App\Repositories\Contracts\ProviderRepositoryInterface;
use App\Repositories\Eloquent\WalletRepository;
use App\Services\ParadigmWalletService;
use Modules\GlobalSetting\Entities\GlobalSetting;
use App\Models\PayoutHistory;
use App\Models\User;
use App\Models\Bookings;
use Modules\GlobalSetting\app\Models\Language;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ProviderController extends Controller
{
    protected ProviderRepositoryInterface $providerRepository;

    public function __construct(ProviderRepositoryInterface $providerRepository)
    {
        $this->providerRepository = $providerRepository;
    }

    public function index()
    {
        $response = $this->providerRepository->index();
        return view('people.list', $response);
    }

    public function getsubscription(Request $request)
    {
        $response = $this->providerRepository->getsubscription($request);
        return response()->json($response, $response['code']);
    }

    public function gettotalbookingcount(Request $request)
    {
        $response = $this->providerRepository->gettotalbookingcount($request);
        return response()->json($response, $response['code']);
    }

    function calculateProviderBalance(int $providerId): float
    {
        $commissionRate = 0;
        $commissionSetting = GlobalSetting::where('key', 'commission_rate_percentage')->first();
        if ($commissionSetting) {
            $commissionRate = (float) $commissionSetting->value;
        }

        $transactions = Bookings::with(['product'])
            ->where('booking_status', 6)
            ->whereHas('product', function ($query) use ($providerId) {
                $query->where('created_by', $providerId);
            })->get();

        $totalGrossAmount = 0;
        $totalCommission = 0;
        $totalReducedAmount = 0;
        $remainingAmount = 0;

        if ($transactions) {
            foreach ($transactions as $booking) {
                $grossAmount = $booking->total_amount ?? 0;

                $commissionAmount = ($grossAmount * $commissionRate) / 100;
                $reducedAmount = $grossAmount - $commissionAmount;

                $totalGrossAmount += $grossAmount;
                $totalCommission += $commissionAmount;
                $totalReducedAmount += $reducedAmount;
            }

            $enteredAmount = PayoutHistory::where('user_id', $providerId)->sum('process_amount');
            $remainingAmount = $totalReducedAmount - $enteredAmount;
        }
        return number_format($remainingAmount, 2, '.', '');
    }

    public function gettotalbookingcountapi(Request $request)
    {
        $response = $this->providerRepository->gettotalbookingcountapi($request);
        return response()->json($response, $response['code']);
    }

    public function getlatestbookingsapi(Request $request)
    {
        $response = $this->providerRepository->getlatestbookingsapi($request);
        return response()->json($response, $response['code']);
    }

    public function getlatestbookings(Request $request)
    {
        $response = $this->providerRepository->getlatestbookings($request);
        return response()->json($response, $response['code']);
    }

    public function getlatestreviews(Request $request)
    {
        $response = $this->providerRepository->getlatestreviews($request);
        return response()->json($response, $response['code']);
    }

    public function getlatestreviewsapi(Request $request)
    {
        $response = $this->providerRepository->getlatestreviewsapi($request);
        return response()->json($response, $response['code']);
    }

    public function getlatestproductservice(Request $request)
    {
        $response = $this->providerRepository->getlatestproductservice($request);
        return response()->json($response, $response['code']);
    }

    public function getsubscribedpack(Request $request)
    {
        $response = $this->providerRepository->getsubscribedpack($request);
        return response()->json($response, $response['code']);
    }

    public function getsubscribedpackapi(Request $request)
    {
        $response = $this->providerRepository->getsubscribedpackapi($request);
        return response()->json($response, $response['code']);
    }

    public function providerCalendarIndex(Request $request)
    {
        $response = $this->providerRepository->providerCalendarIndex($request);
        return view('provider.booking.calendarnew', $response);
    }

    public function getstafflist()
    {
        $users = User::where('user_type', 4)->get();
        return response()->json($users);
    }

    public function providergetBookingsapi(Request $request)
    {
        $response = $this->providerRepository->providergetBookingsapi($request);
        return response()->json($response);
    }

    public function providergetBookings(Request $request)
    {
        $response = $this->providerRepository->providergetBookings($request);
        return response()->json($response);
    }

    public function providergetBookApi(Request $request)
    {
        $response = $this->providerRepository->providergetBookApi($request);
        return response()->json($response, $response['code']);
    }

    public function getlanguage(): mixed
    {
        $languages = Language::select('id', 'code')
            ->where('status', 1)
            ->get();
        if (Auth::check()) {
            $language_id = User::select('user_language_id')->where('id', Auth::id())->first();
            if ($language_id) {
                $language_id = $language_id->user_language_id;
            } else {
                $defaultLanguage = $languages->firstWhere('is_default', 1);
                $language_id = $defaultLanguage ? $defaultLanguage->id : null;
            }
        } elseif (Cookie::get('languageId')) {
            $language_id = Cookie::get('languageId');
        } else {
            $defaultLanguage = $languages->firstWhere('is_default', 1);
            $language_id = $defaultLanguage ? $defaultLanguage->id : null;
        }
        return $language_id;
    }

    public function getStaffDetails(Request $request)
    {
        $response = $this->providerRepository->getStaffDetails($request);
        return response()->json($response);
    }

    public function getStaffDetailsApi(Request $request)
    {
        $response = $this->providerRepository->getStaffDetailsApi($request);
        return response()->json($response, $response['code']);
    }

    function mapDateFormatToSQL($phpFormat)
    {
        $replacements = [
            'd' => '%d',
            'D' => '%a',
            'j' => '%e',
            'l' => '%W',
            'F' => '%M',
            'm' => '%m',
            'M' => '%b',
            'n' => '%c',
            'Y' => '%Y',
            'y' => '%y',
        ];

        return strtr($phpFormat, $replacements);
    }

    protected function mapTimeFormatToSQL($timeFormat)
    {
        $map = [
            'hh:mm A' => '%h:%i %p', // 12-hour format with AM/PM
            'hh:mm a' => '%h:%i %p', // Same as above
            'HH:mm'   => '%H:%i',    // 24-hour format
        ];

        return $map[$timeFormat] ?? '%H:%i'; // Default to 24-hour format
    }

    public function getBranchStaff(Request $request)
    {
        $response = $this->providerRepository->getBranchStaff($request);
        return response()->json($response);
    }

    public function getCustomer(Request $request)
    {
        $response = $this->providerRepository->getCustomer($request);
        return response()->json($response);
    }

    public function fetchStaffService(Request $request)
    {
        $response = $this->providerRepository->fetchStaffService($request);
        return response()->json($response);
    }

    public function providerCalenderBooking(Request $request)
    {
        $response = $this->providerRepository->providerCalenderBooking($request);
        return response()->json($response, $response['code']);
    }

    public function providerCalenderBookingApi(Request $request)
    {
        $response = $this->providerRepository->providerCalenderBookingApi($request);
        return response()->json($response, $response['code']);
    }

    public function getUserList(Request $request)
    {
        $response = $this->providerRepository->getUserList($request);
        return response()->json($response, $response['code']);
    }

    public function getServiceList(Request $request)
    {
        $response = $this->providerRepository->getServiceList($request);
        return response()->json($response, $response['code']);
    }

    public function getBranchList(Request $request)
    {
        $response = $this->providerRepository->getBranchList($request);
        return response()->json($response, $response['code']);
    }

    public function getStaffLists(Request $request)
    {
        $response = $this->providerRepository->getStaffLists($request);
        return response()->json($response, $response['code']);
    }

    /**
     * Display provider wallet page
     */
    public function wallet()
    {
        return view('provider.wallet');
    }

    /**
     * Get wallet data for provider
     * Uses ParadigmWalletService (safe wrapper for existing wallet system)
     */
    public function walletData()
    {
        try {
            $userId = Auth::id();
            
            // If user is staff (user_type = 4), use parent provider's ID for wallet data
            $user = Auth::user();
            if ($user && $user->user_type == 4) {
                $userDetail = $user->userDetail;
                if ($userDetail && $userDetail->parent_id) {
                    $userId = $userDetail->parent_id;
                }
            }
            
            $walletRepository = new WalletRepository();

            // Get wallet balance using the repository
            $balanceData = $walletRepository->getWalletBalance($userId);

            // Get all wallet history (both credits and debits)
            $allHistory = $walletRepository->getAllWalletHistory($userId);

            // Get currency
            $currency = $walletRepository->getCurrencySymbol();

            return response()->json([
                'success' => true,
                'balance' => $balanceData['balance'],
                'totalCredit' => $balanceData['total'],
                'totalDebit' => $balanceData['debit'],
                'currency' => $currency ? $currency->symbol : '$',
                'transactions' => $allHistory
            ]);
        } catch (\Exception $e) {
            \Log::error("Provider walletData failed: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Could not load wallet data. ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display wallet top-up page
     */
    public function walletTopup()
    {
        return view('provider.wallet-topup');
    }

    /**
     * Get completed client assessments for provider - shows unique clients only
     */
    public function getClientLink()
    {
        try {
            $providerId = auth()->id();
            $user = auth()->user();
            
            // If user is staff (user_type = 4), use parent provider's ID for paradigm profiles
            if ($user->user_type == 4) {
                $userDetail = $user->userDetail;
                if ($userDetail && $userDetail->parent_id) {
                    $providerId = $userDetail->parent_id;
                }
            }
            
            // SECURITY FIX: Verify user is a provider or staff (user_type 1, 2, or 4)
            if (!in_array($user->user_type, [1, 2, 4])) {
                abort(403, 'Unauthorized - Provider access only');
            }
            
            // Get unique client emails who have completed assessments for this provider
            $uniqueClientEmails = \App\Models\ParadigmAssessmentResult::where('provider_id', $providerId)
                ->whereNotNull('paradigm_type') // Only completed assessments have paradigm_type
                ->whereNotNull('email')
                ->distinct()
                ->pluck('email')
                ->toArray();
            
            $uniqueClients = [];
            
            foreach ($uniqueClientEmails as $clientEmail) {
                // Get the most recent completed assessment for this client
                $latestResult = \App\Models\ParadigmAssessmentResult::with(['template'])
                    ->where('provider_id', $providerId)
                    ->where('email', $clientEmail)
                    ->whereNotNull('paradigm_type')
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($latestResult) {
                    $uniqueClients[] = [
                        'result_id' => $latestResult->id,
                        'client_name' => $latestResult->name ?? 'N/A',
                        'client_email' => $latestResult->email,
                        'template_name' => $latestResult->template->name ?? 'N/A',
                        'paradigm_type' => $latestResult->paradigm_type,
                        'completed_at' => $latestResult->created_at->format('Y-m-d H:i'),
                        'client_id' => $latestResult->client_id // Add client_id for unlinking
                    ];
                }
            }
            
            return response()->json($uniqueClients);
        } catch (\Exception $e) {
            \Log::error('Failed to load client assessments: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load client assessments'
            ], 500);
        }
    }

    /**
     * Revoke assessment assignment (refund if not completed)
     */
    public function revokeAssignment(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:paradigm_payments,id'
        ]);

        try {
            $userId = auth()->id();
            $user = auth()->user();
            $isAdmin = $user && $user->user_type == 1;
            
            // Determine provider ID (handle staff with parent_id)
            $providerId = $userId;
            if ($user && $user->user_type == 4) {
                $userDetail = $user->userDetail;
                if ($userDetail && $userDetail->parent_id) {
                    $providerId = $userDetail->parent_id;
                }
            }
            
            // Admin can revoke any payment, providers can only revoke their own
            $paymentQuery = \App\Models\ParadigmPayment::where('id', $request->payment_id);
            if (!$isAdmin) {
                $paymentQuery->where('provider_id', $providerId);
            }
            $payment = $paymentQuery->firstOrFail();

            // For non-admins, only allow revoke if paid by provider
            if (!$isAdmin && $payment->paid_by !== 'provider') {
                return response()->json([
                    'success' => false,
                    'message' => __('Only provider-paid assessments can be revoked')
                ], 400);
            }
            
            // Check if paradigm profile is completed
            $result = \App\Models\ParadigmAssessmentResult::where('email', $payment->client_email)
                ->where('template_id', $payment->template_id)
                ->where('provider_id', $payment->provider_id)
                ->where('created_at', '>=', $payment->created_at)
                ->whereNotNull('paradigm_type') // Completed profiles have paradigm_type
                ->first();
            
            if ($result) {
                return response()->json([
                    'success' => false,
                    'message' => __('Cannot revoke: Paradigm profile already completed (non-refundable)')
                ], 400);
            }
            
            $clientEmail = $payment->client_email;
            $paymentProviderId = $payment->provider_id;
            
            // Get the actual amount from template pricing (since amount is 0 when paid_by is 'provider')
            $amount = $payment->amount;
            if ($payment->paid_by === 'provider' && $amount == 0) {
                // Get amount from template pricing
                $pricing = \App\Models\ParadigmPricing::where('template_id', $payment->template_id)
                    ->where('is_active', true)
                    ->first();
                $amount = $pricing ? $pricing->price : 0;
            }
            
            // Refund credits to provider wallet (only if paid by provider and amount > 0)
            if ($payment->paid_by === 'provider' && $amount > 0) {
                $walletService = new \App\Services\ParadigmWalletService();
                $refund = $walletService->refundAssessment(
                    $paymentProviderId,
                    $amount,
                    $payment->id,
                    "Revoked assessment for {$clientEmail}" . ($isAdmin ? ' (by admin)' : '')
                );
                
                if (!$refund) {
                    return response()->json([
                        'success' => false,
                        'message' => __('Failed to process refund')
                    ], 500);
                }
            }
            
            // Delete the payment (cascade will handle access token deletion)
            $payment->delete();
            
            // Log the revocation
            \Log::info(($isAdmin ? 'Admin' : 'Provider') . ' revoked paradigm assessment assignment', [
                'payment_id' => $request->payment_id,
                'client_email' => $clientEmail,
                'provider_id' => $paymentProviderId,
                'revoked_by' => $userId,
                'revoked_by_type' => $isAdmin ? 'admin' : 'provider',
                'refunded_amount' => $payment->paid_by === 'provider' ? $amount : 0,
                'revoked_at' => now()
            ]);
            
            $message = $isAdmin 
                ? __('Assessment revoked successfully by admin')
                : __('Assessment revoked and credits refunded successfully');
            
            return response()->json([
                'success' => true,
                'message' => $message
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to revoke paradigm assessment assignment', [
                'payment_id' => $request->payment_id,
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to revoke assessment assignment')
            ], 500);
        }
    }
    
    /**
     * Delete a single assessment assignment
     */
    public function deleteAssignment(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:paradigm_payments,id'
        ]);

        try {
            $providerId = auth()->id();
            $payment = \App\Models\ParadigmPayment::where('id', $request->payment_id)
                ->where('provider_id', $providerId)
                ->firstOrFail();

            $clientEmail = $payment->client_email;
            
            // Delete the payment (cascade will handle related data)
            $payment->delete();
            
            // Log the deletion
            \Log::info('Provider deleted paradigm assessment assignment', [
                'payment_id' => $request->payment_id,
                'client_email' => $clientEmail,
                'provider_id' => $providerId,
                'deleted_at' => now()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => __('Assessment assignment deleted successfully')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to delete paradigm assessment assignment', [
                'payment_id' => $request->payment_id,
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to delete assessment assignment')
            ], 500);
        }
    }
    
    /**
     * Delete a single client link (assessment result)
     */
    public function deleteClientLink(Request $request)
    {
        $request->validate([
            'result_id' => 'required|exists:paradigm_assessment_results,id'
        ]);

        try {
            $providerId = auth()->id();
            $result = \App\Models\ParadigmAssessmentResult::where('id', $request->result_id)
                ->where('provider_id', $providerId)
                ->firstOrFail();

            $clientName = $result->name ?? 'Unknown';
            
            // Delete the result (cascade will handle related data)
            $result->delete();
            
            // Log the deletion
            \Log::info('Provider deleted paradigm client result', [
                'result_id' => $request->result_id,
                'client_name' => $clientName,
                'provider_id' => $providerId,
                'deleted_at' => now()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => __('Client result deleted successfully')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to delete paradigm client result', [
                'result_id' => $request->result_id,
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to delete client result')
            ], 500);
        }
    }
    
    /**
     * Delete all assessment assignments for provider
     */
    public function deleteAllAssignments(Request $request)
    {
        try {
            $providerId = auth()->id();
            $count = \App\Models\ParadigmPayment::where('provider_id', $providerId)->count();
            
            if ($count === 0) {
                return response()->json([
                    'success' => false,
                    'message' => __('No assessment assignments to delete')
                ]);
            }
            
            // Log before deletion
            \Log::info('Provider deleting all paradigm assessment assignments', [
                'count' => $count,
                'provider_id' => $providerId,
                'deleted_at' => now()
            ]);
            
            // Delete all assignments for this provider
            \App\Models\ParadigmPayment::where('provider_id', $providerId)->delete();
            
            return response()->json([
                'success' => true,
                'message' => __('All assessment assignments deleted successfully')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to delete all paradigm assessment assignments', [
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to delete all assessment assignments')
            ], 500);
        }
    }
    
    /**
     * Delete all client links (assessment results) for provider
     */
    public function deleteAllClientLinks(Request $request)
    {
        try {
            $providerId = auth()->id();
            $count = \App\Models\ParadigmAssessmentResult::where('provider_id', $providerId)->count();
            
            if ($count === 0) {
                return response()->json([
                    'success' => false,
                    'message' => __('No client results to delete')
                ]);
            }
            
            // Log before deletion
            \Log::info('Provider deleting all paradigm client results', [
                'count' => $count,
                'provider_id' => $providerId,
                'deleted_at' => now()
            ]);
            
            // Delete all client results for this provider
            \App\Models\ParadigmAssessmentResult::where('provider_id', $providerId)->delete();
            
            return response()->json([
                'success' => true,
                'message' => __('All client results deleted successfully')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to delete all paradigm client results', [
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to delete all client results')
            ], 500);
        }
    }
    
    /**
     * Unlink a specific client from provider (remove all their assessment results)
     */
    public function unlinkClient(Request $request)
    {
        $request->validate([
            'client_email' => 'required|email'
        ]);
        
        try {
            $providerId = auth()->id();
            $clientEmail = $request->client_email;
            
            // Get client details for logging
            $clientResult = \App\Models\ParadigmAssessmentResult::where('provider_id', $providerId)
                ->where('email', $clientEmail)
                ->first();
            
            $clientName = $clientResult ? $clientResult->name : 'Unknown';
            $resultCount = \App\Models\ParadigmAssessmentResult::where('provider_id', $providerId)
                ->where('email', $clientEmail)
                ->count();
            
            // Delete all assessment results for this client from this provider
            \App\Models\ParadigmAssessmentResult::where('provider_id', $providerId)
                ->where('email', $clientEmail)
                ->delete();
            
            // Also delete any payment assignments for this client
            \App\Models\ParadigmPayment::where('provider_id', $providerId)
                ->where('client_email', $clientEmail)
                ->delete();
            
            // Delete access tokens for this client
            \App\Models\ParadigmAccessToken::where('provider_id', $providerId)
                ->where('client_email', $clientEmail)
                ->delete();
            
            // Log the unlinking
            \Log::info('Provider unlinked client', [
                'provider_id' => $providerId,
                'client_email' => $clientEmail,
                'client_name' => $clientName,
                'deleted_results' => $resultCount,
                'unlinked_at' => now()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => "Successfully unlinked client {$clientName}. All assessment results and assignments have been removed."
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to unlink client', [
                'provider_id' => auth()->id(),
                'client_email' => $request->client_email,
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Failed to unlink client')
            ], 500);
        }
    }

    public function addWalletAmount(Request $request)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:1',
                'paymentMethod' => 'required|in:stripe,payfast,paypal',
                'userId' => 'required|integer'
            ]);
            
            // If user is staff (user_type = 4), use parent provider's wallet
            $userId = $validated['userId'];
            $user = \App\Models\User::find($userId);
            if ($user && $user->user_type == 4) {
                $userDetail = $user->userDetail;
                if ($userDetail && $userDetail->parent_id) {
                    $userId = $userDetail->parent_id; // Top up parent provider's wallet
                }
            }

            // Map payment methods to database enum values
            $paymentTypeMap = [
                'stripe' => 'stripe',
                'payfast' => 'payfast',
                'paypal' => 'Paypal'
            ];

            $dbPaymentType = $paymentTypeMap[$validated['paymentMethod']];
            
            // Generate unique transaction ID
            $transactionId = 'TXN' . strtoupper(uniqid());
            
            // Use WalletRepository (same as client wallet - NO CHANGES to existing system)
            $walletRepository = app(\App\Repositories\Contracts\WalletInterface::class);
            $walletHistory = $walletRepository->addWalletAmount([
                'user_id' => $userId, // Use parent provider's ID if staff
                'amount' => $validated['amount'], // Price in ZAR
                'credits' => $validated['credits'] ?? $validated['amount'], // Number of credits
                'payment_method' => $dbPaymentType,
                'transaction_id' => $transactionId,
                'status' => 'Pending',
                'type' => 1, // Credit
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Wallet record created successfully',
                'data' => $walletHistory,
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating wallet record',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function walletPaymentSuccess(Request $request)
    {
        // Confirm the transaction (same as client wallet)
        if ($request->has('token')) {
            $walletRepository = app(\App\Repositories\Contracts\WalletInterface::class);
            $walletRepository->confirmTransaction($request->token, 'Completed');
        }
        
        return view('provider.wallet-payment-success');
    }

}
