<?php

namespace App\Http\Controllers;

use App\Models\ParadigmPayment;
use App\Models\ParadigmAccessToken;
use App\Services\ParadigmWalletService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class UserParadigmController extends Controller
{
    protected $walletService;

    public function __construct()
    {
        $this->walletService = new ParadigmWalletService();
    }

    /**
     * Show Buy Assessment page
     * Credits amount comes from /admin/paradigm/sales-settings (CreditPackage with is_paradigm_package = true)
     */
    public function buyAssessment()
    {
        // Get the Paradigm Profile credit package from sales-settings
        $package = \App\Models\CreditPackage::where('is_paradigm_package', true)
            ->where('is_active', true)
            ->first();
        
        if (!$package) {
            return redirect()->route('user.dashboard')
                ->with('error', 'Paradigm Profile credits package not configured. Please contact administrator.');
        }

        // Get template_id from paradigm_pricing (for display purposes)
        $pricing = DB::table('paradigm_pricing')->first();
        
        // Create a pricing-like object for the view (using credits from package)
        $pricingData = (object) [
            'price' => $package->credits, // Credits amount from sales-settings
            'template_id' => $pricing ? ($pricing->template_id ?? 1) : 1,
            'currency' => 'USD',
            'price_zar' => $package->final_price_zar // Admin-set price in ZAR
        ];

        // Get user's wallet balance
        $userId = Auth::id();
        $balanceData = $this->walletService->getDetailedBalance($userId);
        $currency = $this->walletService->getCurrencySymbol();

        return view('user.paradigm.buy-assessment', [
            'pricing' => $pricingData,
            'balance' => $balanceData['balance'],
            'currency' => $currency,
            'package' => $package // Pass package for admin-set price
        ]);
    }

    /**
     * Process self-purchase of assessment
     * Uses wallet credits to purchase
     * Credits amount comes from /admin/paradigm/sales-settings (CreditPackage with is_paradigm_package = true)
     */
    public function processPurchase(Request $request)
    {
        try {
            $userId = Auth::id();
            $user = Auth::user();

            // Get the Paradigm Profile credit package from sales-settings
            $package = \App\Models\CreditPackage::where('is_paradigm_package', true)
                ->where('is_active', true)
                ->first();
            
            if (!$package) {
                return response()->json([
                    'success' => false,
                    'message' => 'Paradigm Profile credits package not configured. Please contact administrator.'
                ], 400);
            }

            // Get the credits amount from the package (set in sales-settings)
            $amount = (float) $package->credits;
            
            // Get template_id from paradigm_pricing (for the assessment template)
            $pricing = DB::table('paradigm_pricing')->first();
            $templateId = $pricing ? ($pricing->template_id ?? 1) : 1;

            // Check wallet balance (only if price > 0)
            if ($amount > 0 && !$this->walletService->hasSufficientBalance($userId, $amount)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient wallet balance. Please top up your wallet first.',
                    'required' => $amount,
                    'current' => $this->walletService->getBalance($userId)
                ], 400);
            }

            DB::beginTransaction();

            try {
                // Get first admin user ID for self-purchased assessments
                $adminUser = \App\Models\User::where('user_type', 1)->first();
                $adminProviderId = $adminUser ? $adminUser->id : $userId;
                
                // Get user details for client_name and client_surname (required for assessment link)
                $userDetail = $user->userDetail;
                $clientName = $userDetail ? ($userDetail->first_name ?? '') : '';
                $clientSurname = $userDetail ? ($userDetail->last_name ?? '') : '';
                
                // If no userDetail, try to get from user name
                if (empty($clientName) && !empty($user->name)) {
                    $nameParts = explode(' ', $user->name, 2);
                    $clientName = $nameParts[0] ?? '';
                    $clientSurname = $nameParts[1] ?? '';
                }
                
                // Create paradigm_payments record (self_purchased = 1)
                $payment = new ParadigmPayment();
                $payment->provider_id = $adminProviderId; // Use ADMIN ID as provider for self-purchase
                $payment->client_email = $user->email;
                $payment->client_id = $userId; // Client is the actual user
                $payment->client_name = $clientName; // Required for assessment link
                $payment->client_surname = $clientSurname; // Required for assessment link
                $payment->client_gender = $userDetail ? ($userDetail->gender ?? null) : null;
                $payment->client_dob = $userDetail ? ($userDetail->dob ?? null) : null;
                $payment->client_whatsapp = $user->phone_number ?? null;
                $payment->template_id = $templateId;
                $payment->amount = $amount;
                $payment->currency = 'USD';
                $payment->status = $amount > 0 ? 'paid' : 'free';
                $payment->payment_link = ParadigmPayment::generatePaymentLink();
                $payment->paid_at = now();
                $payment->self_purchased = 1; // NEW: Marks as self-purchase
                $payment->save();

                // Create access token (same as existing system)
                $accessToken = ParadigmAccessToken::create([
                    'payment_id' => $payment->id,
                    'token' => Str::random(64),
                    'template_id' => $payment->template_id,
                    'provider_id' => $adminProviderId, // Use ADMIN ID as provider for self-purchase
                    'client_id' => $userId,
                    'status' => 'active',
                    'expires_at' => now()->addDays(30), // 30-day access
                ]);

                // Generate assessment link
                $assessmentLink = url('/paradigm/assessment-direct/' . $payment->payment_link);

                // Send email and WhatsApp notifications (same as provider flow)
                try {
                    // Get template name
                    $template = \App\Models\ParadigmFormTemplate::find($templateId);
                    $templateName = $template ? $template->name : 'Paradigm Profile';
                    
                    // Get admin user for provider details (self-purchased uses admin as provider)
                    $adminUser = \App\Models\User::where('user_type', 1)->first();
                    $adminProviderId = $adminUser ? $adminUser->id : null;
                    
                    // Send email and WhatsApp notifications (EXACT SAME as provider flow)
                    $this->sendParadigmInvitationNotifications(
                        $user->email,
                        $templateName,
                        'client', // paidBy - client purchased it themselves
                        $amount,
                        'USD',
                        $user->phone_number,
                        $adminProviderId,
                        $payment
                    );
                    
                    Log::info('Paradigm invitation notifications sent to user', [
                        'user_id' => $userId,
                        'email' => $user->email,
                        'phone' => $user->phone_number,
                        'assessment_link' => $assessmentLink
                    ]);
                } catch (\Exception $notificationEx) {
                    Log::warning('Failed to send paradigm invitation notifications: ' . $notificationEx->getMessage(), [
                        'user_id' => $userId,
                        'error' => $notificationEx->getTraceAsString()
                    ]);
                    // Don't fail the purchase if notifications fail
                }

                // Deduct from wallet only if price > 0
                if ($amount > 0) {
                    $walletDeduction = $this->walletService->deductClientWallet(
                        $userId,
                        $amount,
                        $payment->id,
                        'Self-purchased Paradigm Assessment'
                    );

                    if (!$walletDeduction) {
                        DB::rollBack();
                        return response()->json([
                            'success' => false,
                            'message' => 'Failed to deduct from wallet'
                        ], 500);
                    }
                }

                DB::commit();

                Log::info('User self-purchased Paradigm assessment', [
                    'user_id' => $userId,
                    'payment_id' => $payment->id,
                    'amount' => $amount
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Assessment purchased successfully! You can now take the test.',
                    'payment_id' => $payment->id,
                    'access_token' => $accessToken->token,
                    'payment_link' => $payment->payment_link,
                    'assessment_link' => $assessmentLink
                ]);

            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }

        } catch (\Exception $e) {
            Log::error('User Paradigm purchase failed', [
                'user_id' => Auth::id(),
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error processing purchase. Please try again.'
            ], 500);
        }
    }

    /**
     * Show user's self-purchased assessments
     */
    public function myPaidAssessments()
    {
        $userId = Auth::id();
        $user = Auth::user();

        // Get only self-purchased assessments (where user is both provider and client)
        $payments = ParadigmPayment::with(['template', 'accessTokens'])
            ->where('provider_id', $userId)
            ->where('client_id', $userId)
            ->where('self_purchased', 1) // Only self-purchased
            ->orderBy('created_at', 'desc')
            ->get();

        return view('user.paradigm.my-paid-assessments', [
            'payments' => $payments
        ]);
    }

    /**
     * Show page to buy credits for Paradigm Profile assessment
     * Uses credits amount from /admin/paradigm/sales-settings (CreditPackage with is_paradigm_package = true)
     */
    public function buyCredits()
    {
        // Get the Paradigm Profile credit package from sales-settings
        $package = \App\Models\CreditPackage::where('is_paradigm_package', true)
            ->where('is_active', true)
            ->first();
        
        if (!$package) {
            return redirect()->route('user.dashboard')
                ->with('error', 'Paradigm Profile credits package not configured. Please contact administrator.');
        }

        $requiredCredits = (float) $package->credits;
        
        // Get user's wallet balance
        $userId = Auth::id();
        $balanceData = $this->walletService->getDetailedBalance($userId);
        $currentBalance = $balanceData['balance'];
        
        // Check if user already has sufficient credits
        if ($currentBalance >= $requiredCredits) {
            // Redirect to buy-assessment page to purchase profile
            return redirect()->route('user.paradigm.buy')->with('info', 'You have sufficient credits. Proceeding to purchase your profile.');
        }

        return view('user.paradigm.buy-credits', [
            'package' => $package,
            'requiredCredits' => $requiredCredits,
            'currentBalance' => $currentBalance,
            'priceZAR' => $package->final_price_zar,
            'priceUSD' => $package->usd_equivalent,
            'currency' => $this->walletService->getCurrencySymbol()
        ]);
    }
    
    /**
     * Send email and WhatsApp notifications for Paradigm assessment invitation
     * Same logic as ParadigmPaymentController@sendAssignmentEmail
     */
    private function sendParadigmInvitationNotifications($clientEmail, $templateName, $paidBy, $amount = 0, $currency = 'USD', $clientWhatsApp = null, $overrideProviderId = null, $payment = null)
    {
        try {
            // Configure SMTP from admin settings (required for email to work)
            \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
            
            // Get the email template from database (Template 106 - Paradigm Invitation)
            $template = DB::table('templates')
                ->where('notification_type', 10)
                ->where('type', 1)
                ->where('status', 1)
                ->first();
            
            // Get company settings
            $settings = DB::table('general_settings')
                ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no', 'company_address'])
                ->pluck('value', 'key');
            
            $companyName = $settings['company_name'] ?? config('app.name', 'System');
            $companyWebsite = $settings['website'] ?? config('app.url', url('/'));
            $companyEmail = $settings['site_email'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyAddress = $settings['company_address'] ?? '';
            
            // Build placeholders
            $appName = $companyName;
            $appUrl = $companyWebsite;
            $loginUrl = $appUrl . '/login';
            $registerUrl = $appUrl;
            $dashboardUrl = $appUrl . '/user/paradigm/assessments';
            
            // Use direct access link if payment has client details, otherwise use old flow
            $baseUrl = 'https://app.iseekhelpoffice.com';
            
            if ($payment && $payment->client_name && $payment->client_surname) {
                $assessmentLink = $baseUrl . '/paradigm/assessment-direct/' . $payment->payment_link;
            } else {
                $assessmentLink = $baseUrl . '/paradigm/assessment'; // Link to start assessment (old flow)
            }
            
            // Generate a unique access token
            $accessToken = strtoupper(substr(md5($clientEmail . time()), 0, 12));
            
            // Get client name - from payment if available, otherwise from user, otherwise fallback
            $clientName = 'Client';
            if ($payment && $payment->client_name) {
                $clientName = trim($payment->client_name . ' ' . ($payment->client_surname ?? ''));
            }
            if (empty($clientName) || $clientName === 'Client') {
                $clientUser = \App\Models\User::where('email', $clientEmail)->first();
                if ($clientUser) {
                    if ($clientUser->userDetail) {
                        $firstName = $clientUser->userDetail->first_name ?? '';
                        $lastName = $clientUser->userDetail->last_name ?? '';
                        $clientName = trim($firstName . ' ' . $lastName) ?: ($clientUser->name ?? 'Client');
                    } else {
                        $clientName = $clientUser->name ?? 'Client';
                    }
                }
            }
            
            // Get provider name - use overrideProviderId (admin) for self-purchased, otherwise use company name
            $providerName = $companyName;
            if ($overrideProviderId) {
                $provider = \App\Models\User::find($overrideProviderId);
                if ($provider) {
                    if ($provider->userDetail) {
                        $providerFirstName = $provider->userDetail->first_name ?? '';
                        $providerLastName = $provider->userDetail->last_name ?? '';
                        $providerName = trim($providerFirstName . ' ' . $providerLastName) ?: ($provider->name ?? $companyName);
                    } else {
                        $providerName = $provider->name ?? $companyName;
                    }
                }
            }
            
            // Build payment info based on type
            $paymentInfo = '';
            if ($paidBy === 'free') {
                $paymentInfo = 'This assessment is free of charge.';
            } elseif ($paidBy === 'provider') {
                $paymentInfo = 'This assessment has been paid for by your practitioner.';
            } elseif ($paidBy === 'client') {
                $paymentInfo = "Payment of {$currency} {$amount} has been processed. Your assessment is ready to start.";
            }
            
            if ($template) {
                // All possible placeholders for Paradigm Invitation
                $placeholders = [
                    '{{client_name}}', '{{client_email}}', '{{template_name}}', 
                    '{{provider_name}}', '{{access_token}}', '{{assessment_link}}',
                    '{{payment_info}}', '{{login_url}}', '{{register_url}}', 
                    '{{dashboard_url}}', '{{app_name}}', '{{company_name}}',
                    '{{company_website}}', '{{company_email}}', '{{company_phone}}',
                    '{{company_address}}'
                ];
                
                $replacements = [
                    $clientName, $clientEmail, $templateName,
                    $providerName, $accessToken, $assessmentLink,
                    $paymentInfo, $loginUrl, $registerUrl,
                    $dashboardUrl, $appName, $companyName,
                    $companyWebsite, $companyEmail, $companyPhone,
                    $companyAddress
                ];
                
                $subject = str_replace($placeholders, $replacements, $template->subject);
                $emailContent = str_replace($placeholders, $replacements, $template->content);
                
                // Use Samplemail class
                $data = [
                    'subject' => $subject,
                    'message' => $emailContent,
                    'attachment' => null
                ];
                
                \Illuminate\Support\Facades\Mail::to($clientEmail)->send(new \Modules\Communication\app\Mail\Samplemail($data));
            }
            
            // Send Paradigm invitation WhatsApp
            try {
                $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
                
                // Use provided WhatsApp number if available, otherwise try to get from registered user
                $phoneNumber = $clientWhatsApp;
                
                if (empty($phoneNumber)) {
                    // Fallback: try to get phone from registered user
                    $clientUser = \App\Models\User::where('email', $clientEmail)->first();
                    $phoneNumber = $clientUser->phone_number ?? null;
                }
                
                if (!empty($phoneNumber)) {
                    // Generate assessment link for CLIENT WhatsApp
                    $assessmentLink = null;
                    $baseUrl = 'https://app.iseekhelpoffice.com';
                    if ($payment && $payment->payment_link) {
                        if ($payment->client_name && $payment->client_surname) {
                            $assessmentLink = $baseUrl . '/paradigm/assessment-direct/' . $payment->payment_link;
                        } else {
                            $assessmentLink = $baseUrl . '/paradigm/assessment';
                        }
                    }
                    
                    // Send CLIENT-specific WhatsApp template
                    if ($assessmentLink) {
                        $smsRepository->sendParadigmInvitationClientWhatsApp(
                            $phoneNumber,
                            $clientName,
                            $assessmentLink
                        );
                    }
                }
            } catch (\Exception $whatsappEx) {
                Log::warning('Failed to send paradigm invitation WhatsApp: ' . $whatsappEx->getMessage());
            }
            
        } catch (\Exception $e) {
            // Log error but don't fail the assignment
            Log::error('Failed to send assignment email: ' . $e->getMessage());
        }
    }
}

