<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;
use Modules\GlobalSetting\Entities\GlobalSetting;

class CheckProviderVerification
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = Auth::user();
        
        // Only apply to provider users (user_type = 2)
        if ($user && $user->user_type == 2) {
            $providerApprovalStatus = GlobalSetting::where('key', 'provider_approval_status')->value('value') ?? 0;
            $idVerificationEnabled = GlobalSetting::where('key', 'provider_id_verification_enabled')->value('value') ?? 0;
            
            // Check Provider Approval system
            if ($providerApprovalStatus == 1 && $user->provider_verified_status == 0) {
                return redirect()->route('provider.identity-verification')
                    ->with('verification_message', __('Your account is not verified yet. Please complete identity verification and wait for admin confirmation.'));
            }
            
            // Check ID Verification system
            if ($idVerificationEnabled == 1) {
                $identityVerification = $user->identityVerification;
                
                // Allow access to identity verification page
                if ($request->routeIs('provider.identity-verification')) {
                    return $next($request);
                }
                
                // If no identity verification record exists, redirect to verification
                if (!$identityVerification) {
                    return redirect()->route('provider.identity-verification')
                        ->with('verification_message', __('Please complete your identity verification to access the dashboard.'));
                }
                
                // If identity verification is pending, block access completely
                if ($identityVerification->verification_status === 'pending') {
                    return response()->view('provider.verification-pending', [], 403);
                }
                
                // If identity verification is rejected, redirect to verification page
                if ($identityVerification->verification_status === 'rejected') {
                    return redirect()->route('provider.identity-verification')
                        ->with('verification_message', __('Your identity verification was rejected. Please resubmit your documents.') . ($identityVerification->rejection_reason ? ' Reason: ' . $identityVerification->rejection_reason : ''));
                }
            }
        }
        
        return $next($request);
    }
}
