<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class ProviderRegistrationRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        // If only requesting OTP, minimal validation
        if ($this->has('otp_only') && $this->get('otp_only')) {
            return [
                'provider_email' => 'required|email|max:100',
            ];
        }

        // Check if registration requires OTP (register status = 1)
        $regStatus = \DB::table('general_settings')->where('key', 'register')->value('value');
        
        // If OTP is enabled, relax some validations for initial submission
        if ($regStatus === "1") {
            return [
                'category_id' => 'required|integer|exists:categories,id',
                'subcategory_ids' => 'required|array|min:1',
                'subcategory_ids.*' => 'integer|exists:categories,id',
                'provider_first_name' => 'required',
                'provider_last_name' => 'required',
                'provider_name' => [
                    'required',
                    Rule::unique('users', 'name')
                ],
                'provider_email' => [
                    'required',
                    'email',
                    Rule::unique('users', 'email')
                ],
                'provider_phone_number' => 'required|string|min:8',
                'provider_password' => 'required',
                'provider_dob' => 'required|date|before:today',
                'provider_gender' => 'required|in:male,female',
                'provider_bio' => 'nullable|string|max:3000',
                'provider_profile_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
                'provider_address' => 'required|string|max:150',
                'provider_city' => 'required|integer',
                'provider_state' => 'required|integer',
                'provider_country' => 'required|integer',
                'provider_postal_code' => 'required|string|max:10',
                'provider_language' => 'nullable|array',
                'provider_language.*' => 'integer|exists:languages,id',
                'provider_terms_policy' => 'sometimes|accepted',
            ];
        }

        $rules = [
            'category_id' => 'required|integer|exists:categories,id',
            'subcategory_ids' => 'required|array|min:1',
            'subcategory_ids.*' => 'integer|exists:categories,id',
            'provider_first_name' => 'required',
            'provider_last_name' => 'required',
            'provider_name' => [
                'required',
                Rule::unique('users', 'name')
            ],
            'provider_email' => [
                'required',
                'email',
                Rule::unique('users', 'email')
            ],
            'provider_phone_number' => 'required|string|min:8',
            'provider_password' => 'required',
            'provider_dob' => 'required|date|before:today',
            'provider_gender' => 'required|in:male,female',
            'provider_bio' => 'nullable|string|max:3000',
            'provider_profile_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'provider_address' => 'required|string|max:150',
            'provider_city' => 'required|integer',
            'provider_state' => 'required|integer',
            'provider_country' => 'required|integer',
            'provider_postal_code' => 'required|string|max:10',
            'provider_language' => 'nullable|array',
            'provider_language.*' => 'integer|exists:languages,id',
        ];

        if ($this->has('is_mobile') && $this->get('is_mobile') === "yes") {
            $rules['provider_terms_policy'] = 'required|accepted';
            $rules['company_website'] = 'nullable|url';
        }

        return $rules;
    }

    public function messages(): array
    {
        return [
            'provider_name.required' => 'Name is required.',
            'provider_email.required' => 'Email is required.',
            'provider_email.email' => 'Enter a valid email address.',
            'provider_email.unique' => 'This email is already registered.',
            'provider_password.required' => 'Password is required.',
            'provider_phone_number.required' => 'Phone Number is required.',
            'provider_terms_policy.required' => 'You must accept the terms and policy.',
            'company_website.url' => 'Enter a valid website URL.',
            'provider_dob.required' => 'Date of birth is required.',
            'provider_dob.date' => 'Enter a valid date of birth.',
            'provider_dob.before' => 'Date of birth must be before today.',
            'provider_gender.required' => 'Gender is required.',
            'provider_gender.in' => 'Please select a valid gender.',
            'provider_bio.max' => 'Bio cannot exceed 3000 characters.',
            'category_id.required' => 'Category is required.',
            'subcategory_ids.required' => 'Please select at least one subcategory.',
            'subcategory_ids.array' => 'Subcategory must be an array.',
            'subcategory_ids.min' => 'Please select at least one subcategory.',
            'provider_address.required' => 'Address is required.',
            'provider_address.max' => 'Address cannot exceed 150 characters.',
            'provider_city.required' => 'City is required.',
            'provider_city.max' => 'City cannot exceed 50 characters.',
            'provider_state.required' => 'State is required.',
            'provider_state.max' => 'State cannot exceed 50 characters.',
            'provider_country.required' => 'Country is required.',
            'provider_country.max' => 'Country cannot exceed 50 characters.',
            'provider_postal_code.required' => 'Postal code is required.',
            'provider_postal_code.max' => 'Postal code cannot exceed 10 characters.',
            'provider_language.max' => 'Language cannot exceed 100 characters.',
            'provider_profile_image.required' => 'Profile picture is required.',
            'provider_profile_image.image' => 'Profile picture must be an image file.',
            'provider_profile_image.mimes' => 'Profile picture must be a JPEG, PNG, JPG, or GIF file.',
            'provider_profile_image.max' => 'Profile picture must not exceed 2MB.'
        ];
    }

    protected function failedValidation(\Illuminate\Contracts\Validation\Validator $validator)
    {
        \Log::error('Provider Registration Validation Failed', [
            'errors' => $validator->errors()->toArray(),
            'input' => $this->except(['password', 'provider_password'])
        ]);

        parent::failedValidation($validator);
    }
}