<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UserRegistrationRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        // If only requesting OTP, minimal validation
        if ($this->has('otp_only') && $this->get('otp_only')) {
            return [
                'email' => 'required|email|max:100',
            ];
        }

        // Check if registration requires OTP (register status = 1)
        $regStatus = \DB::table('general_settings')->where('key', 'register')->value('value');
        
        // If OTP is enabled, only validate required fields for OTP generation
        if ($regStatus === "1") {
            return [
                'first_name' => 'required|string|max:100',
                'last_name' => 'required|string|max:100',
                'name' => [
                    'required',
                    'max:100',
                    Rule::unique('users', 'name')
                ],
                'email' => [
                    'required',
                    'email',
                    'max:100',
                    Rule::unique('users', 'email')
                ],
                'password' => 'required|string|min:8|max:100',
                'phone_number' => 'required|string|min:10|max:15',
                'terms_policy' => 'required|accepted',
            ];
        }

        $rules = [
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'name' => [
                'required',
                'max:100',
                Rule::unique('users', 'name')
            ],
            'email' => [
                'required',
                'email',
                'max:100',
                Rule::unique('users', 'email')
            ],
            'password' => 'required|string|min:8|max:100',
            'phone_number' => 'required|string|min:10|max:15',
        ];

        if ($this->has('is_mobile') && $this->get('is_mobile') === "yes") {
            $rules['terms_policy'] = 'required|accepted';
        } else {
            $rules['terms_policy'] = 'required|accepted';
        }

        return $rules;
    }

    public function messages(): array
    {
        return [
            'name.required' => 'Name is required.',
            'email.required' => 'Email is required.',
            'email.email' => 'Enter a valid email address.',
            'email.unique' => 'This email is already registered.',
            'password.required' => 'Password is required.',
            'password.min' => 'Password must be at least 8 characters.',
            'phone_number.required' => 'Phone Number is required.',
            'terms_policy.required' => 'You must accept the terms and conditions.',
            'terms_policy.accepted' => 'You must accept the terms and conditions.',
        ];
    }

    protected function failedValidation(\Illuminate\Contracts\Validation\Validator $validator)
    {
        \Log::error('User Registration Validation Failed', [
            'errors' => $validator->errors()->toArray(),
            'input' => $this->except(['password', 'provider_password'])
        ]);

        parent::failedValidation($validator);
    }
}