<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ParadigmBadgeTitle extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'type_code',
        'badge_title',
        'description',
        'category',
        'preference_level'
    ];

    /**
     * Get the template that owns this badge title
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Scope for specific type code
     */
    public function scopeByType($query, string $typeCode)
    {
        return $query->where('type_code', $typeCode);
    }

    /**
     * Scope for specific category
     */
    public function scopeByCategory($query, string $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Get all 15 Paradigm personality types
     */
    public static function getAllParadigmTypes(): array
    {
        return [
            'The Analyst',
            'The Organiser', 
            'The Visionary',
            'The Helper',
            'The Specialist',
            'The Strategist',
            'The Advisor',
            'The Architect',
            'The Harmoniser',
            'The Creator',
            'The Innovator',
            'The Stabiliser',
            'The Integrator',
            'The Vision Builder',
            'The Whole-Brain Thinker'
        ];
    }

    /**
     * Check if type code is valid
     */
    public static function isValidTypeCode(string $typeCode): bool
    {
        return in_array($typeCode, self::getAllParadigmTypes());
    }

    /**
     * Get default badge titles for all 15 Paradigm types
     */
    public static function getDefaultBadgeTitles(): array
    {
        return [
            'The Analyst' => [
                'badge_title' => 'The Analyst',
                'description' => 'Logical, systematic, and detail-oriented thinkers who excel at analyzing complex problems.',
                'category' => 'Single Frame',
                'preference_level' => 'Very High'
            ],
            'The Organiser' => [
                'badge_title' => 'The Organiser',
                'description' => 'Structured, methodical, and efficient individuals who thrive on organization and planning.',
                'category' => 'Single Frame',
                'preference_level' => 'Very High'
            ],
            'The Visionary' => [
                'badge_title' => 'The Visionary',
                'description' => 'Creative, innovative, and future-focused individuals who see possibilities others miss.',
                'category' => 'Single Frame',
                'preference_level' => 'Very High'
            ],
            'The Helper' => [
                'badge_title' => 'The Helper',
                'description' => 'Empathetic, supportive, and people-focused individuals who prioritize relationships.',
                'category' => 'Single Frame',
                'preference_level' => 'Very High'
            ],
            'The Specialist' => [
                'badge_title' => 'The Specialist',
                'description' => 'Analytical and organized individuals who combine logical thinking with systematic approaches.',
                'category' => 'Dual Frame',
                'preference_level' => 'High'
            ],
            'The Strategist' => [
                'badge_title' => 'The Strategist',
                'description' => 'Analytical and visionary individuals who excel at long-term planning and innovation.',
                'category' => 'Dual Frame',
                'preference_level' => 'High'
            ],
            'The Advisor' => [
                'badge_title' => 'The Advisor',
                'description' => 'Analytical and supportive individuals who provide thoughtful guidance and support.',
                'category' => 'Dual Frame',
                'preference_level' => 'High'
            ],
            'The Architect' => [
                'badge_title' => 'The Architect',
                'description' => 'Organized and visionary individuals who design and build innovative solutions.',
                'category' => 'Dual Frame',
                'preference_level' => 'High'
            ],
            'The Harmoniser' => [
                'badge_title' => 'The Harmoniser',
                'description' => 'Organized and supportive individuals who create balance and harmony in their environments.',
                'category' => 'Dual Frame',
                'preference_level' => 'High'
            ],
            'The Creator' => [
                'badge_title' => 'The Creator',
                'description' => 'Visionary and supportive individuals who bring innovative ideas to life through collaboration.',
                'category' => 'Dual Frame',
                'preference_level' => 'High'
            ],
            'The Innovator' => [
                'badge_title' => 'The Innovator',
                'description' => 'Analytical, organized, and visionary individuals who drive breakthrough innovations.',
                'category' => 'Triple Frame',
                'preference_level' => 'Average'
            ],
            'The Stabiliser' => [
                'badge_title' => 'The Stabiliser',
                'description' => 'Analytical, organized, and supportive individuals who provide stability and reliability.',
                'category' => 'Triple Frame',
                'preference_level' => 'Average'
            ],
            'The Integrator' => [
                'badge_title' => 'The Integrator',
                'description' => 'Analytical, visionary, and supportive individuals who integrate diverse perspectives.',
                'category' => 'Triple Frame',
                'preference_level' => 'Average'
            ],
            'The Vision Builder' => [
                'badge_title' => 'The Vision Builder',
                'description' => 'Organized, visionary, and supportive individuals who build and implement transformative visions.',
                'category' => 'Triple Frame',
                'preference_level' => 'Average'
            ],
            'The Whole-Brain Thinker' => [
                'badge_title' => 'The Whole-Brain Thinker',
                'description' => 'Individuals who demonstrate balanced thinking across all four brain quadrants.',
                'category' => 'All Frame',
                'preference_level' => 'Low'
            ]
        ];
    }

    /**
     * Get preference level categories
     */
    public static function getPreferenceLevels(): array
    {
        return [
            'Very High' => '95%+ preference',
            'High' => '80-94% preference', 
            'Average' => '65-79% preference',
            'Low' => '50-64% preference',
            'Very Low' => '<50% preference'
        ];
    }

    /**
     * Get dominance categories
     */
    public static function getDominanceCategories(): array
    {
        return [
            'Single Frame',
            'Dual Frame',
            'Triple Frame', 
            'All Frame'
        ];
    }
}
