<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ParadigmClientResult extends Model
{
    use HasFactory;

    protected $table = 'paradigm_client_results';

    protected $fillable = [
        'client_id',
        'provider_id',
        'template_id',
        'payment_id',
        'result_id',
        'status',
        'started_at',
        'completed_at',
        'expires_at',
        'can_retake'
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'expires_at' => 'datetime',
        'can_retake' => 'boolean'
    ];

    /**
     * Get the client who owns the result
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(User::class, 'client_id');
    }

    /**
     * Get the provider who owns the result
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(User::class, 'provider_id');
    }

    /**
     * Get the template that owns the result
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Get the payment that owns the result
     */
    public function payment(): BelongsTo
    {
        return $this->belongsTo(ParadigmPayment::class, 'payment_id');
    }

    /**
     * Get the assessment result
     */
    public function result(): BelongsTo
    {
        return $this->belongsTo(ParadigmAssessmentResult::class, 'result_id');
    }

    /**
     * Check if assessment is completed
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed' && $this->completed_at !== null;
    }

    /**
     * Check if assessment is in progress
     */
    public function isInProgress(): bool
    {
        return $this->status === 'in_progress' && $this->started_at !== null;
    }

    /**
     * Check if assessment has expired
     */
    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Mark assessment as started
     */
    public function markAsStarted(): void
    {
        $this->update([
            'status' => 'in_progress',
            'started_at' => now()
        ]);
    }

    /**
     * Mark assessment as completed
     */
    public function markAsCompleted(int $resultId): void
    {
        $this->update([
            'status' => 'completed',
            'result_id' => $resultId,
            'completed_at' => now()
        ]);
    }
}