<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class ParadigmMarketplacePurchase extends Model
{
    use HasFactory;

    protected $fillable = [
        'result_id',
        'purchaser_id',
        'purchase_price',
        'purchased_at',
        'wallet_transaction_id',
        'is_sold',
        'linked_practitioner_id',
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'purchased_at' => 'datetime',
        'is_sold' => 'boolean',
    ];

    /**
     * Get the assessment result
     */
    public function result(): BelongsTo
    {
        return $this->belongsTo(ParadigmAssessmentResult::class, 'result_id');
    }

    /**
     * Get the purchaser (practitioner/admin)
     */
    public function purchaser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'purchaser_id');
    }

    /**
     * Get the linked practitioner
     */
    public function linkedPractitioner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'linked_practitioner_id');
    }

    /**
     * Get the wallet transaction
     */
    public function walletTransaction(): BelongsTo
    {
        return $this->belongsTo(WalletHistory::class, 'wallet_transaction_id');
    }

    /**
     * Get link request for this purchase
     */
    public function linkRequest(): HasOne
    {
        return $this->hasOne(ParadigmLinkRequest::class, 'marketplace_purchase_id');
    }

    /**
     * Get transfers for this purchase
     */
    public function transfers(): HasMany
    {
        return $this->hasMany(ParadigmPractitionerTransfer::class, 'marketplace_purchase_id');
    }

    /**
     * Check if profile is sold
     */
    public function isSold(): bool
    {
        return $this->is_sold;
    }

    /**
     * Mark profile as sold and link to practitioner
     */
    public function markAsSold(int $practitionerId): void
    {
        $this->update([
            'is_sold' => true,
            'linked_practitioner_id' => $practitionerId,
        ]);
    }
}

