<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ParadigmPdfElement extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'page_id',
        'page_number',
        'element_type',
        'element_key',
        'text_template',
        'x_pct',
        'y_pct',
        'w_pct',
        'h_pct',
        'font_px',
        'font_family',
        'font_style',
        'text_align',
        'font_weight',
        'color',
        'bg_color',
        'fg_color',
        'is_background',
        'background_image_path',
        'display_order',
        'css_styles',
        'css_classes',
        'css_id'
    ];

    protected $casts = [
        'x_pct' => 'decimal:2',
        'y_pct' => 'decimal:2',
        'w_pct' => 'decimal:2',
        'is_background' => 'boolean'
    ];

    /**
     * Get the template that owns this PDF element
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Get the PDF page that owns this element
     */
    public function page(): BelongsTo
    {
        return $this->belongsTo(ParadigmPdfPage::class, 'page_id');
    }

    /**
     * Scope for elements by type
     */
    public function scopeByType($query, string $type)
    {
        return $query->where('element_type', $type);
    }

    /**
     * Scope for elements by key
     */
    public function scopeByKey($query, string $key)
    {
        return $query->where('element_key', $key);
    }

    /**
     * Scope for background elements
     */
    public function scopeBackground($query)
    {
        return $query->where('is_background', true);
    }

    /**
     * Scope for content elements
     */
    public function scopeContent($query)
    {
        return $query->where('is_background', false);
    }

    /**
     * Get valid element keys for Paradigm Profile (74 Total Elements)
     */
    public static function getValidElementKeys(): array
    {
        return [
            // Personal data elements
            'name', 'surname', 'gender', 'age', 'phone', 'email',
            
            // ===== ALL 74 PARADIGM PROFILE ELEMENTS =====
            
            // ===== PARADIGM MAIN 14 ELEMENTS =====
            // 4 Frame Scores (numbers only, no %)
            'paradigm_main_lu_score', 'paradigm_main_ll_score', 'paradigm_main_ru_score', 'paradigm_main_rl_score',
            // 8 Field Scores (numbers only, no %)
            'paradigm_main_analyst_score', 'paradigm_main_organizer_score', 'paradigm_main_visionary_score', 'paradigm_main_helper_score',
            'paradigm_main_specialist_score', 'paradigm_main_strategist_score', 'paradigm_main_advisor_score', 'paradigm_main_architect_score',
            // Final Results (text)
            'paradigm_main_type', 'paradigm_main_dominance_type',
            
            // Percentage Values (23)
            'lu_percentage', 'll_percentage', 'ru_percentage', 'rl_percentage',
            'fact_finder_percentage', 'question_asker_percentage', 'rule_keeper_percentage', 'task_manager_percentage',
            'creator_percentage', 'big_thinker_percentage', 'empath_percentage', 'friend_maker_percentage',
            'extroversion_percentage', 'introversion_percentage',
            'emotional_balance_percentage', 'spiritual_balance_percentage', 'physical_balance_percentage',
            'social_balance_percentage', 'financial_balance_percentage', 'occupational_balance_percentage',
            'intellectual_balance_percentage', 'environmental_balance_percentage', 'overall_wellness_score',
            
            // Integer Score & Rank Values (20)
            'connection_style_verbal_validator_score', 'connection_style_presence_seeker_score', 'connection_style_symbolic_appreciator_score', 'connection_style_practical_supporter_score', 'connection_style_grounded_connector_score',
            'conflict_style_assertive_challenger_score', 'conflict_style_integrative_bridge_builder_score', 'conflict_style_fair_negotiator_score', 'conflict_style_quiet_protector_score', 'conflict_style_peacekeeper_score',
            'connection_style_verbal_validator_rank', 'connection_style_presence_seeker_rank', 'connection_style_symbolic_appreciator_rank', 'connection_style_practical_supporter_rank', 'connection_style_grounded_connector_rank',
            'conflict_style_assertive_challenger_rank', 'conflict_style_integrative_bridge_builder_rank', 'conflict_style_fair_negotiator_rank', 'conflict_style_quiet_protector_rank', 'conflict_style_peacekeeper_rank',
            
            // User & Report Information (8)
            'user_full_name', 'report_date', 'user_gender', 'user_age_range',
            'practitioner_name', 'practitioner_email', 'practitioner_phone', 'practitioner_url',
            
            // Profile Types & Labels (6)
            'paradigm_type', 'dominance_category_short',
            'lu_preference_level', 'll_preference_level', 'ru_preference_level', 'rl_preference_level',
            
            // Ranked Style Names (10)
            'connection_style_rank_1_name', 'connection_style_rank_2_name', 'connection_style_rank_3_name', 'connection_style_rank_4_name', 'connection_style_rank_5_name',
            'conflict_style_rank_1_name', 'conflict_style_rank_2_name', 'conflict_style_rank_3_name', 'conflict_style_rank_4_name', 'conflict_style_rank_5_name',
            
            // Quotient Levels (6)
            'social_intelligence_level', 'emotional_intelligence_level', 'relevancy_quotient_level',
            'social_intelligence_score', 'emotional_intelligence_score', 'relevancy_quotient_score',
            
            // Conditional Booleans (4)
            'is_single_frame', 'is_dual_frame', 'is_triple_frame', 'is_all_frame',
            
            // Paradigm PDF builder element types (prefixed for compatibility)
            'paradigm_score_emotional_balance', 'paradigm_score_spiritual_balance', 'paradigm_score_physical_balance',
            'paradigm_score_social_balance', 'paradigm_score_financial_balance', 'paradigm_score_occupational_balance',
            'paradigm_score_intellectual_balance', 'paradigm_score_environmental_balance', 'paradigm_score_overall_wellness',
            'paradigm_personality_type', 'paradigm_type_description', 'paradigm_badges',
            'paradigm_name', 'paradigm_email', 'paradigm_gender', 'paradigm_age', 'paradigm_phone',
            'paradigm_personal_name', 'paradigm_personal_email', 'paradigm_personal_gender', 'paradigm_personal_age', 'paradigm_personal_phone',
            
            // ===== NEW ELEMENTS - ADDED 2025-10-11 =====
            // Combined user details block
            'user_details_block',
            // Combined provider details block
            'provider_details_block',
            // Combined full name (first + last)
            'full_name',
        ];
    }

    /**
     * Check if element key is valid
     */
    public static function isValidElementKey(string $key): bool
    {
        return in_array($key, self::getValidElementKeys());
    }

    /**
     * Get element position in pixels (for PDF rendering)
     */
    public function getPositionInPixels(float $pageWidth, float $pageHeight): array
    {
        return [
            'x' => ($this->x_pct / 100) * $pageWidth,
            'y' => ($this->y_pct / 100) * $pageHeight,
            'width' => ($this->w_pct / 100) * $pageWidth
        ];
    }

    /**
     * Check if element is a score element (percentages, levels, etc.)
     */
    public function isScoreElement(): bool
    {
        return str_contains($this->element_key, '_percentage') || 
               str_contains($this->element_key, '_level') ||
               str_contains($this->element_key, '_score') ||
               str_starts_with($this->element_key, 'paradigm_score_') ||
               str_starts_with($this->element_key, 'paradigm_main_') && str_contains($this->element_key, '_score');
    }

    /**
     * Check if element is a personal data element
     */
    public function isPersonalDataElement(): bool
    {
        return in_array($this->element_key, ['name', 'surname', 'gender', 'age', 'phone', 'email']) ||
               str_starts_with($this->element_key, 'paradigm_personal_');
    }

    /**
     * Check if element is a Paradigm type element
     */
    public function isParadigmTypeElement(): bool
    {
        return in_array($this->element_key, ['paradigm_type', 'dominance_category_short']) ||
               str_contains($this->element_key, 'paradigm_type') ||
               str_contains($this->element_key, 'paradigm_badges');
    }
}
