<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ParadigmPersonalField extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'field_key',
        'field_label',
        'field_type',
        'is_required',
        'options',
        'display_order',
        'field_settings'
    ];

    protected $casts = [
        'is_required' => 'boolean',
        'options' => 'array',
        'field_settings' => 'array'
    ];

    /**
     * Get the template that owns this personal field
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Scope for required fields
     */
    public function scopeRequired($query)
    {
        return $query->where('is_required', true);
    }

    /**
     * Scope for optional fields
     */
    public function scopeOptional($query)
    {
        return $query->where('is_required', false);
    }

    /**
     * Get options as an array.
     */
    public function getOptionsArray(): array
    {
        if (is_array($this->options)) {
            return $this->options;
        }

        if (is_string($this->options)) {
            // Attempt to decode if it's a JSON string representing an array
            $decoded = json_decode($this->options, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                return $decoded;
            }
            
            // Fallback for comma-separated strings
            $options = array_map('trim', explode(',', $this->options));
            // Ensure that if the string is empty, we don't return an array with an empty string
            return array_filter($options, function($value) {
                return $value !== '';
            });
        }
        
        return [];
    }

    /**
     * Check if field has options
     */
    public function hasOptions(): bool
    {
        return !empty($this->options);
    }

    /**
     * Get field options as text (one option per line)
     */
    public function getOptionsText(): string
    {
        if (empty($this->options)) {
            return '';
        }
        
        return implode("\n", $this->options);
    }

    /**
     * Get field settings
     */
    public function getFieldSettings(): array
    {
        return $this->field_settings ?? [];
    }

    /**
     * Get valid field types for Paradigm Profile
     */
    public static function getValidFieldTypes(): array
    {
        return [
            'text',
            'email',
            'tel',
            'number',
            'select',
            'radio',
            'checkbox',
            'textarea',
            'date',
            'time',
            'datetime-local'
        ];
    }

    /**
     * Validate field type
     */
    public function validateFieldType(): bool
    {
        return in_array($this->field_type, self::getValidFieldTypes());
    }

    /**
     * Check if field is a select/radio/checkbox type
     */
    public function isChoiceField(): bool
    {
        return in_array($this->field_type, ['select', 'radio', 'checkbox']);
    }

    /**
     * Check if field is a text input type
     */
    public function isTextField(): bool
    {
        return in_array($this->field_type, ['text', 'email', 'tel', 'number', 'textarea']);
    }

    /**
     * Check if field is a date/time type
     */
    public function isDateTimeField(): bool
    {
        return in_array($this->field_type, ['date', 'time', 'datetime-local']);
    }
}
