<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Class ProviderIdentityVerification
 *
 * @property int $id
 * @property int $user_id
 * @property string|null $id_document_type
 * @property string|null $id_document_front
 * @property string|null $id_document_back
 * @property string|null $selfie_photo
 * @property string $verification_status
 * @property string|null $rejection_reason
 * @property \Illuminate\Support\Carbon|null $submitted_at
 * @property \Illuminate\Support\Carbon|null $verified_at
 * @property int|null $verified_by
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property \Illuminate\Support\Carbon|null $deleted_at
 */
class ProviderIdentityVerification extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'provider_identity_verifications';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'id_document_type',
        'id_document_front',
        'id_document_back',
        'selfie_photo',
        'verification_status',
        'rejection_reason',
        'submitted_at',
        'verified_at',
        'verified_by',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'submitted_at' => 'datetime',
        'verified_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Get the user that owns the verification
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the admin who verified this
     */
    public function verifier(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    /**
     * Check if verification is pending
     */
    public function isPending(): bool
    {
        return $this->verification_status === 'pending';
    }

    /**
     * Check if verification is verified
     */
    public function isVerified(): bool
    {
        return $this->verification_status === 'verified';
    }

    /**
     * Check if verification is rejected
     */
    public function isRejected(): bool
    {
        return $this->verification_status === 'rejected';
    }

    /**
     * Check if all documents are uploaded
     */
    public function hasAllDocuments(): bool
    {
        return !empty($this->id_document_front) && !empty($this->selfie_photo);
    }

    /**
     * Get the ID front image URL
     */
    public function getIdFrontUrl(): ?string
    {
        return $this->id_document_front 
            ? url('storage/provider-verification/' . $this->user_id . '/' . $this->id_document_front)
            : null;
    }

    /**
     * Get the ID back image URL
     */
    public function getIdBackUrl(): ?string
    {
        return $this->id_document_back 
            ? url('storage/provider-verification/' . $this->user_id . '/' . $this->id_document_back)
            : null;
    }

    /**
     * Get the selfie image URL
     */
    public function getSelfieUrl(): ?string
    {
        return $this->selfie_photo 
            ? url('storage/provider-verification/' . $this->user_id . '/' . $this->selfie_photo)
            : null;
    }

    /**
     * Mark as submitted
     */
    public function markAsSubmitted(): void
    {
        $this->update([
            'submitted_at' => now(),
            'verification_status' => 'pending'
        ]);
    }

    /**
     * Approve verification
     */
    public function approve(int $adminId): void
    {
        $this->update([
            'verification_status' => 'verified',
            'verified_at' => now(),
            'verified_by' => $adminId,
            'rejection_reason' => null
        ]);
    }

    /**
     * Reject verification
     */
    public function reject(int $adminId, string $reason): void
    {
        $this->update([
            'verification_status' => 'rejected',
            'verified_at' => now(),
            'verified_by' => $adminId,
            'rejection_reason' => $reason
        ]);
    }
}

