<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Modules\Categories\app\Models\Categories;
use Modules\Product\app\Models\Product;
use App\Models\UserDetail;
use App\Models\Bookings;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Class User
 *
 * @property int $id
 * @property string $name
 * @property string $email
 * @property string|null $password
 * @property string|null $phone_number
 * @property string|null $user_type
 * @property int|null $auth_provider_id
 * @property string|null $auth_provider
 * @property string $status
 * @property \Illuminate\Support\Carbon|null $email_verified_at
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property \Illuminate\Support\Carbon|null $deleted_at
 * @property string|null $remember_token
 */
class User extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone_number',
        'user_type',
        'auth_provider_id',
        'auth_provider',
        'status',
        'user_language_id',
        'role_id',
        'sub_service_type',
        'provider_verified_status',
        'category_id',
        'staff_verification_status',
        'staff_verified_by',
        'staff_verified_at',
        'staff_rejection_reason',
        'dob',
        'gender',
        'address',
        'city',
        'state',
        'country',
        'postal_code',
        'bio',
        'language',
        'paddle_customer_id',
        'free_assessment_used',
        'free_assessment_id',
        'is_company_owner',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'is_company_owner' => 'boolean',
    ];

    public static string $userSecretKey = 'userId';


    /**
     * Get the category associated with the user.
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Categories::class, 'category_id');
    }

    /**
     * Get the user details associated with the user.
     */
    public function userDetails(): HasOne
    {
        return $this->hasOne(UserDetail::class, 'user_id');
    }

    /**
     * Get the user who created this user.
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the products created by the user.
     */
    public function products(): HasMany
    {
        return $this->hasMany(Product::class, 'created_by');
    }

    public function bookings()
    {
        return $this->hasMany(Bookings::class);
    }

    public function userDetail(): HasOne
    {
        return $this->hasOne(UserDetail::class, 'user_id');
    }

    public function providerDetails(): HasMany
    {
        return $this->hasMany(\App\Models\ProviderDetail::class, 'user_id');
    }

    public function packageTransactions(): HasMany
    {
        return $this->hasMany(\App\Models\PackageTrx::class, 'provider_id');
    }

    public function activePackageTransaction(): HasOne
    {
        return $this->hasOne(\App\Models\PackageTrx::class, 'provider_id')
            ->where('status', 1)
            ->where('end_date', '>', now())
            ->latest();
    }

    /**
     * Get the identity verification for this provider
     */
    public function identityVerification(): HasOne
    {
        return $this->hasOne(ProviderIdentityVerification::class, 'user_id');
    }

    /**
     * Get the documents for this staff member
     */
    public function staffDocuments(): HasMany
    {
        return $this->hasMany(StaffDocument::class, 'staff_id');
    }

    /**
     * Get the admin who verified this staff member
     */
    public function staffVerifier(): BelongsTo
    {
        return $this->belongsTo(User::class, 'staff_verified_by');
    }

    /**
     * Check if user is a company owner (main provider converted to staff)
     */
    public function isCompanyOwner(): bool
    {
        return (bool) $this->is_company_owner;
    }
}
