<?php

namespace App\Services;

use App\Repositories\Eloquent\WalletRepository;
use App\Models\WalletHistory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

/**
 * MarketplaceWalletService
 * 
 * Wrapper service for marketplace lead unlocking wallet operations.
 * Uses existing WalletRepository without modifying it.
 * 
 * IMPORTANT: This service does NOT modify any existing wallet functionality.
 * It only provides a safe interface for marketplace lead unlocking.
 */
class MarketplaceWalletService
{
    protected WalletRepository $walletRepo;

    public function __construct()
    {
        $this->walletRepo = new WalletRepository();
    }

    /**
     * Get user's wallet balance
     * Uses existing WalletRepository method
     * 
     * @param int $userId
     * @return float
     */
    public function getBalance(int $userId): float
    {
        try {
            if ($userId <= 0) {
                throw new \InvalidArgumentException('Invalid user ID provided');
            }
            
            $balanceData = $this->walletRepo->getWalletBalance($userId);
            
            if (!isset($balanceData['balance']) || !is_numeric($balanceData['balance'])) {
                Log::warning('Invalid balance data returned for user', [
                    'user_id' => $userId,
                    'balance_data' => $balanceData
                ]);
                return 0.0;
            }
            
            return (float) $balanceData['balance'];
        } catch (\Exception $e) {
            Log::error('Failed to get wallet balance', [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return 0.0;
        }
    }

    /**
     * Get detailed wallet balance (total credits, debits, balance)
     * Uses existing WalletRepository method
     * 
     * @param int $userId
     * @return array ['total' => float, 'debit' => float, 'balance' => float]
     */
    public function getDetailedBalance(int $userId): array
    {
        return $this->walletRepo->getWalletBalance($userId);
    }

    /**
     * Check if user has sufficient balance for marketplace lead unlock
     * 
     * @param int $userId
     * @param float $amount
     * @return bool
     */
    public function hasSufficientBalance(int $userId, float $amount): bool
    {
        $balance = $this->getBalance($userId);
        return $balance >= $amount;
    }

    /**
     * Deduct amount from provider wallet for marketplace lead unlock
     * Uses existing WalletRepository::processWalletPayment() method
     * 
     * @param int $providerId
     * @param float $amount
     * @param int $leadId - Reference to user_form_inputs.id
     * @param string $description
     * @return WalletHistory|false
     */
    public function deductProviderWallet(int $providerId, float $amount, int $leadId, string $description)
    {
        try {
            // Validate input parameters
            if ($providerId <= 0) {
                throw new \InvalidArgumentException('Invalid provider ID');
            }
            
            if ($amount <= 0) {
                throw new \InvalidArgumentException('Invalid amount for deduction');
            }
            
            if ($leadId <= 0) {
                throw new \InvalidArgumentException('Invalid lead ID');
            }
            
            if (empty(trim($description))) {
                throw new \InvalidArgumentException('Description is required');
            }

            // Use database transaction with locking to prevent race conditions
            return DB::transaction(function () use ($providerId, $amount, $leadId, $description) {
                // Lock the user's wallet records to prevent concurrent modifications
                $walletRecords = DB::table('wallet_history')
                    ->where('user_id', $providerId)
                    ->lockForUpdate()
                    ->get();
                
                // Double-check balance with locked data
                $currentBalance = $this->getBalance($providerId);
                if ($currentBalance < $amount) {
                    Log::warning('Insufficient balance for marketplace lead unlock', [
                        'provider_id' => $providerId,
                        'required_amount' => $amount,
                        'balance' => $currentBalance,
                        'lead_id' => $leadId
                    ]);
                    return false;
                }

                // Use existing wallet repository method (same as bookings/leads)
                // NOTE: payment_type must be 'Others' (enum constraint)
                // We use transaction_id to identify marketplace transactions
                $walletTransaction = $this->walletRepo->processWalletPayment([
                    'user_id' => $providerId,
                    'amount' => $amount,
                    'payment_type' => 'Others', // Using enum value
                    'reference_id' => $leadId, // Links to user_form_inputs table
                    'transaction_id' => 'MARKETPLACE-LEAD-' . $leadId . '-' . Str::uuid() // Unique identifier
                ]);

                if ($walletTransaction) {
                    Log::info('Provider wallet deducted for marketplace lead unlock', [
                        'provider_id' => $providerId,
                        'amount' => $amount,
                        'wallet_history_id' => $walletTransaction->id,
                        'lead_id' => $leadId
                    ]);
                }

                return $walletTransaction;
            });

        } catch (\Exception $e) {
            Log::error('Failed to deduct provider wallet for marketplace lead', [
                'provider_id' => $providerId,
                'amount' => $amount,
                'lead_id' => $leadId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Refund amount to provider wallet (if lead unlock fails or is cancelled)
     * Uses existing WalletRepository::processWalletPayment() method
     * 
     * @param int $providerId
     * @param float $amount
     * @param int $leadId - Reference to user_form_inputs.id
     * @param string $description
     * @return WalletHistory|false
     */
    public function refundProviderWallet(int $providerId, float $amount, int $leadId, string $description)
    {
        try {
            // Use existing wallet repository method for credit (positive amount)
            $walletTransaction = $this->walletRepo->processWalletPayment([
                'user_id' => $providerId,
                'amount' => -$amount, // Negative amount for credit/refund
                'payment_type' => 'Others', // Using enum value
                'reference_id' => $leadId, // Links to user_form_inputs table
                'transaction_id' => 'MARKETPLACE-REFUND-' . $leadId . '-' . Str::uuid() // Unique identifier
            ]);

            if ($walletTransaction) {
                Log::info('Provider wallet refunded for marketplace lead', [
                    'provider_id' => $providerId,
                    'amount' => $amount,
                    'wallet_history_id' => $walletTransaction->id,
                    'lead_id' => $leadId
                ]);
            }

            return $walletTransaction;

        } catch (\Exception $e) {
            Log::error('Failed to refund provider wallet for marketplace lead', [
                'provider_id' => $providerId,
                'amount' => $amount,
                'lead_id' => $leadId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get wallet transaction history for marketplace leads
     * 
     * @param int $userId
     * @param int $limit
     * @return \Illuminate\Support\Collection
     */
    public function getMarketplaceTransactions(int $userId, int $limit = 50)
    {
        try {
            return DB::table('wallet_history')
                ->where('user_id', $userId)
                ->where('transaction_id', 'LIKE', 'MARKETPLACE-%')
                ->orderBy('created_at', 'desc')
                ->limit($limit)
                ->get();
        } catch (\Exception $e) {
            Log::error('Failed to get marketplace wallet transactions', [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return collect();
        }
    }
}
