<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Modules\GlobalSetting\app\Models\Currency;

class PaddleService
{
    private $vendorId;
    private $apiKey;
    private $webhookSecret;
    private $environment;
    private $baseUrl;

    public function __construct()
    {
        $this->vendorId = config('paddle.vendor_id');
        $this->apiKey = config('paddle.api_key');
        $this->webhookSecret = config('paddle.webhook_secret');
        $this->environment = config('paddle.environment', 'live');
        
        // Paddle Billing API base URL
        $this->baseUrl = $this->environment === 'sandbox' 
            ? 'https://sandbox-api.paddle.com' 
            : 'https://api.paddle.com';
    }

    /**
     * Get system currency code (USD, EUR, GBP, etc.)
     */
    private function getSystemCurrency()
    {
        $currency = Currency::where('is_default', 1)->first();
        return $currency ? strtoupper($currency->code) : 'USD';
    }

    /**
     * Create a Paddle transaction for wallet top-up
     * Paddle Billing API: POST /transactions
     * 
     * Can work two ways:
     * 1. With existing customer_id (for returning customers)
     * 2. With customer email (Paddle collects info during checkout)
     */
    public function createWalletCheckout($amount, $userId, $currency = null)
    {
        try {
            if (!$currency) {
                $currency = $this->getSystemCurrency();
            }

            $user = auth()->user();
            
            // Get price ID for wallet top-up
            $priceId = $this->getWalletPriceId($amount, $currency);
            
            if (!$priceId) {
                return [
                    'success' => false,
                    'error' => 'Wallet top-up is not configured for this amount. Please contact administrator.',
                ];
            }

            // Try to get existing customer ID, but don't fail if it doesn't exist
            $customerId = $this->getPaddleCustomerId($user);
            
            // Calculate quantity based on amount (each unit = $1)
            // This allows any custom amount using a single $1 USD price
            $quantity = (int)$amount;

            $payload = [
                'items' => [
                    [
                        'price_id' => $priceId,
                        'quantity' => $quantity, // e.g., 75 units of $1 = $75
                    ]
                ],
                'custom_data' => [
                    'user_id' => $userId,
                    'type' => 'wallet_topup',
                    'amount' => $amount,
                    'currency' => $currency,
                ],
                'billing_details' => [
                    'enable_checkout' => true,
                    'purchase_order_number' => 'WALLET-' . $userId . '-' . time(),
                ],
            ];

            // If we have a customer ID, use it. Otherwise, provide email and let Paddle collect info
            if ($customerId) {
                $payload['customer_id'] = $customerId;
            } else {
                $payload['customer_email'] = $user->email;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/transactions', $payload);

            if ($response->successful()) {
                $data = $response->json();
                
                // Get checkout URL from transaction
                $checkoutUrl = $data['data']['checkout']['url'] ?? null;
                
                if (!$checkoutUrl) {
                    Log::error('Paddle transaction created but no checkout URL', ['response' => $data]);
                    return [
                        'success' => false,
                        'error' => 'Failed to get checkout URL from Paddle',
                    ];
                }
                
                return [
                    'success' => true,
                    'checkout_url' => $checkoutUrl,
                    'transaction_id' => $data['data']['id'],
                ];
            }

            Log::error('Paddle wallet checkout failed', [
                'status' => $response->status(),
                'response' => $response->json()
            ]);

            return [
                'success' => false,
                'error' => 'Failed to create Paddle checkout',
                'details' => $response->json(),
            ];

        } catch (\Exception $e) {
            Log::error('Paddle wallet checkout error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return [
                'success' => false,
                'error' => 'Paddle service error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get Paddle customer ID for user (if exists)
     * Does NOT create - Paddle will create customer automatically during checkout
     */
    private function getPaddleCustomerId($user)
    {
        // Check if user has Paddle customer ID stored
        return DB::table('users')
            ->where('id', $user->id)
            ->value('paddle_customer_id');
    }

    /**
     * Store Paddle customer ID after webhook confirmation
     * This is called from webhook when Paddle creates the customer
     */
    public function storePaddleCustomerId($userId, $customerId)
    {
        DB::table('users')
            ->where('id', $userId)
            ->update(['paddle_customer_id' => $customerId]);
    }

    /**
     * Create a Paddle subscription
     * Paddle Billing API: POST /subscriptions
     * 
     * Can work two ways:
     * 1. With existing customer_id (for returning customers)
     * 2. With customer email (Paddle collects info during checkout)
     */
    public function createSubscription($priceId, $userId, $currency = null)
    {
        try {
            if (!$currency) {
                $currency = $this->getSystemCurrency();
            }

            $user = auth()->user();
            
            // Try to get existing customer ID
            $customerId = $this->getPaddleCustomerId($user);

            $payload = [
                'items' => [
                    [
                        'price_id' => $priceId,
                        'quantity' => 1,
                    ]
                ],
                'custom_data' => [
                    'user_id' => $userId,
                    'type' => 'subscription',
                ],
                'billing_details' => [
                    'enable_checkout' => true,
                ],
            ];

            // If we have a customer ID, use it. Otherwise, provide email
            if ($customerId) {
                $payload['customer_id'] = $customerId;
            } else {
                $payload['customer_email'] = $user->email;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/subscriptions', $payload);

            if ($response->successful()) {
                $data = $response->json();
                
                // Get checkout URL for payment
                $checkoutUrl = $data['data']['checkout']['url'] ?? null;
                
                return [
                    'success' => true,
                    'checkout_url' => $checkoutUrl,
                    'subscription_id' => $data['data']['id'],
                ];
            }

            Log::error('Paddle subscription creation failed', [
                'status' => $response->status(),
                'response' => $response->json()
            ]);

            return [
                'success' => false,
                'error' => 'Failed to create Paddle subscription',
                'details' => $response->json(),
            ];

        } catch (\Exception $e) {
            Log::error('Paddle subscription error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Paddle service error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get transaction details from Paddle
     */
    public function getTransaction($transactionId)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->get($this->baseUrl . '/transactions/' . $transactionId);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to fetch transaction',
            ];

        } catch (\Exception $e) {
            Log::error('Paddle get transaction error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Paddle service error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get subscription details from Paddle
     */
    public function getSubscription($subscriptionId)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->get($this->baseUrl . '/subscriptions/' . $subscriptionId);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to fetch subscription',
            ];

        } catch (\Exception $e) {
            Log::error('Paddle get subscription error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Paddle service error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Cancel a subscription
     */
    public function cancelSubscription($subscriptionId)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/subscriptions/' . $subscriptionId . '/cancel', [
                'effective_from' => 'next_billing_period'
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to cancel subscription',
            ];

        } catch (\Exception $e) {
            Log::error('Paddle cancel subscription error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Paddle service error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Verify webhook signature (Paddle Billing uses Paddle-Signature header)
     * Reference: https://developer.paddle.com/webhooks/verify-webhooks
     */
    public function verifyWebhook($payload, $signature)
    {
        if (empty($signature) || empty($this->webhookSecret)) {
            return false;
        }

        // Paddle signature format: ts=timestamp;h1=signature
        $parts = [];
        foreach (explode(';', $signature) as $part) {
            list($key, $value) = explode('=', $part, 2);
            $parts[$key] = $value;
        }

        if (!isset($parts['ts']) || !isset($parts['h1'])) {
            return false;
        }

        $timestamp = $parts['ts'];
        $receivedSignature = $parts['h1'];

        // Check timestamp is within 5 minutes
        if (abs(time() - $timestamp) > 300) {
            Log::warning('Paddle webhook timestamp too old', ['timestamp' => $timestamp]);
            return false;
        }

        // Verify signature
        $signedPayload = $timestamp . ':' . $payload;
        $expectedSignature = hash_hmac('sha256', $signedPayload, $this->webhookSecret);

        return hash_equals($expectedSignature, $receivedSignature);
    }

    /**
     * Get wallet price ID
     * Returns the $1 USD price ID for quantity-based pricing
     */
    private function getWalletPriceId($amount, $currency)
    {
        return config('paddle.wallet_price_id');
    }

    /**
     * Check if Paddle is enabled and configured
     */
    public function isEnabled()
    {
        return !empty($this->vendorId) && 
               !empty($this->apiKey) && 
               !empty($this->webhookSecret);
    }

    /**
     * Get Paddle configuration status
     */
    public function getConfigStatus()
    {
        return [
            'enabled' => $this->isEnabled(),
            'vendor_id' => !empty($this->vendorId),
            'api_key' => !empty($this->apiKey),
            'webhook_secret' => !empty($this->webhookSecret),
            'environment' => $this->environment,
            'base_url' => $this->baseUrl,
            'currency' => $this->getSystemCurrency(),
        ];
    }

    /**
     * Get price preview (for displaying localized prices)
     */
    public function getPricePreview($priceId, $currency = null, $customerIp = null)
    {
        try {
            if (!$currency) {
                $currency = $this->getSystemCurrency();
            }

            $params = [
                'items' => [
                    [
                        'price_id' => $priceId,
                        'quantity' => 1,
                    ]
                ],
            ];

            if ($customerIp) {
                $params['customer_ip_address'] = $customerIp;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/pricing-preview', $params);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to get price preview',
            ];

        } catch (\Exception $e) {
            Log::error('Paddle price preview error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Paddle service error: ' . $e->getMessage(),
            ];
        }
    }
}
