<?php

namespace App\Services;

use App\Models\ParadigmAssessmentResult;
use App\Models\ParadigmFormTemplate;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;

class ParadigmPdfServiceDomPDF
{
    public function generatePdf(ParadigmAssessmentResult $result)
    {
        $filename = 'paradigm-profile-' . $result->id . '.pdf';
        $folder = 'reports';
        $path = $folder . '/' . $filename;

        // Check if file exists in storage
        if (Storage::disk('public')->exists($path)) {
            return response()->download(Storage::disk('public')->path($path), $filename);
        }

        $template = $result->template;

        // 1. Prepare Data
        $data = $this->prepareDataForTemplate($result);

        // 2. Process Pages and Elements
        $processedPages = $this->processPagesForTemplate($template);

        // 3. Render PDF
        $pdf = PDF::loadView('pdfs.paradigm-assessment', [
            'processed_pages' => $processedPages,
            'data' => $data,
            'template' => $template
        ]);
        
        // Set paper size and orientation
        $pdf->setPaper('a4', 'portrait');

        // Save to storage
        Storage::disk('public')->put($path, $pdf->output());

        return response()->download(Storage::disk('public')->path($path), $filename);
    }

    private function prepareDataForTemplate(ParadigmAssessmentResult $result): array
    {
        // Default values
        $data = [
            'name' => $result->name ?? '',
            'surname' => $result->surname ?? '',
            'email' => $result->email ?? '',
            'gender' => $result->gender ?? '',
            'age_range' => $result->age_range ?? '',
            'phone' => $result->phone ?? '',
            'date_completed' => optional($result->completed_at)->format('Y-m-d H:i') ?? '',
        ];

        // Provider Information - Check if self-purchased
        // If self-purchased (provider_id == client_id in payment), use admin/company details
        $payment = \App\Models\ParadigmPayment::where('id', $result->payment_id ?? 0)->first();
        $isSelfPurchased = false;
        
        if ($payment && ($payment->self_purchased == 1 || $payment->provider_id == $payment->client_id)) {
            $isSelfPurchased = true;
        }
        
        if ($isSelfPurchased) {
            // Use admin/company details as "provider" for self-purchases
            $settings = \DB::table('general_settings')
                ->whereIn('key', ['company_name', 'site_email', 'phone_no', 'address'])
                ->pluck('value', 'key');
            
            $data['provider_name'] = $settings['company_name'] ?? 'Admin';
            $data['provider_last_name'] = ''; // Company doesn't have last name
            $data['provider_phone'] = $settings['phone_no'] ?? '';
            $data['provider_email'] = $settings['site_email'] ?? '';
        } else {
            // Use actual provider details for regular provider-to-client flow
            if ($result->provider) {
                $provider = $result->provider;
                $provider->load('userDetails'); // Load relationship
                
                // Check if provider is a company owner - use MAIN BRANCH details
                if ($provider->user_type == 4 && isset($provider->is_company_owner) && $provider->is_company_owner) {
                    // Fetch main branch details
                    $mainBranch = \App\Models\Branches::where('created_by', $provider->id)
                        ->where('is_main_branch', true)
                        ->whereNull('deleted_at')
                        ->first();
                    
                    if ($mainBranch) {
                        // Use main branch company details
                        $data['provider_name'] = $mainBranch->branch_name ?? '';
                        $data['provider_last_name'] = ''; // Companies don't have last names
                        $data['provider_phone'] = $mainBranch->branch_mobile ?? '';
                        $data['provider_email'] = $mainBranch->branch_email ?? '';
                        $data['is_company_owner'] = true;
                        $data['main_branch_name'] = $mainBranch->branch_name ?? '';
                        $data['main_branch_email'] = $mainBranch->branch_email ?? '';
                        $data['main_branch_mobile'] = $mainBranch->branch_mobile ?? '';
                    } else {
                        // Fallback to personal details if no main branch found
                        $data['provider_name'] = optional($provider->userDetails)->first_name ?? '';
                        $data['provider_last_name'] = optional($provider->userDetails)->last_name ?? '';
                        $data['provider_phone'] = $provider->phone_number ?? '';
                        $data['provider_email'] = $provider->email ?? '';
                        $data['is_company_owner'] = true;
                    }
                } else {
                    // Regular provider - use personal details
                    $data['provider_name'] = optional($provider->userDetails)->first_name ?? '';
                    $data['provider_last_name'] = optional($provider->userDetails)->last_name ?? '';
                    $data['provider_phone'] = $provider->phone_number ?? '';
                    $data['provider_email'] = $provider->email ?? '';
                    $data['is_company_owner'] = false;
                }
            } else {
                // Default empty values if no provider
                $data['provider_name'] = '';
                $data['provider_last_name'] = '';
                $data['provider_phone'] = '';
                $data['provider_email'] = '';
                $data['is_company_owner'] = false;
            }
        }

        // Unpack scores_json
        $scores = $result->scores_json; // This is already an array due to model casting
        
        // Part 1: Brain Frames
        $frameScores = $scores['paradigm_main']['scores']['part1'] ?? [];
        $framePercentages = $scores['paradigm_main']['percentages']['part1'] ?? [];
        $data = array_merge($data, [
            'lu_percentage' => $framePercentages['LU'] ?? 0,
            'll_percentage' => $framePercentages['LL'] ?? 0,
            'ru_percentage' => $framePercentages['RU'] ?? 0,
            'rl_percentage' => $framePercentages['RL'] ?? 0,
        ]);

        // Part 2: Brain Fields
        $fieldPercentages = $scores['paradigm_main']['percentages']['part2'] ?? [];
        $data = array_merge($data, [
            'question_asker_percentage' => $fieldPercentages['LU – The Question Asker'] ?? 0,
            'fact_finder_percentage' => $fieldPercentages['LU – The Fact Finder'] ?? 0,
            'rule_keeper_percentage' => $fieldPercentages['LL – The Rule Keeper'] ?? 0,
            'task_manager_percentage' => $fieldPercentages['LL – The Task Manager'] ?? 0,
            'creator_percentage' => $fieldPercentages['RU – The Creator'] ?? 0,
            'big_thinker_percentage' => $fieldPercentages['RU – The Big Thinker'] ?? 0,
            'friend_maker_percentage' => $fieldPercentages['RL – The Friend Maker'] ?? 0,
            'empath_percentage' => $fieldPercentages['RL – The Empath'] ?? 0,
        ]);

        // Part 3: Personality Blueprint (E/I) - NEW
        $part3Data = $scores['part3'] ?? [];
        $data = array_merge($data, [
            'e_score' => $part3Data['e_score'] ?? 0,
            'i_score' => $part3Data['i_score'] ?? 0,
            'personality_type' => $part3Data['personality_type'] ?? '',
        ]);

        // Part 4: Conflict Styles - NEW
        $part4Data = $scores['part4'] ?? [];
        $data = array_merge($data, [
            'assertive_challenger' => $part4Data['assertive_challenger'] ?? 0,
            'integrative_bridge_builder' => $part4Data['integrative_bridge_builder'] ?? 0,
            'fair_negotiator' => $part4Data['fair_negotiator'] ?? 0,
            'quiet_protector' => $part4Data['quiet_protector'] ?? 0,
            'peacekeeper' => $part4Data['peacekeeper'] ?? 0,
            'dominant_conflict_style' => $part4Data['dominant_conflict_style'] ?? '',
        ]);

        // Part 5: Connection Styles - NEW
        $part5Data = $scores['part5'] ?? [];
        $data = array_merge($data, [
            'verbal_validator' => $part5Data['verbal_validator'] ?? 0,
            'presence_seeker' => $part5Data['presence_seeker'] ?? 0,
            'symbolic_appreciator' => $part5Data['symbolic_appreciator'] ?? 0,
            'practical_supporter' => $part5Data['practical_supporter'] ?? 0,
            'grounded_connector' => $part5Data['grounded_connector'] ?? 0,
            'dominant_connection_style' => $part5Data['dominant_connection_style'] ?? '',
        ]);

        // Part 6: Intelligence/Quotient - NEW (Checkbox display)
        $part6Data = $scores['part6'] ?? [];
        $data = array_merge($data, [
            'si_category' => $part6Data['si_category'] ?? 'low',
            'eq_category' => $part6Data['eq_category'] ?? 'low',
            'rq_category' => $part6Data['rq_category'] ?? 'low',
            'si_score' => $part6Data['si_score'] ?? 0,
            'eq_score' => $part6Data['eq_score'] ?? 0,
            'rq_score' => $part6Data['rq_score'] ?? 0,
        ]);

        // Part 7: Paradigm Balance - NEW (8 wellness percentages, NO % symbol in display)
        $part7Data = $scores['part7'] ?? [];
        $data = array_merge($data, [
            'emotional_balance' => $part7Data['emotional_balance'] ?? 0,
            'spiritual_balance' => $part7Data['spiritual_balance'] ?? 0,
            'physical_balance' => $part7Data['physical_balance'] ?? 0,
            'social_balance' => $part7Data['social_balance'] ?? 0,
            'financial_balance' => $part7Data['financial_balance'] ?? 0,
            'occupational_balance' => $part7Data['occupational_balance'] ?? 0,
            'intellectual_balance' => $part7Data['intellectual_balance'] ?? 0,
            'environmental_balance' => $part7Data['environmental_balance'] ?? 0,
            'overall_wellness' => $part7Data['overall_wellness'] ?? 0,
        ]);

        // Paradigm Type and Dominance
        $data['paradigm_type'] = $result->paradigm_type ?? 'Not Calculated';
        $data['dominance_category'] = $scores['paradigm_main']['dominance'] ?? 'Not Calculated';
        
        return $data;
    }

    private function processPagesForTemplate(ParadigmFormTemplate $template): array
    {
        $processedPages = [];
        $pages = $template->pdfPages()->with('pdfElements')->orderBy('page_number')->get();

        foreach ($pages as $page) {
            $backgroundImagePath = null;
            if ($page->background_image_path && Storage::disk('public')->exists($page->background_image_path)) {
                $fileContent = Storage::disk('public')->get($page->background_image_path);
                $mimeType = Storage::disk('public')->mimeType($page->background_image_path);
                $backgroundImagePath = 'data:' . $mimeType . ';base64,' . base64_encode($fileContent);
            }

            $processedPages[] = [
                'page_number' => $page->page_number,
                'background_image' => $backgroundImagePath,
                'pdf_elements' => $page->pdfElements->sortBy('display_order')->toArray(),
            ];
        }

        return $processedPages;
    }
}
