<?php

namespace App\Services;

use App\Models\ParadigmFormTemplate;
use Illuminate\Support\Collection;

class ParadigmScoringService
{
    public function processAssessment(ParadigmFormTemplate $template, array $answers, array $personalData): array
    {
        $questions = $template->questions()->get();

        // Part 1: Brain Frames (existing - do not modify)
        $part1Questions = $questions->where('question_type', 'part1');
        $part1Answers = $this->extractModuleAnswers('part1', $answers, $part1Questions);
        $part1Results = $this->processBrainProfile4Frame($part1Answers, $part1Questions);

        // Part 2: Brain Fields (existing - do not modify)
        $part2Questions = $questions->where('question_type', 'part2');
        $part2Answers = $this->extractModuleAnswers('part2', $answers, $part2Questions);
        $part2Results = $this->processBrainProfile8Field($part2Answers, $part2Questions);

        // Part 3: Personality Blueprint (E/I) - NEW
        $part3Questions = $questions->where('question_type', 'part3');
        $part3Answers = $this->extractModuleAnswers('part3', $answers, $part3Questions);
        $part3Results = $this->processPart3($part3Answers, $part3Questions);

        // Part 4: Conflict Styles - NEW
        $part4Questions = $questions->where('question_type', 'part4');
        $part4Answers = $this->extractModuleAnswers('part4', $answers, $part4Questions);
        $part4Results = $this->processPart4($part4Answers, $part4Questions);

        // Part 5: Connection Styles - NEW
        $part5Questions = $questions->where('question_type', 'part5');
        $part5Answers = $this->extractModuleAnswers('part5', $answers, $part5Questions);
        $part5Results = $this->processPart5($part5Answers, $part5Questions);

        // Part 6: Social, Emotional Intelligence & Relevancy Quotient - NEW
        $part6Questions = $questions->whereIn('question_type', ['part6_si', 'part6_eq', 'part6_rq']);
        // Extract answers for Part 6 (all 3 subsections)
        $part6Answers = [];
        foreach ($part6Questions as $question) {
            if (array_key_exists((string)$question->id, $answers)) {
                $part6Answers[$question->id] = $answers[(string)$question->id];
            }
        }
        $part6Results = $this->processPart6($part6Answers, $part6Questions);

        // Part 7: Paradigm Balance - NEW
        $part7Questions = $questions->whereIn('question_type', [
            'part7_emotional', 'part7_spiritual', 'part7_physical', 'part7_social',
            'part7_financial', 'part7_occupational', 'part7_intellectual', 'part7_environmental'
        ]);
        // Extract answers for Part 7 (all 8 dimensions)
        $part7Answers = [];
        foreach ($part7Questions as $question) {
            if (array_key_exists((string)$question->id, $answers)) {
                $part7Answers[$question->id] = $answers[(string)$question->id];
            }
        }
        $part7Results = $this->processPart7($part7Answers, $part7Questions);

        // Paradigm Main Results (Part 1 & 2 - existing)
        $paradigmMainResults = [
            'scores' => [
                'part1' => $part1Results['scores'],
                'part2' => $part2Results['scores'],
            ],
            'percentages' => [
                'part1' => $part1Results['percentages'],
                'part2' => $part2Results['percentages'],
            ],
            'dominance' => $part1Results['dominance'],
        ];

        return [
            'personal_data' => $personalData,
            'scores' => [
                'paradigm_main' => $paradigmMainResults,
                'part3' => $part3Results, // NEW: Part 3 results
                'part4' => $part4Results, // NEW: Part 4 results
                'part5' => $part5Results, // NEW: Part 5 results
                'part6' => $part6Results, // NEW: Part 6 results
                'part7' => $part7Results, // NEW: Part 7 results
            ],
            'overall_type' => $part1Results['types'],
            'completed_at' => now(),
            'session_id' => 'SES-' . uniqid()
        ];
    }

    private function extractModuleAnswers(string $type, array $allAnswers, Collection $allQuestions): array
    {
        $moduleQuestions = $allQuestions->where('question_type', $type);
        $moduleAnswers = [];
        foreach ($moduleQuestions as $question) {
            // Explicitly check for string key from JSON, which is the root of the problem
            if (array_key_exists((string)$question->id, $allAnswers)) {
                $moduleAnswers[$question->id] = $allAnswers[(string)$question->id];
            }
        }
        return $moduleAnswers;
    }

    private function processBrainProfile4Frame(array $answers, Collection $questions): array
    {
        $scores = ['LU' => 0, 'LL' => 0, 'RU' => 0, 'RL' => 0];
        $pointMap = [1 => 3, 2 => 2, 3 => 1, 4 => 0];

        foreach ($answers as $questionId => $ranking) {
            $question = $questions->find($questionId);
            if (!$question) continue;

            if (count($ranking) === 3) {
                $allOptions = ['A', 'B', 'C', 'D'];
                $rankedOptions = array_keys($ranking);
                $unrankedOption = array_diff($allOptions, $rankedOptions);
                if (!empty($unrankedOption)) {
                    $unrankedKey = array_pop($unrankedOption);
                    $ranking[$unrankedKey] = 4;
                }
            }

            foreach ($ranking as $option => $rank) {
                $sideColumn = strtolower($option) . '_side';
                $quadrant = $question->{$sideColumn};
                if (isset($scores[$quadrant]) && isset($pointMap[$rank])) {
                    $scores[$quadrant] += $pointMap[$rank];
                }
            }
        }

        $percentages = [];
        foreach ($scores as $quadrant => $score) {
            $percentages[$quadrant] = round(($score / 90) * 100, 2);
        }

        $dominanceResult = $this->determineDominance($percentages);
        $types = $this->mapDominanceToType($dominanceResult['dominant_frames'], $percentages);

        return [
            'scores' => $scores,
            'percentages' => $percentages,
            'dominance' => $dominanceResult['class'],
            'types' => $types,
        ];
    }

    private function processBrainProfile8Field(array $answers, Collection $questions): array
    {
        $scores = [
            'LU – The Question Asker' => 0, 'LU – The Fact Finder' => 0,
            'LL – The Rule Keeper' => 0, 'LL – The Task Manager' => 0,
            'RU – The Creator' => 0, 'RU – The Big Thinker' => 0,
            'RL – The Friend Maker' => 0, 'RL – The Empath' => 0,
        ];

        foreach ($answers as $questionId => $choice) {
            $question = $questions->find($questionId);
            if ($question) {
                $sideColumn = strtolower($choice) . '_side';
                $field = $question->{$sideColumn};
                if (array_key_exists($field, $scores)) {
                    $scores[$field]++;
                }
            }
        }

        $percentages = [];
        $totalQuestionsPerField = 16; // Each field appears 16 times across 64 questions
        foreach ($scores as $field => $score) {
            $percentages[$field] = round(($score / $totalQuestionsPerField) * 100);
        }

        return [
            'scores' => $scores,
            'percentages' => $percentages,
        ];
    }
    
    private function mapDominanceToType(array $dominantFrames, array $percentages): string
    {
        // Get bracket for each frame
        $getBracket = function($percentage) {
            if ($percentage >= 95) return 'Very High';
            if ($percentage >= 80) return 'High';
            if ($percentage >= 65) return 'Average';
            if ($percentage >= 50) return 'Low';
            return 'Very Low';
        };
        
        // Find the highest scoring frame's bracket
        $highestFrame = array_keys($percentages, max($percentages))[0];
        $highestBracket = $getBracket($percentages[$highestFrame]);
        
        // Get frames that are in the same bracket as the highest
        $sameBracketFrames = [];
        foreach ($dominantFrames as $frame) {
            if ($getBracket($percentages[$frame]) === $highestBracket) {
                $sameBracketFrames[] = $frame;
            }
        }
        
        // Sort and create key for type mapping
        sort($sameBracketFrames);
        $key = implode(' + ', $sameBracketFrames);
        
        $typeMap = [
            'LU' => 'The Analyst', 'LL' => 'The Organiser', 'RU' => 'The Visionary', 'RL' => 'The Helper',
            'LL + LU' => 'The Specialist', 'LU + RU' => 'The Strategist', 'LU + RL' => 'The Advisor',
            'LL + RU' => 'The Architect', 'LL + RL' => 'The Harmoniser', 'RL + RU' => 'The Creator',
            'LL + LU + RU' => 'The Innovator', 'LL + LU + RL' => 'The Stabiliser',
            'LU + RL + RU' => 'The Integrator', 'LL + RL + RU' => 'The Vision Builder',
            'LL + LU + RL + RU' => 'The Whole-Brain Thinker',
        ];
        return $typeMap[$key] ?? 'Unknown';
    }

    private function determineDominance(array $percentages): array
    {
        if (empty($percentages)) {
            return ['class' => 'Not Calculated', 'dominant_frames' => []];
        }
        
        // Get bracket for each frame
        $getBracket = function($percentage) {
            if ($percentage >= 95) return 'Very High';
            if ($percentage >= 80) return 'High';
            if ($percentage >= 65) return 'Average';
            if ($percentage >= 50) return 'Low';
            return 'Very Low';
        };
        
        // Find frames that are 50% or above (dominant frames)
        $dominantFrames = [];
        $frameBrackets = [];
        foreach ($percentages as $frame => $percentage) {
            if ($percentage >= 50) {
                $dominantFrames[] = $frame;
                $frameBrackets[$frame] = $getBracket($percentage);
            }
        }
        
        // Find the highest scoring frame's bracket
        $highestFrame = array_keys($percentages, max($percentages))[0];
        $highestBracket = $getBracket($percentages[$highestFrame]);
        
        // Count how many dominant frames are in the same bracket as the highest
        $sameBracketCount = 0;
        foreach ($dominantFrames as $frame) {
            if ($frameBrackets[$frame] === $highestBracket) {
                $sameBracketCount++;
            }
        }
        
        // Determine dominance class based on bracket logic
        $dominanceClass = '';
        if ($sameBracketCount === 1) {
            $dominanceClass = 'Single-Frame Dominance';
        } elseif ($sameBracketCount === 2) {
            $dominanceClass = 'Dual-Frame Dominance';
        } elseif ($sameBracketCount === 3) {
            $dominanceClass = 'Triple-Frame Dominance';
        } elseif ($sameBracketCount === 4) {
            $dominanceClass = 'All-Frame Dominance';
        } else {
            $dominanceClass = 'Not Calculated';
        }
        
        return ['class' => $dominanceClass, 'dominant_frames' => $dominantFrames];
    }

    /**
     * Process Part 3: Personality Blueprint (E/I Trait)
     * SAFE: Only adds new scoring logic, does not modify Part 1 or Part 2
     * 
     * Returns PERCENTAGES (not raw counts)
     */
    private function processPart3(array $answers, Collection $questions): array
    {
        $eCount = 0;
        $iCount = 0;
        $totalQuestions = $questions->count();
        
        foreach ($questions as $question) {
            $answer = $answers[$question->id] ?? null;
            
            if ($answer === 'A' && $question->a_side === 'E') {
                $eCount++;
            } elseif ($answer === 'B' && $question->b_side === 'I') {
                $iCount++;
            }
        }
        
        // Calculate percentages
        $ePercentage = $totalQuestions > 0 ? round(($eCount / $totalQuestions) * 100) : 0;
        $iPercentage = $totalQuestions > 0 ? round(($iCount / $totalQuestions) * 100) : 0;
        
        // Determine personality type (E or I)
        $personalityType = $eCount > $iCount ? 'E' : 'I';
        
        return [
            'e_score' => $ePercentage,      // Now returns 70 instead of 7
            'i_score' => $iPercentage,      // Now returns 30 instead of 3
            'personality_type' => $personalityType
        ];
    }

    /**
     * Process Part 4: Conflict Styles
     * SAFE: Only adds new scoring logic, does not modify existing parts
     * 
     * Returns RANKS (1-5), not raw counts
     */
    private function processPart4(array $answers, Collection $questions): array
    {
        $styles = [
            'Assertive Challenger' => 0,
            'Integrative Bridge Builder' => 0,
            'Fair Negotiator' => 0,
            'Quiet Protector' => 0,
            'Peacekeeper' => 0
        ];
        
        foreach ($questions as $question) {
            $answer = $answers[$question->id] ?? null;
            
            if ($answer === 'A' && isset($styles[$question->a_side])) {
                $styles[$question->a_side]++;
            } elseif ($answer === 'B' && isset($styles[$question->b_side])) {
                $styles[$question->b_side]++;
            }
        }
        
        // Sort by count (descending) and assign ranks 1-5
        arsort($styles);
        $rank = 1;
        $ranked = [];
        foreach ($styles as $styleName => $count) {
            $ranked[$styleName] = $rank++;
        }
        
        // Find dominant style (rank 1)
        $dominantStyle = array_keys($styles)[0];
        
        return [
            'assertive_challenger' => $ranked['Assertive Challenger'],
            'integrative_bridge_builder' => $ranked['Integrative Bridge Builder'],
            'fair_negotiator' => $ranked['Fair Negotiator'],
            'quiet_protector' => $ranked['Quiet Protector'],
            'peacekeeper' => $ranked['Peacekeeper'],
            'dominant_conflict_style' => $dominantStyle
        ];
    }

    /**
     * Process Part 5: Connection Styles
     * SAFE: Only adds new scoring logic, does not modify existing parts
     * 
     * Note: Part 5 uses letters A/B/C/D/E to represent connection styles
     * Returns RANKS (1-5), not raw counts
     */
    private function processPart5(array $answers, Collection $questions): array
    {
        // Count each letter (connection style)
        $letterCounts = [
            'A' => 0, // Verbal Validator
            'B' => 0, // Presence Seeker
            'C' => 0, // Symbolic Appreciator
            'D' => 0, // Practical Supporter
            'E' => 0, // Grounded Connector
        ];
        
        foreach ($questions as $question) {
            $answer = $answers[$question->id] ?? null;
            
            if ($answer === 'A') {
                $letterCounts[$question->a_side]++;
            } elseif ($answer === 'B') {
                $letterCounts[$question->b_side]++;
            }
        }
        
        // Sort by count (descending) and assign ranks 1-5
        arsort($letterCounts);
        $styleNames = [
            'A' => 'Verbal Validator',
            'B' => 'Presence Seeker',
            'C' => 'Symbolic Appreciator',
            'D' => 'Practical Supporter',
            'E' => 'Grounded Connector',
        ];
        
        $rank = 1;
        $ranked = [];
        foreach ($letterCounts as $letter => $count) {
            $ranked[$styleNames[$letter]] = $rank++;
        }
        
        // Find dominant style (rank 1)
        $dominantLetter = array_keys($letterCounts)[0];
        
        return [
            'verbal_validator' => $ranked['Verbal Validator'],
            'presence_seeker' => $ranked['Presence Seeker'],
            'symbolic_appreciator' => $ranked['Symbolic Appreciator'],
            'practical_supporter' => $ranked['Practical Supporter'],
            'grounded_connector' => $ranked['Grounded Connector'],
            'dominant_connection_style' => $styleNames[$dominantLetter] ?? ''
        ];
    }

    /**
     * Process Part 6: Social, Emotional Intelligence & Relevancy Quotient
     * SAFE: Only adds new scoring logic, does not modify existing parts
     * 
     * Scoring: A=3pts, B=2pts, C=1pt
     * 3 Sections with different max scores and ranges
     */
    private function processPart6(array $answers, Collection $questions): array
    {
        // Separate questions by section
        $siQuestions = $questions->where('question_type', 'part6_si');
        $eqQuestions = $questions->where('question_type', 'part6_eq');
        $rqQuestions = $questions->where('question_type', 'part6_rq');
        
        // Calculate scores (A=3, B=2, C=1)
        $siScore = 0;
        foreach ($siQuestions as $q) {
            $ans = $answers[$q->id] ?? null;
            $siScore += $ans === 'A' ? 3 : ($ans === 'B' ? 2 : ($ans === 'C' ? 1 : 0));
        }
        
        $eqScore = 0;
        foreach ($eqQuestions as $q) {
            $ans = $answers[$q->id] ?? null;
            $eqScore += $ans === 'A' ? 3 : ($ans === 'B' ? 2 : ($ans === 'C' ? 1 : 0));
        }
        
        $rqScore = 0;
        foreach ($rqQuestions as $q) {
            $ans = $answers[$q->id] ?? null;
            $rqScore += $ans === 'A' ? 3 : ($ans === 'B' ? 2 : ($ans === 'C' ? 1 : 0));
        }
        
        // Determine categories
        // SI: Max 36 (12 questions × 3) - High: 30-36, Medium: 20-29, Low: 12-19
        $siCategory = $siScore >= 30 ? 'high' : ($siScore >= 20 ? 'medium' : 'low');
        
        // EQ: Max 60 (20 questions × 3) - High: 50-60, Medium: 35-49, Low: 20-34
        $eqCategory = $eqScore >= 50 ? 'high' : ($eqScore >= 35 ? 'medium' : 'low');
        
        // RQ: Max 36 (12 questions × 3) - High: 30-36, Medium: 22-29, Low: 12-21
        $rqCategory = $rqScore >= 30 ? 'high' : ($rqScore >= 22 ? 'medium' : 'low');
        
        return [
            'si_score' => $siScore,
            'si_category' => $siCategory,
            'eq_score' => $eqScore,
            'eq_category' => $eqCategory,
            'rq_score' => $rqScore,
            'rq_category' => $rqCategory,
        ];
    }

    /**
     * Process Part 7: Paradigm Balance (8 Wellness Dimensions)
     * SAFE: Only adds new scoring logic, does not modify existing parts
     * 
     * Rating Scale: 1-4 for each question
     * Calculation: (Sum / 40) × 100 = Percentage per dimension
     */
    private function processPart7(array $answers, Collection $questions): array
    {
        $dimensions = [
            'part7_emotional' => 'emotional_balance',
            'part7_spiritual' => 'spiritual_balance',
            'part7_physical' => 'physical_balance',
            'part7_social' => 'social_balance',
            'part7_financial' => 'financial_balance',
            'part7_occupational' => 'occupational_balance',
            'part7_intellectual' => 'intellectual_balance',
            'part7_environmental' => 'environmental_balance',
        ];
        
        $results = [];
        $totalPercentage = 0;
        
        foreach ($dimensions as $questionType => $dimensionKey) {
            $dimensionQuestions = $questions->where('question_type', $questionType);
            $score = 0;
            
            foreach ($dimensionQuestions as $q) {
                $rating = (int)($answers[$q->id] ?? 0);
                $score += $rating;
            }
            
            // Calculate percentage: (score / 40) × 100
            $percentage = $score > 0 ? round(($score / 40) * 100) : 0;
            $results[$dimensionKey] = $percentage;
            $totalPercentage += $percentage;
        }
        
        // Calculate overall wellness score (average of 8 dimensions)
        $results['overall_wellness'] = round($totalPercentage / 8);
        
        return $results;
    }
}