<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CompleteTemplateCleanupAndUpdateSeeder extends Seeder
{
    /**
     * Complete Templates Cleanup and Update
     * 
     * COMPREHENSIVE ACTIONS:
     * 1. PERMANENTLY delete all soft-deleted templates
     * 2. Delete duplicate active templates (keep NEWEST per notification_type)
     * 3. UPDATE Type 31 (Professional Details Verified) with new content
     * 4. UPDATE Type 6 (Identity Verification Completed) with new content
     * 
     * This is a COMPLETE cleanup and update in one seeder.
     */
    public function run(): void
    {
        $this->command->info('========================================');
        $this->command->info('🧹 COMPLETE TEMPLATE CLEANUP & UPDATE');
        $this->command->info('========================================');
        $this->command->info('');
        
        try {
            DB::beginTransaction();
            
            // ============================================
            // STEP 1: PERMANENTLY DELETE ALL SOFT-DELETED TEMPLATES
            // ============================================
            $this->command->info('📋 STEP 1: Removing soft-deleted templates...');
            
            $softDeleted = DB::table('templates')
                ->whereNotNull('deleted_at')
                ->count();
            
            if ($softDeleted > 0) {
                DB::table('templates')
                    ->whereNotNull('deleted_at')
                    ->delete();
                
                $this->command->info("   ✅ Permanently deleted {$softDeleted} soft-deleted template(s)");
            } else {
                $this->command->line('   ℹ️  No soft-deleted templates');
            }
            
            $this->command->info('');
            
            // ============================================
            // STEP 2: DELETE ACTIVE DUPLICATES
            // ============================================
            $this->command->info('📋 STEP 2: Removing duplicate active templates...');
            $this->command->line('   Strategy: Keep NEWEST (highest ID) per notification_type');
            $this->command->info('');
            
            $templates = DB::table('templates')
                ->where('type', 1)
                ->where('status', 1)
                ->whereNull('deleted_at')
                ->orderBy('notification_type')
                ->orderBy('id', 'desc')
                ->get(['id', 'notification_type', 'title']);
            
            $grouped = [];
            foreach ($templates as $t) {
                if (!isset($grouped[$t->notification_type])) {
                    $grouped[$t->notification_type] = [];
                }
                $grouped[$t->notification_type][] = $t;
            }
            
            $totalDeleted = 0;
            
            foreach ($grouped as $notifType => $temps) {
                if (count($temps) > 1) {
                    $keepId = $temps[0]->id;
                    $deleteIds = array_slice(array_column($temps, 'id'), 1);
                    
                    if (!empty($deleteIds)) {
                        $deleted = DB::table('templates')
                            ->whereIn('id', $deleteIds)
                            ->delete();
                        
                        $totalDeleted += $deleted;
                        $this->command->line("   Type {$notifType}: Kept ID {$keepId}, deleted " . count($deleteIds));
                    }
                }
            }
            
            if ($totalDeleted > 0) {
                $this->command->info("   ✅ Deleted {$totalDeleted} duplicate(s)");
            } else {
                $this->command->line('   ℹ️  No duplicates found');
            }
            
            $this->command->info('');
            
            // ============================================
            // STEP 3: UPDATE TYPE 31 (PROFESSIONAL DETAILS)
            // ============================================
            $this->command->info('📋 STEP 3: Updating Professional Details template (Type 31)...');
            
            $type31 = DB::table('templates')
                ->where('notification_type', 31)
                ->where('type', 1)
                ->where('status', 1)
                ->whereNull('deleted_at')
                ->first();
            
            if ($type31) {
                DB::table('templates')
                    ->where('id', $type31->id)
                    ->update([
                        'title' => 'Professional Details Verified',
                        'subject' => 'Professional Details Verified - {{company_name}}',
                        'content' => $this->getProfessionalDetailsTemplate(),
                        'updated_at' => Carbon::now(),
                    ]);
                
                $this->command->info("   ✅ Updated Template ID {$type31->id}");
                $this->command->line('      Title: Professional Details Verified');
                $this->command->line('      Button: Login → https://app.iseekhelpoffice.com');
            } else {
                $this->command->warn('   ⚠️  Type 31 template not found');
            }
            
            $this->command->info('');
            
            // ============================================
            // STEP 4: UPDATE TYPE 6 (IDENTITY VERIFICATION)
            // ============================================
            $this->command->info('📋 STEP 4: Updating Identity Verification template (Type 6)...');
            
            $type6 = DB::table('templates')
                ->where('notification_type', 6)
                ->where('type', 1)
                ->where('status', 1)
                ->whereNull('deleted_at')
                ->first();
            
            if ($type6) {
                DB::table('templates')
                    ->where('id', $type6->id)
                    ->update([
                        'title' => 'Identity Verification Completed',
                        'subject' => 'Identity Verification Completed - {{company_name}}',
                        'content' => $this->getIdentityVerificationTemplate(),
                        'updated_at' => Carbon::now(),
                    ]);
                
                $this->command->info("   ✅ Updated Template ID {$type6->id}");
                $this->command->line('      Title: Identity Verification Completed');
                $this->command->line('      Removed: Activated Date, Start building...');
                $this->command->line('      Button: Login → https://app.iseekhelpoffice.com');
            } else {
                $this->command->warn('   ⚠️  Type 6 template not found');
            }
            
            DB::commit();
            
            $this->command->info('');
            $this->command->info('========================================');
            $this->command->info('✅ COMPLETE CLEANUP FINISHED!');
            $this->command->info('========================================');
            $this->command->info("📊 Summary:");
            $this->command->line("   • Soft-deleted removed: {$softDeleted}");
            $this->command->line("   • Duplicates removed: {$totalDeleted}");
            $this->command->line("   • Templates updated: 2 (Type 31 & Type 6)");
            $this->command->info('');
            $this->command->info('✅ Templates table is now clean and updated!');
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error('❌ Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Professional Details Verified Template (Type 31)
     */
    private function getProfessionalDetailsTemplate(): string
    {
        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Professional Details Verified!</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f4f4f4; }
        .container { max-width: 600px; margin: 20px auto; background: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .header { background: linear-gradient(135deg, #044f54 0%, #066a70 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
        .header h1 { margin: 0; font-size: 24px; font-weight: 600; }
        .content { padding: 30px 20px; }
        .content h2 { color: #044f54; margin-top: 0; font-size: 20px; }
        .content p { margin: 15px 0; color: #555; }
        .button-container { text-align: center; margin: 30px 0; }
        .button { display: inline-block; padding: 14px 40px; background: #044f54; color: #ffffff !important; text-decoration: none; border-radius: 5px; font-weight: 600; font-size: 16px; transition: background 0.3s ease; }
        .button:hover { background: #066a70; }
        .info-box { background: #f8f9fa; border-left: 4px solid #044f54; padding: 15px; margin: 20px 0; border-radius: 4px; }
        .info-box p { margin: 5px 0; font-size: 14px; }
        .footer { background: #f8f9fa; padding: 20px; text-align: center; font-size: 13px; color: #6c757d; border-top: 1px solid #e9ecef; }
        .footer a { color: #044f54; text-decoration: none; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Professional Details Verified!</h1>
        </div>
        <div class="content">
            <h2>Congratulations {{provider_name}}!</h2>
            
            <p>Your professional details have been verified on I Seek Help International.</p>
            
            <div class="info-box">
                <p><strong>Account Status:</strong> ✅ Professional Details Verified</p>
            </div>
            
            <p>You can now access your provider dashboard and complete your identity verification to start offering services.</p>
            
            <div class="button-container">
                <a href="https://app.iseekhelpoffice.com" class="button">Login</a>
            </div>
        </div>
        <div class="footer">
            <p><strong>{{company_name}}</strong></p>
            <p>{{company_email}} | {{company_phone}}</p>
            <p><a href="{{website_link}}">Visit our website</a></p>
            <p style="margin-top: 15px; color: #999;">This is an automated email. Please do not reply to this message.</p>
        </div>
    </div>
</body>
</html>';
    }

    /**
     * Identity Verification Completed Template (Type 6)
     */
    private function getIdentityVerificationTemplate(): string
    {
        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Identity Verification Completed!</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f4f4f4; }
        .container { max-width: 600px; margin: 20px auto; background: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .header { background: linear-gradient(135deg, #044f54 0%, #066a70 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
        .header h1 { margin: 0; font-size: 24px; font-weight: 600; }
        .content { padding: 30px 20px; }
        .content h2 { color: #044f54; margin-top: 0; font-size: 20px; }
        .content p { margin: 15px 0; color: #555; }
        .button-container { text-align: center; margin: 30px 0; }
        .button { display: inline-block; padding: 14px 40px; background: #044f54; color: #ffffff !important; text-decoration: none; border-radius: 5px; font-weight: 600; font-size: 16px; transition: background 0.3s ease; }
        .button:hover { background: #066a70; }
        .info-box { background: #f8f9fa; border-left: 4px solid #044f54; padding: 15px; margin: 20px 0; border-radius: 4px; }
        .info-box p { margin: 5px 0; font-size: 14px; }
        .footer { background: #f8f9fa; padding: 20px; text-align: center; font-size: 13px; color: #6c757d; border-top: 1px solid #e9ecef; }
        .footer a { color: #044f54; text-decoration: none; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Identity Verification Completed!</h1>
        </div>
        <div class="content">
            <h2>Congratulations {{provider_name}}!</h2>
            
            <p>Your identity verification has been completed successfully. You can now start offering your services on I Seek Help International.</p>
            
            <div class="info-box">
                <p><strong>Account Status:</strong> ✅ Verified</p>
            </div>
            
            <div class="button-container">
                <a href="https://app.iseekhelpoffice.com" class="button">Login</a>
            </div>
        </div>
        <div class="footer">
            <p><strong>{{company_name}}</strong></p>
            <p>{{company_email}} | {{company_phone}}</p>
            <p><a href="{{website_link}}">Visit our website</a></p>
            <p style="margin-top: 15px; color: #999;">This is an automated email. Please do not reply to this message.</p>
        </div>
    </div>
</body>
</html>';
    }
}

