<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Models\UserDetail;
use App\Models\Branches;

class ConvertExistingProvidersToCompanyOwnersSeeder extends Seeder
{
    /**
     * Run the database seeder.
     */
    public function run(): void
    {
        $this->command->info('🔄 Starting conversion of existing providers with branches...');
        echo "\n";
        
        DB::beginTransaction();
        
        try {
            // Check if required columns exist
            if (!DB::getSchemaBuilder()->hasColumn('users', 'is_company_owner')) {
                $this->command->error('❌ ERROR: is_company_owner column does not exist!');
                $this->command->warn('💡 Run migrations first: php artisan migrate');
                DB::rollBack();
                return;
            }
            
            if (!DB::getSchemaBuilder()->hasColumn('branches', 'is_main_branch')) {
                $this->command->error('❌ ERROR: is_main_branch column does not exist!');
                $this->command->warn('💡 Run migrations first: php artisan migrate');
                DB::rollBack();
                return;
            }
            
            // Debug: Check what we have
            $allProviders = User::where('user_type', 2)->count();
            $allBranches = Branches::whereNull('deleted_at')->count();
            $this->command->line("📊 Total Providers (user_type = 2): {$allProviders}");
            $this->command->line("📊 Total Branches: {$allBranches}");
            echo "\n";
            
            // Find ALL providers (individual OR company) who have at least 1 branch
            // Note: Only user_type = 2 (providers), NOT user_type = 4 (already converted)
            $providers = User::where('user_type', 2)
                ->get()
                ->filter(function ($provider) {
                    $count = Branches::where('created_by', $provider->id)->whereNull('deleted_at')->count();
                    echo "   Checking Provider ID {$provider->id} ({$provider->email}): {$count} branches\n";
                    return $count > 0;
                });
            
            if ($providers->isEmpty()) {
                $this->command->warn('⚠️  No providers with branches found.');
                DB::rollBack();
                return;
            }
            
            $this->command->info("📋 Found {$providers->count()} providers with branches to convert");
            echo "\n";
            
            $converted = 0;
            
            foreach ($providers as $provider) {
                $branchCount = Branches::where('created_by', $provider->id)->whereNull('deleted_at')->count();
                
                $this->command->info("👤 Processing: {$provider->name} (ID: {$provider->id})");
                $this->command->line("   Email: {$provider->email}");
                $this->command->line("   Current user_type: {$provider->user_type}");
                $this->command->line("   Branches: {$branchCount}");
                
                // Convert to company owner staff
                DB::table('users')->where('id', $provider->id)->update([
                    'user_type' => 4,
                    'is_company_owner' => 1,
                    'updated_at' => now()
                ]);
                
                $this->command->line("   ✅ Converted to company owner staff (user_type: {$provider->user_type} → 4)");
                $this->command->line("   ✅ Set is_company_owner = TRUE");
                
                // Set parent_id to self (company owner)
                $userDetail = UserDetail::where('user_id', $provider->id)->first();
                
                if ($userDetail) {
                    DB::table('user_details')->where('user_id', $provider->id)->update([
                        'parent_id' => $provider->id
                    ]);
                    $this->command->line("   ✅ Set parent_id to self ({$provider->id})");
                } else {
                    $this->command->warn("   ⚠️  UserDetail not found for user {$provider->id}");
                }
                
                // Mark first branch as main branch (if not already marked)
                $mainBranchExists = Branches::where('created_by', $provider->id)
                    ->where('is_main_branch', true)
                    ->whereNull('deleted_at')
                    ->exists();
                
                if (!$mainBranchExists) {
                    $firstBranch = Branches::where('created_by', $provider->id)
                        ->whereNull('deleted_at')
                        ->orderBy('id', 'asc')
                        ->first();
                    
                    if ($firstBranch) {
                        DB::table('branches')->where('id', $firstBranch->id)->update([
                            'is_main_branch' => 1
                        ]);
                        $this->command->line("   ✅ Set main branch: {$firstBranch->branch_name} (ID: {$firstBranch->id})");
                    }
                } else {
                    $this->command->line("   ✓ Main branch already set");
                }
                
                $converted++;
                echo "\n";
                
                Log::info("Converted provider {$provider->id} ({$provider->email}) to company owner staff with {$branchCount} branches");
            }
            
            DB::commit();
            
            echo str_repeat("━", 60) . "\n";
            $this->command->info("✅ CONVERSION COMPLETE!");
            $this->command->info("📊 Converted {$converted} providers to company owner staff");
            echo str_repeat("━", 60) . "\n";
            echo "\n";
            
            $this->command->warn("⚠️  IMPORTANT NOTES:");
            $this->command->line("   • These users are now staff (user_type = 4) but marked as company owners");
            $this->command->line("   • They keep all existing data (bookings, wallet, credits, paradigm profiles)");
            $this->command->line("   • Their parent_id points to themselves (self-reference)");
            $this->command->line("   • Their first branch is marked as main branch");
            $this->command->line("   • They can still manage everything as before");
            $this->command->line("   • Profile will now show company details first");
            $this->command->line("   • Paradigm reports will show company name");
            echo "\n";
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error('❌ ERROR during conversion: ' . $e->getMessage());
            $this->command->error('   Stack trace: ' . $e->getTraceAsString());
            Log::error('Seeder error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }
}

