<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\WalletHistory;
use App\Models\ParadigmPayment;
use App\Models\ParadigmAccessToken;
use Illuminate\Support\Facades\DB;

class CreditMayurAndCleanupSeeder extends Seeder
{
    public function run()
    {
        $providerEmail = 'hayley@urbanbalancecounselling.co.za';
        $clientEmail = 'mayurbhana2018@gmail.com';
        
        $this->command->info("========================================");
        $this->command->info("PRACTITIONER: {$providerEmail}");
        $this->command->info("CLIENT: {$clientEmail}");
        $this->command->info("========================================\n");
        
        // ========================================
        // STEP 1: Add 15 Credits to PRACTITIONER (Hayley)
        // ========================================
        
        $provider = User::where('email', $providerEmail)->first();
        
        if (!$provider) {
            $this->command->error("❌ Provider with email {$providerEmail} not found!");
            return;
        }
        
        $this->command->info("✅ Found PRACTITIONER: {$provider->name} (ID: {$provider->id})");
        
        // Add 15 credits to provider's wallet
        WalletHistory::create([
            'user_id' => $provider->id,
            'amount' => 15.00,
            'credits' => 15,
            'payment_type' => 'Others', // Valid ENUM value: Paypal, Credit Card, Bank Transfer, Others
            'status' => 'Completed',
            'transaction_id' => 'ADMIN-CREDIT-' . time(),
            'transaction_date' => now(),
            'type' => 1, // Type 1 = Credit
        ]);
        
        $this->command->info("✅ Added 15 credits to PRACTITIONER's wallet");
        
        // Calculate new balance
        $totalCredits = WalletHistory::where('user_id', $provider->id)
            ->where('status', 'Completed')
            ->where('type', '1')
            ->get()
            ->sum(function($record) {
                return $record->credits ?? $record->amount;
            });
        
        $totalDebits = WalletHistory::where('user_id', $provider->id)
            ->where('status', 'Completed')
            ->where('type', '2')
            ->sum('amount');
        
        $balance = $totalCredits - $totalDebits;
        
        $this->command->info("💰 Practitioner's New Balance: {$balance} credits");
        
        // ========================================
        // STEP 2: Remove ONLY "Client Pays" + "Pending Payment" Record
        // ========================================
        
        $this->command->info("\n--- Searching for CLIENT's 'Client Pays' assignment to delete ---");
        
        // Find the EXACT record to delete
        // IMPORTANT: This is an assignment from Hayley (provider) TO Mayur (client) where client hasn't paid yet
        $recordToDelete = ParadigmPayment::where('provider_id', $provider->id)
            ->where('client_email', $clientEmail)
            ->where('paid_by', 'client')
            ->where('status', 'pending')
            ->first();
        
        if (!$recordToDelete) {
            $this->command->warn("⚠️  No 'Client Pays' + 'Pending Payment' record found");
            $this->command->warn("    Provider: {$providerEmail}");
            $this->command->warn("    Client: {$clientEmail}");
        } else {
            $this->command->info("Found record to DELETE:");
            $this->command->info("  - ID: {$recordToDelete->id}");
            $this->command->info("  - Template: " . ($recordToDelete->template->name ?? 'N/A'));
            $this->command->info("  - Provider: {$providerEmail}");
            $this->command->info("  - Client Email: {$recordToDelete->client_email}");
            $this->command->info("  - Payment Status: {$recordToDelete->paid_by}");
            $this->command->info("  - Assessment Status: {$recordToDelete->status}");
            $this->command->info("  - Created At: {$recordToDelete->created_at}");
            
            // STRICT SAFETY CHECK - Verify it's NOT the "Paid by Provider" record
            if ($recordToDelete->paid_by === 'provider') {
                $this->command->error("❌ ABORT! This is a 'Paid by Provider' record. Will NOT delete.");
                return;
            }
            
            if ($recordToDelete->status !== 'pending') {
                $this->command->error("❌ ABORT! Status is not 'pending'. Will NOT delete.");
                return;
            }
            
            // Delete associated access token first (if exists)
            $deletedTokens = ParadigmAccessToken::where('payment_id', $recordToDelete->id)->delete();
            if ($deletedTokens > 0) {
                $this->command->info("✅ Deleted {$deletedTokens} access token(s)");
            }
            
            // Delete the payment record
            $recordToDelete->delete();
            
            $this->command->info("✅ DELETED 'Client Pays' + 'Pending Payment' record (ID: {$recordToDelete->id})");
        }
        
        // ========================================
        // STEP 3: VERIFY "Paid by Provider" Record is UNTOUCHED
        // ========================================
        
        $this->command->info("\n--- Verifying 'Paid by Provider' record is INTACT ---");
        
        $providerPaidRecords = ParadigmPayment::where('provider_id', $provider->id)
            ->where('client_email', $clientEmail)
            ->where('paid_by', 'provider')
            ->get();
        
        if ($providerPaidRecords->count() > 0) {
            $this->command->info("✅ VERIFIED: 'Paid by Provider' records are INTACT:");
            foreach ($providerPaidRecords as $record) {
                $this->command->info("  - ID: {$record->id}");
                $this->command->info("    Template: " . ($record->template->name ?? 'N/A'));
                $this->command->info("    Provider: {$providerEmail}");
                $this->command->info("    Client: {$clientEmail}");
                $this->command->info("    Payment Status: {$record->paid_by}");
                $this->command->info("    Assessment Status: {$record->status}");
                $this->command->info("    Created At: {$record->created_at}");
            }
        } else {
            $this->command->warn("⚠️  No 'Paid by Provider' records found");
            $this->command->warn("    Provider: {$providerEmail}");
            $this->command->warn("    Client: {$clientEmail}");
        }
        
        // ========================================
        // SUMMARY
        // ========================================
        
        $this->command->info("\n========================================");
        $this->command->info("✅ SEEDER COMPLETED SUCCESSFULLY!");
        $this->command->info("========================================");
        $this->command->info("PRACTITIONER: {$provider->name} ({$providerEmail})");
        $this->command->info("CLIENT: {$clientEmail}");
        $this->command->info("Credits Added to Practitioner: 15");
        $this->command->info("Practitioner's New Balance: {$balance} credits");
        $this->command->info("Records Deleted: " . ($recordToDelete ? "1 ('Client Pays' + 'Pending Payment')" : "0"));
        $this->command->info("'Paid by Provider' Records: PROTECTED & INTACT ✅");
        $this->command->info("========================================\n");
    }
}

