<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\UserDetail;
use App\Models\ProviderDetail;
use App\Models\Bookings;
use App\Models\PackageTrx;
use App\Models\ProviderIdentityVerification;
use App\Models\StaffDocument;
use Modules\Product\app\Models\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DeleteProviderUserSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * This seeder deletes a provider user and all related records.
     * Email: sibanyonibathobile@gmail.com
     * 
     * Run with: php artisan db:seed --class=DeleteProviderUserSeeder
     */
    public function run(): void
    {
        $email = 'sibanyonibathobile@gmail.com';
        
        DB::beginTransaction();
        
        try {
            // Find user including soft deleted ones
            $user = User::withTrashed()->where('email', $email)->first();
            
            if (!$user) {
                echo "❌ User with email {$email} not found.\n";
                DB::rollBack();
                return;
            }
            
            $userId = $user->id;
            $userName = $user->name;
            $userType = $user->user_type;
            
            echo "👤 Found user: {$userName} (ID: {$userId})\n";
            echo "📋 User Type: {$userType}\n";
            echo "📧 Email: {$email}\n";
            echo "⚠️  WARNING: This will permanently delete the user and ALL related data!\n";
            echo "🚀 Starting deletion process...\n\n";
            
            $deletedCount = [];
            
            // Simple hard delete - just delete main related tables
            try {
                // 1. Provider Details
                DB::table('provider_details')->where('user_id', $userId)->delete();
                echo "🗑️  Deleted provider details\n";
            } catch (\Exception $e) {}
            
            try {
                // 2. User Details
                DB::table('user_details')->where('user_id', $userId)->delete();
                echo "🗑️  Deleted user details\n";
            } catch (\Exception $e) {}
            
            try {
                // 3. Password Reset Tokens
                DB::table('password_reset_tokens')->where('email', $email)->delete();
                echo "🗑️  Deleted password reset tokens\n";
            } catch (\Exception $e) {}
            
            try {
                // 4. Personal Access Tokens
                DB::table('personal_access_tokens')
                    ->where('tokenable_type', 'App\\Models\\User')
                    ->where('tokenable_id', $userId)
                    ->delete();
                echo "🗑️  Deleted personal access tokens\n";
            } catch (\Exception $e) {}
            
            // FINAL: Force delete the user (including soft deleted)
            $user->forceDelete();
            echo "✅ User {$email} has been permanently deleted!\n";
            
            // Summary
            echo "\n📊 DELETION SUMMARY:\n";
            echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            foreach ($deletedCount as $table => $count) {
                if ($count > 0) {
                    echo "  - {$table}: {$count} records\n";
                }
            }
            echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            echo "✅ All data for {$email} has been permanently deleted!\n\n";
            
            DB::commit();
            
            Log::info("Provider user {$email} (ID: {$userId}) and all related data permanently deleted.", $deletedCount);
            
        } catch (\Exception $e) {
            DB::rollBack();
            echo "❌ ERROR: " . $e->getMessage() . "\n";
            echo "Stack trace: " . $e->getTraceAsString() . "\n";
            Log::error("Failed to delete provider user {$email}: " . $e->getMessage());
            throw $e;
        }
    }
}

