<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class InvoiceTemplateSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Wallet Top-up Invoice Template
        $walletTemplate = $this->getWalletTopupTemplate();
        
        // Paradigm Purchase Invoice Template
        $paradigmTemplate = $this->getParadigmPurchaseTemplate();
        
        // Insert templates if they don't exist
        $this->insertTemplate('Wallet Top-up Invoice', 'Wallet Top-up', $walletTemplate);
        $this->insertTemplate('Paradigm Assessment Invoice', 'Paradigm Purchase', $paradigmTemplate);
    }
    
    private function insertTemplate($title, $type, $content)
    {
        $exists = DB::table('invoice_templates')
            ->where('invoice_title', $title)
            ->where('invoice_type', $type)
            ->exists();
        
        if (!$exists) {
            DB::table('invoice_templates')->insert([
                'invoice_title' => $title,
                'invoice_type' => $type,
                'template_content' => $content,
                'is_default' => 0,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }
    
    private function getWalletTopupTemplate(): string
    {
        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Wallet Top-up Invoice</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background: #f5f5f5;
            padding: 40px 20px;
        }
        
        .invoice-container {
            max-width: 850px;
            margin: 0 auto;
            background: #ffffff;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            border-radius: 12px;
            overflow: hidden;
        }
        
        .invoice-header {
            background: linear-gradient(135deg, #044f54 0%, #066a70 100%);
            padding: 40px 50px;
            color: white;
            position: relative;
            overflow: hidden;
        }
        
        .invoice-header::before {
            content: "";
            position: absolute;
            top: -50%;
            right: -10%;
            width: 400px;
            height: 400px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 50%;
        }
        
        .invoice-header-content {
            position: relative;
            z-index: 1;
            text-align: center;
        }
        
        .company-name {
            font-size: 28px;
            font-weight: 700;
            margin-bottom: 12px;
            letter-spacing: 1px;
            text-transform: uppercase;
        }
        
        .invoice-title-section {
            margin-top: 15px;
        }
        
        .invoice-title {
            font-size: 32px;
            font-weight: 700;
            margin-bottom: 8px;
            letter-spacing: -0.5px;
        }
        
        .invoice-subtitle {
            font-size: 14px;
            opacity: 0.9;
            font-weight: 300;
        }
        
        .invoice-body {
            padding: 50px;
        }
        
        .info-section {
            display: flex;
            justify-content: space-between;
            margin-bottom: 40px;
            gap: 30px;
            flex-wrap: wrap;
        }
        
        .info-block {
            flex: 1;
            min-width: 200px;
        }
        
        .info-block-title {
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #044f54;
            font-weight: 700;
            margin-bottom: 15px;
            padding-bottom: 8px;
            border-bottom: 2px solid #044f54;
        }
        
        .info-block-content {
            font-size: 14px;
            line-height: 1.8;
        }
        
        .info-block-content strong {
            display: block;
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 5px;
            color: #1a1a1a;
        }
        
        .info-block-content p {
            margin: 3px 0;
            color: #666;
        }
        
        .invoice-meta {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 40px;
        }
        
        .meta-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
        }
        
        .meta-item {
            display: flex;
            flex-direction: column;
        }
        
        .meta-label {
            font-size: 11px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: #888;
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .meta-value {
            font-size: 15px;
            color: #1a1a1a;
            font-weight: 500;
        }
        
        .transaction-badge {
            display: inline-block;
            background: #044f54;
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 13px;
            font-weight: 500;
        }
        
        .items-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
            margin-bottom: 30px;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.06);
        }
        
        .items-table thead {
            background: #044f54;
            color: white;
        }
        
        .items-table thead th {
            padding: 18px 20px;
            text-align: left;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .items-table tbody tr {
            background: white;
            border-bottom: 1px solid #e9ecef;
            transition: background 0.2s ease;
        }
        
        .items-table tbody tr:last-child {
            border-bottom: none;
        }
        
        .items-table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .items-table tbody td {
            padding: 20px;
            font-size: 14px;
            color: #495057;
        }
        
        .items-table tbody td:first-child {
            font-weight: 500;
            color: #1a1a1a;
        }
        
        .items-table tbody td:last-child {
            text-align: right;
            font-weight: 600;
            font-size: 16px;
            color: #044f54;
        }
        
        .payment-method-badge {
            display: inline-block;
            background: #e8f5f6;
            color: #044f54;
            padding: 6px 14px;
            border-radius: 6px;
            font-size: 13px;
            font-weight: 500;
        }
        
        .total-section {
            background: linear-gradient(135deg, #044f54 0%, #066a70 100%);
            border-radius: 8px;
            padding: 25px 30px;
            margin-top: 30px;
            color: white;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .total-label {
            font-size: 18px;
            font-weight: 300;
            letter-spacing: 0.5px;
        }
        
        .total-amount {
            font-size: 36px;
            font-weight: 700;
            letter-spacing: -1px;
        }
        
        .footer {
            background: #f8f9fa;
            padding: 30px 50px;
            text-align: center;
            border-top: 1px solid #e9ecef;
        }
        
        .footer-content {
            max-width: 600px;
            margin: 0 auto;
        }
        
        .thank-you-message {
            font-size: 16px;
            font-weight: 500;
            color: #044f54;
            margin-bottom: 15px;
        }
        
        .footer-text {
            font-size: 13px;
            color: #6c757d;
            line-height: 1.6;
            margin: 8px 0;
        }
        
        .footer-divider {
            width: 60px;
            height: 3px;
            background: #044f54;
            margin: 20px auto;
            border-radius: 2px;
        }
        
        .contact-info {
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid #dee2e6;
        }
        
        .contact-info p {
            font-size: 12px;
            color: #868e96;
            margin: 5px 0;
        }
        
        @media print {
            body {
                background: white;
                padding: 0;
            }
            .invoice-container {
                box-shadow: none;
                border-radius: 0;
            }
        }
        
        @media (max-width: 768px) {
            .invoice-body {
                padding: 30px 20px;
            }
            .invoice-header {
                padding: 30px 20px;
            }
            .company-name {
                font-size: 22px;
            }
            .invoice-title {
                font-size: 24px;
            }
            .total-amount {
                font-size: 28px;
            }
            .items-table thead th,
            .items-table tbody td {
                padding: 12px 10px;
                font-size: 12px;
            }
        }
    </style>
</head>
<body>
    <div class="invoice-container">
        <!-- Header -->
        <div class="invoice-header">
            <div class="invoice-header-content">
                <div class="company-name">{{company_name}}</div>
                <div class="invoice-title-section">
                    <div class="invoice-title">INVOICE</div>
                    <div class="invoice-subtitle">Wallet Top-up Receipt</div>
                </div>
            </div>
        </div>
        
        <!-- Body -->
        <div class="invoice-body">
            <!-- Invoice Meta Information -->
            <div class="invoice-meta">
                <div class="meta-grid">
                    <div class="meta-item">
                        <span class="meta-label">Invoice Number</span>
                        <span class="meta-value">#{{id}}</span>
                    </div>
                    <div class="meta-item">
                        <span class="meta-label">Invoice Date</span>
                        <span class="meta-value">{{date}}</span>
                    </div>
                    <div class="meta-item">
                        <span class="meta-label">Transaction ID</span>
                        <span class="meta-value transaction-badge">{{transaction_id}}</span>
                    </div>
                </div>
            </div>
            
            <!-- Company and Client Information -->
            <div class="info-section">
                <div class="info-block">
                    <div class="info-block-title">From</div>
                    <div class="info-block-content">
                        <strong>{{company_name}}</strong>
                        <p>{{company_address}}</p>
                        <p>Email: {{company_email}}</p>
                        <p>Phone: {{company_phone}}</p>
                        <p>Web: {{website_link}}</p>
                    </div>
                </div>
                
                <div class="info-block">
                    <div class="info-block-title">Bill To</div>
                    <div class="info-block-content">
                        <strong>{{user_name}}</strong>
                        <p>{{user_address}}</p>
                        <p>{{user_city}}, {{user_state}} {{user_postal}}</p>
                        <p>Email: {{user_email}}</p>
                        <p>Phone: {{user_phone}}</p>
                    </div>
                </div>
            </div>
            
            <!-- Items Table -->
            <table class="items-table">
                <thead>
                    <tr>
                        <th>Description</th>
                        <th>Payment Method</th>
                        <th style="text-align: right;">Amount</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td>Wallet Credit Top-up</td>
                        <td><span class="payment-method-badge">{{payment_method}}</span></td>
                        <td style="text-align: right;">${{amount}}</td>
                    </tr>
                </tbody>
            </table>
            
            <!-- Total Section -->
            <div class="total-section">
                <div class="total-row">
                    <span class="total-label">Total Amount</span>
                    <span class="total-amount">${{amount}}</span>
                </div>
            </div>
        </div>
        
        <!-- Footer -->
        <div class="footer">
            <div class="footer-content">
                <div class="thank-you-message">Thank You for Your Payment!</div>
                <div class="footer-divider"></div>
                <p class="footer-text">This is an automatically generated invoice for your wallet top-up transaction.</p>
                <p class="footer-text">Your payment has been processed successfully and the amount has been credited to your wallet.</p>
                
                <div class="contact-info">
                    <p>Questions? Contact us at {{contact}}</p>
                    <p>© {{company_name}} - All rights reserved</p>
                </div>
            </div>
        </div>
    </div>
</body>
</html>';
    }
    
    private function getParadigmPurchaseTemplate(): string
    {
        return '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Paradigm Assessment Invoice</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        .header { text-align: center; margin-bottom: 30px; }
        .logo { max-width: 200px; }
        .company-info { margin: 20px 0; }
        .invoice-details { margin: 30px 0; }
        .table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        .table th, .table td { border: 1px solid #ddd; padding: 12px; text-align: left; }
        .table th { background-color: #044f54; color: white; }
        .total { font-size: 18px; font-weight: bold; text-align: right; margin-top: 20px; }
        .footer { margin-top: 40px; text-align: center; font-size: 12px; color: #044f54; }
        .assessment-info { background-color: #f9f9f9; padding: 15px; margin: 20px 0; border-left: 4px solid #044f54; }
    </style>
</head>
<body>
    <div class="header">
        <img src="{{logo}}" alt="Company Logo" class="logo">
        <h1>Paradigm Assessment Invoice</h1>
    </div>
    
    <div class="company-info">
        <strong>{{company_name}}</strong><br>
        {{company_address}}<br>
        Email: {{company_email}}<br>
        Phone: {{company_phone}}<br>
        Website: {{website_link}}
    </div>
    
    <div class="invoice-details">
        <table style="width: 50%;">
            <tr>
                <td><strong>Invoice #:</strong></td>
                <td>{{id}}</td>
            </tr>
            <tr>
                <td><strong>Date:</strong></td>
                <td>{{date}}</td>
            </tr>
            <tr>
                <td><strong>Payment ID:</strong></td>
                <td>{{payment_id}}</td>
            </tr>
        </table>
    </div>
    
    <div style="margin: 30px 0;">
        <h3>Bill To:</h3>
        <strong>{{user_name}}</strong><br>
        {{user_address}}<br>
        {{user_city}}, {{user_state}} {{user_postal}}<br>
        Email: {{user_email}}<br>
        Phone: {{user_phone}}
    </div>
    
    <div class="assessment-info">
        <h3>Assessment Details</h3>
        <p><strong>Template:</strong> {{template_name}}</p>
        <p><strong>Client:</strong> {{client_name}}</p>
        <p><strong>Access Token:</strong> {{access_token}}</p>
    </div>
    
    <table class="table">
        <thead>
            <tr>
                <th>Description</th>
                <th>Quantity</th>
                <th>Unit Price</th>
                <th>Amount</th>
            </tr>
        </thead>
        <tbody>
            <tr>
                <td>Paradigm Profile Assessment - {{template_name}}</td>
                <td>1</td>
                <td>${{amount}}</td>
                <td>${{amount}}</td>
            </tr>
        </tbody>
    </table>
    
    <div class="total">
        <p>Total Amount: ${{amount}}</p>
        <p style="font-size: 14px; color: #666;">Paid via Wallet</p>
    </div>
    
    <div class="footer">
        <p>Thank you for your purchase!</p>
        <p>Your assessment link has been sent to the client.</p>
        <p>For support, contact: {{contact}}</p>
    </div>
</body>
</html>';
    }
}

