<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Repositories\Eloquent\WalletRepository;

class ParadigmRefundNatalieSeeder extends Seeder
{
	/**
	 * Seeder: Remove "Client Pays" Paradigm assignments for two clients
	 * under provider natalie050267@gmail.com and refund 30 credits to
	 * the provider's wallet.
	 */
	public function run(): void
	{
		$providerEmail = 'natalie050267@gmail.com';
		$clientEmails = [
			'esti.brummer@gmail.com',
			'qmotale@gmail.com',
		];
		$refundAmount = 30.0;

		$provider = DB::table('users')->where('email', $providerEmail)->first();
		if (!$provider) {
			Log::warning('ParadigmRefundNatalieSeeder: Provider not found', ['email' => $providerEmail]);
			if (isset($this->command)) {
				$this->command->warn("Provider {$providerEmail} not found. Skipping.");
			}
			return;
		}

		$payments = DB::table('paradigm_payments')
			->where('provider_id', $provider->id)
			->whereIn('client_email', $clientEmails)
			->where('paid_by', 'client')
			->get();

		if ($payments->isNotEmpty()) {
			DB::transaction(function () use ($payments) {
				$ids = $payments->pluck('id')->all();

				DB::table('paradigm_access_tokens')->whereIn('payment_id', $ids)->delete();
				DB::table('paradigm_payments')->whereIn('id', $ids)->delete();
			});

			Log::info('ParadigmRefundNatalieSeeder: Removed client-pay assignments', [
				'count' => $payments->count(),
				'ids' => $payments->pluck('id')->all(),
			]);

			if (isset($this->command)) {
				$this->command->info('Removed ' . $payments->count() . ' client-pay assignment(s).');
			}
		} else {
			Log::info('ParadigmRefundNatalieSeeder: No matching client-pay assignments found', [
				'provider_id' => $provider->id,
				'clients' => $clientEmails,
			]);
			if (isset($this->command)) {
				$this->command->info('No matching client-pay assignments found.');
			}
		}

		// Credit provider wallet
		$walletRepo = app(WalletRepository::class);
		$walletRepo->addWalletAmount([
			'user_id' => $provider->id,
			'amount' => $refundAmount,
			'credits' => $refundAmount,
			'payment_method' => 'Others',
			'status' => 'Completed',
			'transaction_id' => 'PARADIGM-REFUND-' . now()->timestamp,
			'type' => 1,
		]);

		if (isset($this->command)) {
			$this->command->info("Credited {$refundAmount} credits to {$providerEmail}.");
		}
	}
}

