<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

/**
 * SAFE UPDATE SEEDER - Updates specific Paradigm question text
 * 
 * This seeder updates ONLY the specific question options requested.
 * It VERIFIES before updating and shows what changed.
 */
class UpdateParadigmQuestionTextSeeder extends Seeder
{
    /**
     * Mapping of Parts to assessment_module
     * Part 2 = paradigm_main (brain profile 4-frame)
     * Part 5 = connection_styles
     */
    private $updates = [
        // PART 2 - Section 1 - Question 1 - Option B
        [
            'part' => 'Part 2',
            'section' => 'Section 1',
            'question_num' => 1,
            'option' => 'B',
            'assessment_module' => 'paradigm_main',
            'search_text' => 'Section 1 - Question 1', // Question text to find
            'old_option_b' => 'I like evaluating my own work and others.', // Current text
            'new_option_b' => 'I like to evaluate my own work and that of others.', // New text
        ],
        
        // PART 5 - Question 2 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 2,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 2,
            'old_option_b' => null, // Will find automatically
            'new_option_b' => 'Someone helps me with something I need to do.',
        ],
        
        // PART 5 - Question 5 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 5,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 5,
            'old_option_b' => null,
            'new_option_b' => 'Someone gives me a surprise gift.',
        ],
        
        // PART 5 - Question 7 - Option A
        [
            'part' => 'Part 5',
            'question_num' => 7,
            'option' => 'A',
            'assessment_module' => 'connection_styles',
            'display_order' => 7,
            'old_option_a' => null,
            'new_option_a' => 'I get a gift from someone.',
        ],
        
        // PART 5 - Question 8 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 8,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 8,
            'old_option_b' => null,
            'new_option_b' => 'Someone gives me a nice compliment.',
        ],
        
        // PART 5 - Question 9 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 9,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 9,
            'old_option_b' => null,
            'new_option_b' => 'I get a surprise gift from someone.',
        ],
        
        // PART 5 - Question 10 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 10,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 10,
            'old_option_b' => null,
            'new_option_b' => 'Someone helps me do something.',
        ],
        
        // PART 5 - Question 13 - Option A
        [
            'part' => 'Part 5',
            'question_num' => 13,
            'option' => 'A',
            'assessment_module' => 'connection_styles',
            'display_order' => 13,
            'old_option_a' => null,
            'new_option_a' => 'Someone tells me something good about myself.',
        ],
        
        // PART 5 - Question 17 - Option A
        [
            'part' => 'Part 5',
            'question_num' => 17,
            'option' => 'A',
            'assessment_module' => 'connection_styles',
            'display_order' => 17,
            'old_option_a' => null,
            'new_option_a' => 'Someone helps me finish a special project.',
        ],
        
        // PART 5 - Question 21 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 21,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 21,
            'old_option_b' => null,
            'new_option_b' => 'Someone helps me so I don\'t feel too stressed.',
        ],
        
        // PART 5 - Question 24 - Option B
        [
            'part' => 'Part 5',
            'question_num' => 24,
            'option' => 'B',
            'assessment_module' => 'connection_styles',
            'display_order' => 24,
            'old_option_b' => null,
            'new_option_b' => 'I give someone a special gift',
        ],
    ];

    public function run(): void
    {
        echo "\n" . str_repeat("=", 100) . "\n";
        echo "PARADIGM ASSESSMENT QUESTION UPDATE SEEDER\n";
        echo "VERIFICATION MODE - SHOWING WHAT WILL CHANGE\n";
        echo str_repeat("=", 100) . "\n\n";
        
        $verifiedUpdates = [];
        $errors = [];
        
        // Step 1: VERIFY all questions exist and show what will change
        foreach ($this->updates as $index => $update) {
            $num = $index + 1;
            echo "[$num/" . count($this->updates) . "] Verifying: {$update['part']}";
            
            if (isset($update['section'])) {
                echo " - {$update['section']}";
            }
            
            echo " - Question {$update['question_num']} - Option {$update['option']}\n";
            echo str_repeat("-", 100) . "\n";
            
            // Find the question
            $query = DB::table('paradigm_questions')
                ->where('assessment_module', $update['assessment_module']);
            
            if (isset($update['search_text'])) {
                $query->where('question_text', 'LIKE', '%' . $update['search_text'] . '%');
            }
            
            if (isset($update['display_order'])) {
                $query->where('display_order', $update['display_order']);
            }
            
            $question = $query->first();
            
            if (!$question) {
                echo "❌ ERROR: Question NOT FOUND!\n";
                echo "   Module: {$update['assessment_module']}\n";
                if (isset($update['search_text'])) {
                    echo "   Search: {$update['search_text']}\n";
                }
                if (isset($update['display_order'])) {
                    echo "   Display Order: {$update['display_order']}\n";
                }
                echo "\n";
                $errors[] = $update;
                continue;
            }
            
            // Show current question details
            echo "✅ Question Found:\n";
            echo "   ID: {$question->id}\n";
            echo "   Question: " . substr($question->question_text, 0, 80) . "...\n";
            echo "   Current Option A: {$question->option_a_label}\n";
            echo "   Current Option B: {$question->option_b_label}\n";
            
            // Determine what will change
            $optionColumn = 'option_' . strtolower($update['option']) . '_label';
            $oldKey = 'old_option_' . strtolower($update['option']);
            $newKey = 'new_option_' . strtolower($update['option']);
            
            $currentValue = $question->$optionColumn;
            $newValue = $update[$newKey];
            
            echo "\n";
            echo "   CHANGE TO OPTION {$update['option']}:\n";
            echo "   OLD: \"{$currentValue}\"\n";
            echo "   NEW: \"{$newValue}\"\n";
            
            // Verify old text matches if provided
            if (!empty($update[$oldKey]) && $update[$oldKey] !== $currentValue) {
                echo "\n";
                echo "   ⚠️  WARNING: Old text doesn't match!\n";
                echo "   Expected: \"{$update[$oldKey]}\"\n";
                echo "   Found:    \"{$currentValue}\"\n";
            }
            
            if ($currentValue === $newValue) {
                echo "\n   ℹ️  NOTE: Already updated (old and new are the same)\n";
            }
            
            echo "\n";
            
            // Store for actual update
            $verifiedUpdates[] = [
                'id' => $question->id,
                'part' => $update['part'],
                'question_num' => $update['question_num'],
                'option' => $update['option'],
                'column' => $optionColumn,
                'old_value' => $currentValue,
                'new_value' => $newValue,
                'already_updated' => ($currentValue === $newValue)
            ];
        }
        
        // Summary
        echo str_repeat("=", 100) . "\n";
        echo "VERIFICATION SUMMARY\n";
        echo str_repeat("=", 100) . "\n";
        echo "Total Updates Requested: " . count($this->updates) . "\n";
        echo "Successfully Verified: " . count($verifiedUpdates) . "\n";
        echo "Errors Found: " . count($errors) . "\n";
        echo "Already Updated: " . count(array_filter($verifiedUpdates, fn($u) => $u['already_updated'])) . "\n";
        echo "Need Updating: " . count(array_filter($verifiedUpdates, fn($u) => !$u['already_updated'])) . "\n\n";
        
        if (count($errors) > 0) {
            echo "❌ ERRORS PREVENT UPDATE - FIX THESE FIRST:\n";
            foreach ($errors as $error) {
                echo "   - {$error['part']} - Question {$error['question_num']}\n";
            }
            echo "\n";
            return;
        }
        
        // Ask for confirmation (in actual run, this would be automatic)
        echo "Ready to apply updates...\n\n";
        
        // APPLY UPDATES NOW
        echo str_repeat("=", 100) . "\n";
        echo "APPLYING UPDATES...\n";
        echo str_repeat("=", 100) . "\n\n";
        
        $updated = 0;
        foreach ($verifiedUpdates as $update) {
            if ($update['already_updated']) {
                echo "⏭️  Skipping {$update['part']} Q{$update['question_num']} Option {$update['option']} (already updated)\n";
                continue;
            }
            
            DB::table('paradigm_questions')
                ->where('id', $update['id'])
                ->update([
                    $update['column'] => $update['new_value'],
                    'updated_at' => now()
                ]);
            
            echo "✅ Updated {$update['part']} Q{$update['question_num']} Option {$update['option']}\n";
            $updated++;
        }
        
        echo "\n" . str_repeat("=", 100) . "\n";
        echo "UPDATE COMPLETE!\n";
        echo str_repeat("=", 100) . "\n";
        echo "Total questions updated: {$updated}\n";
        echo "Total questions skipped: " . (count($verifiedUpdates) - $updated) . "\n\n";
    }
}

