/**
 * Admin Lead Pricing Management
 * Handles CRUD operations for pricing rules, templates, and testing
 */

(function() {
    'use strict';

    let pricingTable;
    let isEditMode = false;
    let currentPricingId = null;

    // Initialize when document is ready
    $(document).ready(function() {
        console.log('✅ Admin Lead Pricing initialized');
        initPricingTable();
        setupEventListeners();
        loadAvailableCategories();
    });

    /**
     * Initialize DataTable
     */
    function initPricingTable() {
        pricingTable = $('#pricingTable').DataTable({
            ajax: {
                url: '/admin/leads/pricing/list',
                type: 'POST',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                data: function(d) {
                    d.search = $('#searchFilter').val();
                    d.status = $('#statusFilter').val();
                },
                dataSrc: function(json) {
                    if (json.success) {
                        return json.data;
                    } else {
                        toastr.error('Failed to load pricing rules');
                        return [];
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error loading pricing rules:', error);
                    toastr.error('Error loading pricing rules');
                }
            },
            columns: [
                { data: 'category_name' },
                { 
                    data: 'base_price',
                    render: function(data) {
                        return data + ' Credits';
                    }
                },
                { 
                    data: 'price_per_field',
                    render: function(data) {
                        return data + ' Credits';
                    }
                },
                { 
                    data: 'price_per_100_chars',
                    render: function(data) {
                        return data + ' Credits';
                    }
                },
                { 
                    data: 'min_price',
                    render: function(data) {
                        return data + ' Credits';
                    }
                },
                { 
                    data: 'max_price',
                    render: function(data) {
                        return data + ' Credits';
                    }
                },
                { 
                    data: 'active',
                    render: function(data, type, row) {
                        const checked = data ? 'checked' : '';
                        const statusClass = data ? 'success' : 'secondary';
                        return `
                            <div class="form-check form-switch">
                                <input class="form-check-input toggle-status" 
                                       type="checkbox" ${checked} 
                                       data-id="${row.id}">
                            </div>
                        `;
                    }
                },
                { data: 'created_by' },
                { 
                    data: null,
                    orderable: false,
                    render: function(data, type, row) {
                        return `
                            <div class="table-actions d-flex">
                                <button class="btn btn-sm btn-info me-2 btn-edit" data-id="${row.id}" title="Edit">
                                    <i class="fa fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-danger btn-delete" data-id="${row.id}" title="Delete">
                                    <i class="fa fa-trash"></i>
                                </button>
                            </div>
                        `;
                    }
                }
            ],
            order: [[0, 'asc']],
            pageLength: 25,
            responsive: true,
            language: {
                emptyTable: "No pricing rules found. Click 'Add Pricing Rule' to create one.",
                zeroRecords: "No matching pricing rules found"
            }
        });
    }

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Apply filter
        $('#btnApplyFilter').on('click', function() {
            pricingTable.ajax.reload();
        });

        // Reset filter
        $('#btnResetFilter').on('click', function() {
            $('#searchFilter').val('');
            $('#statusFilter').val('');
            pricingTable.ajax.reload();
        });

        // Edit button (delegated)
        $('#pricingTable').on('click', '.btn-edit', function() {
            const id = $(this).data('id');
            editPricingRule(id);
        });

        // Delete button (delegated)
        $('#pricingTable').on('click', '.btn-delete', function() {
            const id = $(this).data('id');
            deletePricingRule(id);
        });

        // Toggle status (delegated)
        $('#pricingTable').on('change', '.toggle-status', function() {
            const id = $(this).data('id');
            const isActive = $(this).is(':checked');
            toggleStatus(id, isActive);
        });

        // Add pricing modal - reset form when opened
        $('#addPricingModal').on('show.bs.modal', function() {
            if (!isEditMode) {
                resetPricingForm();
                $('#modalTitle').text('Add Pricing Rule');
                loadAvailableCategories();
            }
        });

        // Pricing form submission
        $('#pricingForm').on('submit', function(e) {
            e.preventDefault();
            savePricingRule();
        });

        // Template modal - load categories when opened
        $('#templateModal').on('show.bs.modal', function() {
            loadCategoriesForTemplate();
        });

        // Template type change - show details
        $('#template_type').on('change', function() {
            showTemplateDetails($(this).val());
        });

        // Template form submission
        $('#templateForm').on('submit', function(e) {
            e.preventDefault();
            applyTemplate();
        });

        // Test calculator modal - load categories when opened
        $('#testCalculatorModal').on('show.bs.modal', function() {
            loadCategoriesWithRules();
        });

        // Test calculator form submission
        $('#testCalculatorForm').on('submit', function(e) {
            e.preventDefault();
            calculateTestPrice();
        });

        // Price preview on input change
        $('#base_price, #price_per_field, #price_per_100_chars, #min_price, #max_price').on('input', function() {
            showPricePreview();
        });
    }

    /**
     * Load available categories (without pricing rules)
     */
    function loadAvailableCategories() {
        $.ajax({
            url: '/admin/leads/pricing/categories/available',
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    const $select = $('#category_id');
                    $select.find('option:not(:first)').remove();
                    
                    if (response.categories.length === 0) {
                        $select.append('<option value="" disabled>All categories have pricing rules</option>');
                    } else {
                        response.categories.forEach(function(cat) {
                            $select.append(`<option value="${cat.id}">${cat.name}</option>`);
                        });
                    }
                }
            },
            error: function() {
                toastr.error('Error loading categories');
            }
        });
    }

    /**
     * Load categories for template application
     */
    function loadCategoriesForTemplate() {
        $.ajax({
            url: '/admin/leads/pricing/categories/available',
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    const $select = $('#template_categories');
                    $select.empty();
                    
                    if (response.categories.length === 0) {
                        $select.append('<option value="" disabled>No categories available</option>');
                    } else {
                        response.categories.forEach(function(cat) {
                            $select.append(`<option value="${cat.id}">${cat.name}</option>`);
                        });
                    }
                }
            }
        });
    }

    /**
     * Load categories that have pricing rules (for test calculator)
     */
    function loadCategoriesWithRules() {
        $.ajax({
            url: '/admin/leads/pricing/list',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: { status: '1' }, // Only active rules
            success: function(response) {
                if (response.success) {
                    const $select = $('#test_category_id');
                    $select.find('option:not(:first)').remove();
                    
                    response.data.forEach(function(rule) {
                        $select.append(`<option value="${rule.category_id}">${rule.category_name}</option>`);
                    });
                }
            }
        });
    }

    /**
     * Edit pricing rule
     */
    function editPricingRule(id) {
        isEditMode = true;
        currentPricingId = id;

        $.ajax({
            url: `/admin/leads/pricing/${id}`,
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    const rule = response.data;
                    
                    // Populate form
                    $('#pricing_id').val(rule.id);
                    $('#base_price').val(rule.base_price);
                    $('#price_per_field').val(rule.price_per_field);
                    $('#price_per_100_chars').val(rule.price_per_100_chars);
                    $('#min_price').val(rule.min_price);
                    $('#max_price').val(rule.max_price);
                    $('#active').prop('checked', rule.active);

                    // Load categories and select current one
                    const $categorySelect = $('#category_id');
                    $categorySelect.empty();
                    $categorySelect.append(`<option value="${rule.category_id}" selected>${rule.category.name}</option>`);
                    
                    // Disable category selection when editing
                    $categorySelect.prop('disabled', true);

                    $('#modalTitle').text('Edit Pricing Rule');
                    $('#addPricingModal').modal('show');

                    showPricePreview();
                }
            },
            error: function() {
                toastr.error('Error loading pricing rule');
            }
        });
    }

    /**
     * Save pricing rule (create or update)
     */
    function savePricingRule() {
        const $btn = $('#btnSavePricing');
        $btn.prop('disabled', true);
        $btn.find('.spinner-border').removeClass('d-none');

        const formData = {
            category_id: $('#category_id').val(),
            base_price: $('#base_price').val(),
            price_per_field: $('#price_per_field').val(),
            price_per_100_chars: $('#price_per_100_chars').val(),
            min_price: $('#min_price').val(),
            max_price: $('#max_price').val(),
            active: $('#active').is(':checked') ? 1 : 0,
        };

        const url = isEditMode ? `/admin/leads/pricing/${currentPricingId}` : '/admin/leads/pricing/store';
        const method = isEditMode ? 'PUT' : 'POST';

        $.ajax({
            url: url,
            type: method,
            data: formData,
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    $('#addPricingModal').modal('hide');
                    pricingTable.ajax.reload();
                } else {
                    toastr.error(response.message || 'Error saving pricing rule');
                }
            },
            error: function(xhr) {
                if (xhr.status === 422 && xhr.responseJSON.errors) {
                    displayValidationErrors(xhr.responseJSON.errors);
                } else {
                    toastr.error('Error saving pricing rule');
                }
            },
            complete: function() {
                $btn.prop('disabled', false);
                $btn.find('.spinner-border').addClass('d-none');
            }
        });
    }

    /**
     * Delete pricing rule
     */
    function deletePricingRule(id) {
        if (!confirm('Are you sure you want to delete this pricing rule?')) {
            return;
        }

        $.ajax({
            url: `/admin/leads/pricing/${id}`,
            type: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    pricingTable.ajax.reload();
                } else {
                    toastr.error(response.message || 'Error deleting pricing rule');
                }
            },
            error: function() {
                toastr.error('Error deleting pricing rule');
            }
        });
    }

    /**
     * Toggle pricing rule status
     */
    function toggleStatus(id, isActive) {
        $.ajax({
            url: `/admin/leads/pricing/${id}/toggle`,
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success('Status updated successfully');
                } else {
                    toastr.error('Error updating status');
                    // Revert checkbox
                    pricingTable.ajax.reload();
                }
            },
            error: function() {
                toastr.error('Error updating status');
                pricingTable.ajax.reload();
            }
        });
    }

    /**
     * Show template details
     */
    function showTemplateDetails(template) {
        const templates = {
            'low': {
                name: 'Low Value',
                base: '$5.00',
                field: '$0.50',
                char: '$0.10',
                min: '$5.00',
                max: '$25.00',
                description: 'For low-cost services like cleaning, delivery, tutoring'
            },
            'medium': {
                name: 'Medium Value',
                base: '$15.00',
                field: '$1.00',
                char: '$0.20',
                min: '$10.00',
                max: '$50.00',
                description: 'For mid-range services like plumbing, electrical, painting'
            },
            'high': {
                name: 'High Value',
                base: '$30.00',
                field: '$2.00',
                char: '$0.50',
                min: '$20.00',
                max: '$100.00',
                description: 'For high-value services like construction, legal, consulting'
            }
        };

        if (template && templates[template]) {
            const t = templates[template];
            $('#templateDetails').html(`
                <p class="mb-2"><strong>${t.name} Template</strong></p>
                <ul class="mb-0">
                    <li>Base Price: ${t.base} Credits</li>
                    <li>Price Per Field: ${t.field} Credits</li>
                    <li>Price Per 100 Chars: ${t.char} Credits</li>
                    <li>Min Price: ${t.min} Credits</li>
                    <li>Max Price: ${t.max} Credits</li>
                </ul>
                <p class="mt-2 mb-0 small text-muted">${t.description}</p>
            `);
        } else {
            $('#templateDetails').html('<p class="mb-0 text-muted">Select a template to see details</p>');
        }
    }

    /**
     * Apply template to selected categories
     */
    function applyTemplate() {
        const $btn = $('#btnApplyTemplate');
        $btn.prop('disabled', true);
        $btn.find('.spinner-border').removeClass('d-none');

        const formData = {
            template: $('#template_type').val(),
            category_ids: $('#template_categories').val()
        };

        if (!formData.category_ids || formData.category_ids.length === 0) {
            toastr.error('Please select at least one category');
            $btn.prop('disabled', false);
            $btn.find('.spinner-border').addClass('d-none');
            return;
        }

        $.ajax({
            url: '/admin/leads/pricing/template/apply',
            type: 'POST',
            data: formData,
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    $('#templateModal').modal('hide');
                    pricingTable.ajax.reload();
                } else {
                    toastr.error(response.message || 'Error applying template');
                }
            },
            error: function() {
                toastr.error('Error applying template');
            },
            complete: function() {
                $btn.prop('disabled', false);
                $btn.find('.spinner-border').addClass('d-none');
            }
        });
    }

    /**
     * Calculate test price
     */
    function calculateTestPrice() {
        const $btn = $('#btnCalculate');
        $btn.prop('disabled', true);
        $btn.find('.spinner-border').removeClass('d-none');

        const formData = {
            category_id: $('#test_category_id').val(),
            fields_filled: $('#test_fields_filled').val(),
            message_chars: $('#test_message_chars').val()
        };

        $.ajax({
            url: '/admin/leads/pricing/test/calculate',
            type: 'POST',
            data: formData,
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    const calc = response.calculation;
                    const cappedBadge = calc.capped ? '<span class="badge bg-warning">Capped</span>' : '';
                    
                    $('#calculationDetails').html(`
                        <ul class="mb-0">
                            <li>Base Price: ${calc.base_price} Credits</li>
                            <li>Field Price (${formData.fields_filled} fields × price): ${calc.field_price} Credits</li>
                            <li>Character Price (${Math.ceil(formData.message_chars / 100)} × 100 chars × price): ${calc.char_price} Credits</li>
                            <li><strong>Subtotal: ${calc.subtotal} Credits</strong></li>
                            <li>Min/Max Range: ${calc.min_price} - ${calc.max_price} Credits</li>
                            <li class="mt-2"><strong class="text-success">Final Price: ${calc.final_price} Credits</strong> ${cappedBadge}</li>
                        </ul>
                    `);
                    $('#calculationResult').show();
                } else {
                    toastr.error(response.message || 'Error calculating price');
                }
            },
            error: function() {
                toastr.error('Error calculating price');
            },
            complete: function() {
                $btn.prop('disabled', false);
                $btn.find('.spinner-border').addClass('d-none');
            }
        });
    }

    /**
     * Show price preview in pricing form
     */
    function showPricePreview() {
        const base = parseFloat($('#base_price').val()) || 0;
        const perField = parseFloat($('#price_per_field').val()) || 0;
        const perChar = parseFloat($('#price_per_100_chars').val()) || 0;
        const min = parseFloat($('#min_price').val()) || 0;
        const max = parseFloat($('#max_price').val()) || 0;

        if (base > 0 || perField > 0 || perChar > 0) {
            // Example calculation with 8 fields and 300 characters
            const exampleFields = 8;
            const exampleChars = 300;
            
            const fieldPrice = exampleFields * perField;
            const charPrice = Math.ceil(exampleChars / 100) * perChar;
            const subtotal = base + fieldPrice + charPrice;
            const finalPrice = Math.max(min, Math.min(subtotal, max));

            $('#previewContent').html(`
                <p class="mb-2 small">Example: 8 fields filled, 300 character message</p>
                <ul class="mb-0">
                    <li>Base: $${base.toFixed(2)}</li>
                    <li>Fields: $${fieldPrice.toFixed(2)} (8 × $${perField.toFixed(2)})</li>
                    <li>Chars: $${charPrice.toFixed(2)} (3 × $${perChar.toFixed(2)})</li>
                    <li><strong>Final: $${finalPrice.toFixed(2)}</strong></li>
                </ul>
            `);
            $('#pricePreview').show();
        } else {
            $('#pricePreview').hide();
        }
    }

    /**
     * Display validation errors
     */
    function displayValidationErrors(errors) {
        Object.keys(errors).forEach(function(field) {
            const $field = $(`#${field}`);
            $field.addClass('is-invalid');
            $field.siblings('.invalid-feedback').text(errors[field][0]);
        });
    }

    /**
     * Reset pricing form
     */
    function resetPricingForm() {
        isEditMode = false;
        currentPricingId = null;
        $('#pricingForm')[0].reset();
        $('#pricing_id').val('');
        $('#category_id').prop('disabled', false);
        $('.form-control, .form-select').removeClass('is-invalid');
        $('.invalid-feedback').text('');
        $('#pricePreview').hide();
    }

})();

