/**
 * Geo Country Detection & Auto-Selection
 * Automatically detects user's country based on IP geolocation
 * and auto-selects country in registration forms
 */

class GeoCountryDetect {
    constructor() {
        this.userCountry = null;
        this.userCountryCode = null;
        this.countryPhoneCodes = {
            // Major countries with phone codes
            'South Africa': '+27',
            'United States': '+1',
            'United Kingdom': '+44',
            'Canada': '+1',
            'Australia': '+61',
            'India': '+91',
            'Germany': '+49',
            'France': '+33',
            'Italy': '+39',
            'Spain': '+34',
            'Netherlands': '+31',
            'Belgium': '+32',
            'Switzerland': '+41',
            'Austria': '+43',
            'Sweden': '+46',
            'Norway': '+47',
            'Denmark': '+45',
            'Finland': '+358',
            'Poland': '+48',
            'Portugal': '+351',
            'Greece': '+30',
            'Ireland': '+353',
            'New Zealand': '+64',
            'Singapore': '+65',
            'Malaysia': '+60',
            'Thailand': '+66',
            'Philippines': '+63',
            'Indonesia': '+62',
            'Vietnam': '+84',
            'Japan': '+81',
            'South Korea': '+82',
            'China': '+86',
            'Brazil': '+55',
            'Mexico': '+52',
            'Argentina': '+54',
            'Chile': '+56',
            'Colombia': '+57',
            'Peru': '+51',
            'Egypt': '+20',
            'Nigeria': '+234',
            'Kenya': '+254',
            'Ghana': '+233',
            'Morocco': '+212',
            'Algeria': '+213',
            'Tunisia': '+216',
            'United Arab Emirates': '+971',
            'Saudi Arabia': '+966',
            'Israel': '+972',
            'Turkey': '+90',
            'Russia': '+7',
            'Ukraine': '+380',
            'Czech Republic': '+420',
            'Hungary': '+36',
            'Romania': '+40',
            'Bulgaria': '+359',
            'Pakistan': '+92',
            'Bangladesh': '+880',
            'Sri Lanka': '+94',
            'Nepal': '+977',
            'Afghanistan': '+93',
            'Iraq': '+964',
            'Iran': '+98',
            'Jordan': '+962',
            'Lebanon': '+961',
            'Kuwait': '+965',
            'Qatar': '+974',
            'Bahrain': '+973',
            'Oman': '+968',
            'Yemen': '+967'
        };
        
        this.init();
    }

    async init() {
        try {
            await this.detectCountry();
            // Don't auto-select countries, only detect for manual use
        } catch (error) {
            console.log('Country detection skipped:', error.message);
        }
    }

    async detectCountry() {
        // Skip external API calls to avoid CORS errors
        // Use browser-based detection only
        this.detectFromBrowser();
    }

    detectFromBrowser() {
        // Try to infer country from browser timezone
        const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
        const timezoneCountryMap = {
            'Africa/Johannesburg': 'South Africa',
            'America/New_York': 'United States',
            'America/Los_Angeles': 'United States',
            'America/Chicago': 'United States',
            'Europe/London': 'United Kingdom',
            'America/Toronto': 'Canada',
            'Australia/Sydney': 'Australia',
            'Asia/Kolkata': 'India',
            'Europe/Berlin': 'Germany',
            'Europe/Paris': 'France',
            'Europe/Rome': 'Italy',
            'Europe/Madrid': 'Spain',
            'Asia/Singapore': 'Singapore',
            'Asia/Dubai': 'United Arab Emirates',
            'Asia/Tokyo': 'Japan',
            'Asia/Shanghai': 'China',
            'America/Sao_Paulo': 'Brazil',
            'America/Mexico_City': 'Mexico'
        };

        if (timezoneCountryMap[timezone]) {
            this.userCountry = timezoneCountryMap[timezone];
            console.log('Country inferred from timezone:', this.userCountry);
        }
    }

    autoSelectCountry() {
        if (!this.userCountry) {
            console.log('No country detected, skipping auto-selection');
            return;
        }

        console.log('Attempting to auto-select country:', this.userCountry);

        // Auto-select country in all country dropdowns
        const countrySelectors = [
            '#country',
            '#user_country',
            '#provider_country',
            '.country'
        ];

        countrySelectors.forEach(selector => {
            const $countrySelect = $(selector);
            if ($countrySelect.length > 0) {
                console.log(`Found country dropdown: ${selector}, options: ${$countrySelect.find('option').length}`);
                this.selectCountryOption($countrySelect, this.userCountry);
            }
        });

        // Add country code prefix to phone number fields (immediate and delayed)
        this.addPhonePrefix();
        
        // Also update phone fields when country changes
        this.attachCountryChangeListeners();
    }
    
    attachCountryChangeListeners() {
        const self = this;
        const countrySelectors = [
            '#country',
            '#user_country',
            '#provider_country',
            '.country'
        ];

        countrySelectors.forEach(selector => {
            $(document).on('change', selector, function() {
                const selectedCountry = $(this).find('option:selected').text().trim();
                if (selectedCountry && selectedCountry !== 'Select') {
                    self.updatePhonePrefix(selectedCountry);
                }
            });
        });
    }

    selectCountryOption($select, countryName) {
        // Try to find and select the country option
        let found = false;
        $select.find('option').each(function() {
            const optionText = $(this).text().trim();
            const optionValue = $(this).val();
            
            // Match by text or value
            if (optionText === countryName || optionValue === countryName) {
                $(this).prop('selected', true);
                $select.val(optionValue).trigger('change');
                console.log('Auto-selected country:', countryName, 'in', $select.attr('id'));
                found = true;
                return false; // Break loop
            }
        });
        
        // If not found and dropdown is empty or has only default option, wait and retry (with limit)
        const optionCount = $select.find('option').length;
        if (!found && (optionCount <= 1 || optionCount === 2)) {
            // Only retry a few times to prevent infinite loops
            if (!$select.data('retry-count')) {
                $select.data('retry-count', 0);
            }
            const retryCount = $select.data('retry-count');
            if (retryCount < 3) {
                console.log('Country dropdown not populated yet, retrying in 500ms...');
                $select.data('retry-count', retryCount + 1);
                setTimeout(() => {
                    this.selectCountryOption($select, countryName);
                }, 500);
            } else {
                console.log('Max retries reached for country selection');
            }
        } else if (!found) {
            console.log('Country not found in dropdown:', countryName);
        }
    }

    addPhonePrefix() {
        const phoneCode = this.countryPhoneCodes[this.userCountry];
        if (!phoneCode) return;

        this.updatePhonePrefix(this.userCountry);
    }

    updatePhonePrefix(countryName) {
        const phoneCode = this.countryPhoneCodes[countryName];
        if (!phoneCode) {
            console.log('No phone code found for:', countryName);
            return;
        }

        // Look for existing phone input systems (like intl-tel-input)
        this.updateIntlTelInput(countryName, phoneCode);
        
        // Also update regular phone fields as fallback
        this.updateRegularPhoneFields(phoneCode);
    }

    updateIntlTelInput(countryName, phoneCode) {
        // Try to find intl-tel-input instances
        const phoneInputs = $('input[type="tel"], input[name*="phone"], input[id*="phone"]');
        
        phoneInputs.each(function() {
            const $input = $(this);
            const inputElement = this;
            
            // Check if this is an intl-tel-input by looking for the instance
            let iti = null;
            
            // Try different ways to get the intl-tel-input instance
            if (window.intlTelInput && inputElement._iti) {
                iti = inputElement._iti;
            } else if ($input.data('intl-tel-input')) {
                iti = $input.data('intl-tel-input');
            } else if ($input.closest('.iti').length > 0) {
                // Look for the instance in global scope or attached to element
                const itiElement = $input.closest('.iti').find('input')[0];
                if (itiElement && itiElement._iti) {
                    iti = itiElement._iti;
                }
            }
            
            if (iti && iti.setCountry) {
                try {
                    const countryCode = this.getCountryCodeForIntlTelInput(countryName);
                    if (countryCode) {
                        iti.setCountry(countryCode);
                        console.log('Updated intl-tel-input to', countryCode, 'for', $input.attr('id'));
                        return; // Success, exit early
                    }
                } catch (e) {
                    console.log('Could not update intl-tel-input:', e);
                }
            }
            
            // Fallback: Check for custom phone input with country selector
            const $countryBtn = $input.siblings('button').first();
            if ($countryBtn.length > 0) {
                // Update the country code in the button text
                const currentText = $countryBtn.text();
                const newText = currentText.replace(/\+?\d+/g, phoneCode);
                $countryBtn.text(newText);
                console.log('Updated custom phone input to', phoneCode);
            }
        });
    }

    getCountryCodeForIntlTelInput(countryName) {
        // Map country names to ISO country codes for intl-tel-input
        const countryMap = {
            'South Africa': 'za',
            'United States': 'us',
            'United Kingdom': 'gb',
            'Canada': 'ca',
            'Australia': 'au',
            'India': 'in',
            'Germany': 'de',
            'France': 'fr',
            'Italy': 'it',
            'Spain': 'es',
            'Netherlands': 'nl',
            'Belgium': 'be',
            'Switzerland': 'ch',
            'Austria': 'at',
            'Sweden': 'se',
            'Norway': 'no',
            'Denmark': 'dk',
            'Finland': 'fi',
            'Poland': 'pl',
            'Portugal': 'pt',
            'Greece': 'gr',
            'Ireland': 'ie',
            'New Zealand': 'nz',
            'Singapore': 'sg',
            'Malaysia': 'my',
            'Thailand': 'th',
            'Philippines': 'ph',
            'Indonesia': 'id',
            'Vietnam': 'vn',
            'Japan': 'jp',
            'South Korea': 'kr',
            'China': 'cn',
            'Brazil': 'br',
            'Mexico': 'mx',
            'Argentina': 'ar',
            'Chile': 'cl',
            'Colombia': 'co',
            'Peru': 'pe',
            'Egypt': 'eg',
            'Nigeria': 'ng',
            'Kenya': 'ke',
            'Ghana': 'gh',
            'Morocco': 'ma',
            'Algeria': 'dz',
            'Tunisia': 'tn',
            'United Arab Emirates': 'ae',
            'Saudi Arabia': 'sa',
            'Israel': 'il',
            'Turkey': 'tr',
            'Russia': 'ru',
            'Ukraine': 'ua',
            'Czech Republic': 'cz',
            'Hungary': 'hu',
            'Romania': 'ro',
            'Bulgaria': 'bg'
        };
        
        return countryMap[countryName] || '';
    }

    updateRegularPhoneFields(phoneCode) {
        // Update regular phone fields as fallback
        const phoneSelectors = [
            '#phone_number',
            '#user_phone_number',
            '#provider_phone_number'
        ];

        phoneSelectors.forEach(selector => {
            const $phoneInput = $(selector);
            if ($phoneInput.length > 0 && !$phoneInput.closest('.iti').length) {
                // Only update if not already wrapped in intl-tel-input
                let basePlaceholder = $phoneInput.data('original-placeholder');
                if (!basePlaceholder) {
                    basePlaceholder = ($phoneInput.attr('placeholder') || 'Enter Phone Number').replace(/^\+\d+\s*/, '');
                    $phoneInput.data('original-placeholder', basePlaceholder);
                }
                $phoneInput.attr('placeholder', `${phoneCode} ${basePlaceholder}`);
                $phoneInput.data('country-code', phoneCode);
                
                console.log('Updated regular phone field to', phoneCode, 'for', selector);
            }
        });
    }

    // Public method to get detected country
    getDetectedCountry() {
        return {
            name: this.userCountry,
            code: this.userCountryCode,
            phoneCode: this.countryPhoneCodes[this.userCountry]
        };
    }
}

// Initialize on document ready
$(document).ready(function() {
    // Add CSS for phone prefix badge
    if (!$('#phone-prefix-styles').length) {
        $('head').append(`
            <style id="phone-prefix-styles">
                .phone-input-wrapper {
                    display: flex !important;
                    align-items: center !important;
                    border: 1px solid #ced4da;
                    border-radius: 0.375rem;
                    padding: 0.375rem 0.75rem;
                    background: white;
                }
                .phone-input-wrapper:focus-within {
                    border-color: #86b7fe;
                    box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
                }
                .phone-input-wrapper .phone-prefix {
                    font-size: 0.875rem;
                    font-weight: 600;
                    padding: 0.25rem 0.5rem;
                    white-space: nowrap;
                    flex-shrink: 0;
                }
                .phone-input-wrapper .form-control {
                    border: none !important;
                    padding: 0 !important;
                    box-shadow: none !important;
                    flex: 1;
                }
                .phone-input-wrapper .form-control:focus {
                    outline: none !important;
                    box-shadow: none !important;
                }
            </style>
        `);
    }
    
    // Create global instance but don't auto-initialize
    window.geoCountryDetect = new GeoCountryDetect();
    
    // Only initialize when needed, not on page load

           // Trigger only when "Become Provider" button is clicked
           $(document).on('click', '[data-bs-target="#provider"]', function() {
               console.log('🚀 Become Provider button clicked - will set country to South Africa');
           });

           // Set country when provider modal opens
           $('#provider').on('shown.bs.modal', function() {
               setTimeout(function() {
                   // First, try to select South Africa in the country dropdown
                   const countrySelect = document.querySelector("#provider_country");
                   if (countrySelect) {
                       console.log('🔍 Looking for country dropdown:', countrySelect);
                       console.log('🔍 Country dropdown options:', countrySelect.options.length);
                       
                       // Wait for options to be populated
                       let attempts = 0;
                       const trySelectCountry = () => {
                           attempts++;
                           const options = countrySelect.options;
                           console.log(`🔍 Country attempt ${attempts}: ${options.length} options found`);
                           
                           if (options.length > 1) {
                               // Look for South Africa option
                               for (let i = 0; i < options.length; i++) {
                                   if (options[i].text === 'South Africa') {
                                       countrySelect.value = options[i].value;
                                       countrySelect.dispatchEvent(new Event('change'));
                                       console.log('✅ Selected South Africa in country dropdown');
                                       
                                       // Country selected - phone field will be handled by user manually
                                       console.log('✅ Country selected - user can manually set phone field');
                                       break;
                                   }
                               }
                           } else if (attempts < 10) {
                               setTimeout(trySelectCountry, 500);
                           }
                       };
                       trySelectCountry();
                   }

                   // Country selection completed
               }, 2000); // Increased delay to give more time for initialization
           });

           // Trigger only when "Join Us" button is clicked
           $(document).on('click', '[data-bs-target="#register-modal"]', function() {
               console.log('🚀 Join Us button clicked - will set country to South Africa');
           });

           // Auto-select country in search forms on page load
           setTimeout(function() {
               // Check if search forms exist and auto-select South Africa
               const searchFormSelectors = ['#search_country', '#search_country_location'];
               
               searchFormSelectors.forEach(function(selector) {
                   if ($(selector).length > 0) {
                       console.log('🔍 Search form detected - auto-selecting South Africa in', selector);
                       const searchCountrySelect = document.querySelector(selector);
                       if (searchCountrySelect) {
                           let attempts = 0;
                           const trySelectSearchCountry = () => {
                               attempts++;
                               const options = searchCountrySelect.options;
                               console.log(`🔍 Search country attempt ${attempts} for ${selector}: ${options.length} options found`);
                               
                               if (options.length > 1) { // Changed back to > 1 since we only have "Select" placeholder
                                   // Look for South Africa option
                                   for (let i = 0; i < options.length; i++) {
                                       if (options[i].text === 'South Africa') {
                                           searchCountrySelect.value = options[i].value;
                                           searchCountrySelect.dispatchEvent(new Event('change'));
                                           console.log('✅ Auto-selected South Africa in search form', selector);
                                           
                                           // Add visual feedback
                                           $(selector).closest('.input-group').addClass('auto-detected');
                                           setTimeout(() => {
                                               $(selector).closest('.input-group').removeClass('auto-detected');
                                           }, 3000);
                                           break;
                                       }
                                   }
                               } else if (attempts < 20) { // Increased attempts to wait for JSON loading
                                   setTimeout(trySelectSearchCountry, 750);
                               }
                           };
                           trySelectSearchCountry();
                       }
                   }
               });
           }, 3000); // Increased initial delay to ensure JSON data is loaded

           // Set country when user registration modal opens
           $('#register-modal').on('shown.bs.modal', function() {
               setTimeout(function() {
                   // Try to select South Africa in the user country dropdown
                   const countrySelect = document.querySelector("#country");
                   if (countrySelect) {
                       console.log('🔍 Looking for user country dropdown:', countrySelect);
                       console.log('🔍 User country dropdown options:', countrySelect.options.length);
                       
                       // Wait for options to be populated
                       let attempts = 0;
                       const trySelectCountry = () => {
                           attempts++;
                           const options = countrySelect.options;
                           console.log(`🔍 User country attempt ${attempts}: ${options.length} options found`);
                           
                           if (options.length > 1) {
                               // Look for South Africa option
                               for (let i = 0; i < options.length; i++) {
                                   if (options[i].text === 'South Africa') {
                                       countrySelect.value = options[i].value;
                                       countrySelect.dispatchEvent(new Event('change'));
                                       console.log('✅ Selected South Africa in user country dropdown');
                                       break;
                                   }
                               }
                           } else if (attempts < 10) {
                               setTimeout(trySelectCountry, 500);
                           }
                       };
                       trySelectCountry();
                   } else {
                       console.log('❌ User country dropdown not found');
                   }
               }, 1000);
           });
});

// Export for manual use
window.GeoCountryDetect = GeoCountryDetect;

